//*****************************************************
// october 2015: Updated for use in 10th reduction
// calibration parameters changed to 10th reduction dEdx
//
//24-feb-2009: introduced "beta_in" and "Set_Beta":
// beta_in is set to 100 by default. If it is not overwritten
// by "Set_Beta" (ToFNuclei.h), the code is used as before.
// If beta_in was set by the user using "Set_Beta", this beta
// is taken for the calculations. 
//
//26-feb-2009: Inserted S115B calibrations from Lucia
//
//02-apr-2009: bug found in some "if" statements for the 
// "Layercharge" definition, but only important if ADC=4095 
//
//15-apr-2009: small bug found in the calculation of the PMT
// charge, the charge for the PMTs 1,9,28.. was calculated
// twice, but the final result was OK
//
//05-may-2009: bug found in the calculation of the PMT
// charge for standalone ToF due to changes made for rev 1.4: 
// For PMTs 4,8,39,41 (counting 0..47) no charge was calculated 
// calibration arrays now including correct brackets to avoid
// compiler warnings  
//06-aug-2009: charge_layer_trk_raw  and charge_layer_std_raw
// added: These are the "raw" values before doing the 2nd-order 
// correction
//*****************************************************
#include <ToFNuclei.h>

//--------------------------------------
/**
 * Default constructor 
 */
//ToFNuclei::ToFNuclei(){
//  Clear();
//};

ToFNuclei::ToFNuclei(PamLevel2 *l2p,const char* alg){  
  //
  Clear();
  //
  L2 = l2p;
  //
  beta_in = 100.;
  //
  if ( !L2->IsORB() ) printf(" WARNING: OrbitalInfo Tree is needed, the plugin could not work properly without it \n");
  //
  OBT = 0;
  PKT = 0;
  atime = 0;
  //
  sntr = 0;
  debug = false;
  trkAlg = alg;
  //
};

void ToFNuclei::Clear(){
  //
  ncount_trk = 0;
  charge_trk = 0.;
  memset (charge_layer_trk,0,6*sizeof(Float_t));  //
  memset (charge_layer_trk_raw,0,6*sizeof(Float_t));  //
  memset (charge_PMT_trk,0,48*sizeof(Float_t));  //
  memset (layerinfo_trk,0,6*sizeof(Int_t));  //
  ncount_std = 0;
  charge_std = 0.;
  memset (charge_layer_std,0,6*sizeof(Float_t));  //
  memset (charge_layer_std_raw,0,6*sizeof(Float_t));  //
  memset (charge_PMT_std,0,48*sizeof(Float_t));  //
  memset (layerinfo_std,0,6*sizeof(Int_t));  //
};

void ToFNuclei::Print(){
  //
  Process();
  //
  printf("========================================================================\n");
  printf(" OBT: %u PKT: %u ATIME: %u Track %i \n",OBT,PKT,atime,tr);
  printf(" std : charge %f ncount %d layers %f %f %f %f %f %f \n",charge_std,ncount_std,charge_layer_std[0],charge_layer_std[1],charge_layer_std[2],charge_layer_std[3],charge_layer_std[4],charge_layer_std[5]);
  printf(" std : layerinfo %d %d %d %d %d %d \n",layerinfo_std[0],layerinfo_std[1],layerinfo_std[2],layerinfo_std[3],layerinfo_std[4],layerinfo_std[5]);

  printf(" trk : charge %f ncount %d layers %f %f %f %f %f %f \n",charge_trk,ncount_trk,charge_layer_trk[0],charge_layer_trk[1],charge_layer_trk[2],charge_layer_trk[3],charge_layer_trk[4],charge_layer_trk[5]);
  printf(" trk : layerinfo %d %d %d %d %d %d \n",layerinfo_trk[0],layerinfo_trk[1],layerinfo_trk[2],layerinfo_trk[3],layerinfo_trk[4],layerinfo_trk[5]);
  printf("========================================================================\n");
  //
};

void ToFNuclei::Delete(){
  Clear();
};


void ToFNuclei::Process(){
  Process(0);
};

void ToFNuclei::Process(Int_t ntr){
  //  
  if ( !L2 ){
    printf(" ERROR: cannot find PamLevel2 object, use the correct constructor or check your program!\n");
    printf(" ERROR: CaloExample variables not filled \n");
    return;
  };
  //
  Bool_t newentry = false;
  
  //
  if ( L2->IsORB() ){
    if ( L2->GetOrbitalInfo()->pkt_num != PKT || L2->GetOrbitalInfo()->OBT != OBT || L2->GetOrbitalInfo()->absTime != atime || ntr != sntr ){
      newentry = true;
      OBT = L2->GetOrbitalInfo()->OBT;
      PKT = L2->GetOrbitalInfo()->pkt_num;
      atime = L2->GetOrbitalInfo()->absTime;
      sntr = ntr;
    };
  } else {
    newentry = true;
  };
  //
  
  if ( !newentry ) {
  return;
                  }
  //
  tr = ntr;
  //
  if ( debug ) printf(" Processing event at OBT %u PKT %u time %u \n",OBT,PKT,atime);
  //
//  charge_trk = 0.;
//  memset (charge_layer_trk,0,6*sizeof(Float_t));  //
//  charge_std = 0.;
//  memset (charge_layer_std,0,6*sizeof(Float_t));  //

  Clear();

  //  cout<<"beta_in = "<<beta_in<<endl;


  Double_t time;

  if ( L2->IsORB() ){
   time = L2->GetOrbitalInfo()->absTime; 
  } else {
   time = 1153000000.;  // simulated data no OrbitalInfo  fixed date 15-jul-2006
  };


//=======================================================================
//==============          Siegen version 1           ====================
//==============   track dependent dEdx and beta     ====================
//=======================================================================

//if (L2->GetTrkLevel2()->GetNTracks()!=1) return;
//if (L2->GetTrkLevel2()->GetNTracks()==1) {
 
if( L2->GetNTracks(trkAlg)>=1 ){
/*
Float_t  PMTsat[48] = {
3162.14, 3165.48, 3153.85, 3085.73, 3089.65, 3107.64, 3097.52, 3078.37, 3130.05, 3087.07, 3112.22, 
3102.92, 3080.58, 3092.55, 3087.94, 3125.03, 3094.09, 3143.16,
3125.51, 3181.27, 3092.09, 3124.98, 3069.3, 3095.53, 3097.11, 3133.53, 3114.73, 3113.01, 3091.19, 
3097.99, 3033.84, 3134.98, 3081.37, 3111.04, 3066.77, 3108.17,
3133, 3111.06, 3052.52, 3140.66, 3106.33, 3094.85, 3150.85, 3118.8, 3096.24, 3118.47,3111.36, 3117.11 
};
*/

// new values from Napoli dec 2008
Float_t  PMTsat[48] = {
3176.35,3178.19,3167.38,3099.73,3117.00,3126.29,3111.44,3092.27,
3146.48,3094.41,3132.13,3115.37,3099.32,3110.97,3111.80,3143.14,
3106.72,3153.44,3136.00,3188.96,3104.73,3140.45,3073.18,3106.62,
3112.48,3146.92,3127.24,3136.52,3109.59,3112.89,3045.15,3147.26,
3095.92,3121.05,3083.25,3123.62,3150.92,3125.30,3067.60,3160.18,
3119.36,3108.92,3164.77,3133.64,3111.47,3131.98,3128.87,3135.56 };

for (Int_t i=0; i<48;i++) PMTsat[i] = PMTsat[i] - 5.;  // safety margin


Float_t C0t[48][9] = {
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90}} ;



Float_t A0t[48][9] = {
{0, 0.966862, 6.90746, 13.0828, 27.5276, 37.8862, 46.0339, 59.9999, 8100},
{0, 0.89781, 6.27511, 12.0001, 16.3328, 26.9085, 48.9715, 60.0001, 8100},
{0, 1.28799, 5.18918, 14.087, 23.9813, 26.9957, 37.3137, 57.6302, 8100},
{0, 1.15409, 5.62046, 12.9582, 29.0246, 36.1749, 34.2915, 62.8713, 8100},
{0, 1.15346, 6.52882, 13.8033, 23.4424, 32.5014, 35.5784, 49.9999, 8100},
{0, 1.06303, 6.30765, 12.6676, 23.5195, 27.4434, 29.3732, 50.6973, 8100},
{0, 1.19687, 6.39298, 14.352, 23.7507, 31.3431, 38.2434, 60, 8100},
{0, 1.08344, 5.69174, 13.1589, 29.0491, 33.8223, 34.5832, 42.4801, 8100},
{0, 1.27778, 6.04128, 12.1344, 20, 29.5231, 40.6637, 70, 8100},
{0, 1.1633, 6.24587, 15.3981, 22.6264, 25.4787, 40.9823, 54.3096, 8100},
{0, 1.25099, 5.51503, 13.0747, 25.7338, 28.0607, 29.3234, 43.1166, 8100},
{0, 1.14791, 5.28525, 11.3329, 22.0258, 29.1518, 31.5672, 48.5941, 8100},
{0, 0.989178, 6.46598, 16.008, 27.7832, 38.1067, 51.0926, 83.9029, 8100},
{0, 0.791611, 6.25114, 14.4726, 22.3348, 40.498, 38.8456, 76.8415, 8100},
{0, 1.08442, 6.22537, 10.786, 13.7616, 22.0316, 34.2567, 70.0001, 8100},
{0, 1.06481, 5.81616, 15.9999, 25.0001, 27.7402, 39.9999, 62.4627, 8100},
{0, 1.54106, 7.73501, 14.8365, 33.808, 38.6967, 55.3031, 55.1627, 8100},
{0, 0.181194, 8.88375, 19.761, 30.0001, 37.5278, 60.0001, 79.9997, 8100},
{0, 1.20333, 6.4235, 14.438, 20.6791, 37.59, 43.0118, 84.5498, 8100},
{0, 1.38501, 6.33531, 12.6738, 26.5495, 32.5919, 39.7116, 66.775, 8100},
{0, 1.2447, 6.41472, 13.1281, 24.5903, 27.8925, 34.4098, 52.9463, 8100},
{0, 1.10032, 6.84365, 14.8743, 29.1848, 36.0101, 42.1412, 79.1916, 8100},
{0, 1.37387, 6.1759, 12.3971, 26.6771, 34.5125, 38.8504, 67.9609, 8100},
{0, 1.26074, 6.44693, 13.4109, 23.0619, 31.6812, 37.3077, 57.2156, 8100},
{0, 1.27824, 6.99981, 15.2662, 31.25, 40.268, 40.9496, 83.4595, 8100},
{0, 1.27135, 6.1494, 14.6932, 25.7259, 35.6644, 43.8106, 65.4656, 8100},
{0, 1.40868, 6.80116, 14.308, 22.7333, 28.8201, 43.6075, 75.1262, 8100},
{0, 2.17689, 7.12873, 14.4537, 26.6741, 33.406, 40.8852, 59.659, 8100},
{0, 1.8449, 11.9433, 21.2362, 34.4344, 43.4968, 54.9996, 60, 8100},
{0, 1.71308, 7.86445, 15.1826, 20.2075, 22.4894, 30, 40, 8100},
{0, 2.47299, 6.72486, 15.6655, 25.0002, 29.7029, 31.7548, 45, 8100},
{0, 1.73646, 7.28222, 14.6779, 20.0001, 23.1851, 24.6151, 34.9945, 8100},
{0, 2.10843, 8.61062, 17.3962, 28.7763, 39.014, 52.9373, 80.1161, 8100},
{0, 2.12514, 9.41508, 19.6945, 38.8657, 49.4172, 70.8189, 117.259, 8100},
{0, 1.96994, 8.71348, 16.2668, 20.6351, 22.7219, 30.0001, 50, 8100},
{0, 1.97711, 9.8935, 20.0511, 28.0006, 34.1434, 45.5466, 49.9974, 8100},
{0, 3.02174, 11.9467, 22.1838, 44.8834, 67.8526, 87.5638, 99.9968, 8100},
{0, 3.23019, 12.396, 21.1859, 43.4576, 69.1176, 102.143, 129.986, 8100},
{0, 3.05255, 11.4421, 22.1753, 39.4975, 57.5947, 75.0661, 109.538, 8100},
{0, 3.03561, 11.281, 21.7485, 35.0295, 55.2597, 77.795, 138.547, 8100},
{0, 2.74373, 10.8424, 22.9863, 46.0158, 63.3989, 85.9676, 100.002, 8100},
{0, 2.91456, 10.4374, 18.7076, 37.4149, 44.8928, 53.9337, 69.9997, 8100},
{0, 3.35025, 7.22152, 27.5385, 38.8628, 57.2992, 83.4745, 139.076, 8100},
{0, 3.29637, 11.1582, 23.0533, 46.9451, 65.503, 94.1756, 135.554, 8100},
{0, 3.17507, 12.3589, 22.0947, 47.3915, 78.6842, 117.078, 121.698, 8100},
{0, 3.27437, 12.2762, 22.9416, 48.1462, 72.4744, 117.944, 129.998, 8100},
{0, 3.3915, 11.4755, 24.4284, 42.7813, 65.1621, 93.6156, 147.765, 8100},
{0, 3.11795, 10.5757, 22.8067, 41.8451, 68.3004, 91.8733, 145.854, 8100} } ;


Float_t A1t[48][9] = {
{0, -0.445487, -6.83011, -7.76746, -26.8019, -33.6372, -33.4238, -34.3206, 0},
{0, -0.154046, -5.11675, -6.68917, -1.29614, -7.64427, -42.6283, -37.5536, 0},
{0, -0.926441, -2.41216, -10.4314, -18.6624, -2.56534, -5.91767, -4.52255, 0},
{0, -0.669591, -3.17426, -8.41107, -30.2702, -28.1341, -0.952674, -28.7299, 0},
{0, -0.650638, -5.55242, -9.21873, -16.4049, -19.1945, -9.79077, -25.7112, 0},
{0, -0.357887, -5.01401, -6.99931, -16.2411, -5.48061, 7.12013, -7.63631, 0},
{0, -0.638868, -5.04599, -9.72543, -15.2232, -6.64514, 2.25032, 9.31463, 0},
{0, -0.357529, -3.33754, -8.25226, -29.5745, -20.183, 1.57311, 29.3872, 0},
{0, -0.812946, -4.17262, -5.66565, -4, -4.22773, -0.362511, -15.5266, 0},
{0, -0.348653, -4.64069, -14.5046, -9.84901, -0.193718, -12.1145, 16.7739, 0},
{0, -0.861006, -3.09587, -8.27656, -22.4036, -8.06958, 8.55529, 16.6237, 0},
{0, -0.622119, -2.31927, -3.8959, -14.6788, -11.5365, 4.07618, 13.565, 0},
{0, -0.5254, -5.57273, -15.4304, -24.9533, -23.3748, -30.7083, -60.3975, 0},
{0, 0.0969597, -4.99926, -12.3993, -14.8107, -37.9522, -10.6047, -50.5152, 0},
{0, -0.933961, -5.87726, -3.98088, 2.03266, 1.00194, -13.1254, -37.0729, 0},
{0, -0.455788, -5.27479, -18.0196, -27.5684, -14.3756, -30.1785, -37.5799, 0},
{0, -1.54773, -9.60851, -13.3097, -42.6674, -40.9163, -58.1322, -19.1456, 0},
{0, 4.8413, -11.2091, -20.5107, -26.3785, -10.1217, -32.6092, -38.9083, 0},
{0, -0.732962, -5.46743, -11.9727, -11.1357, -31.2803, -22.1011, -75.8073, 0},
{0, -1.23536, -5.59948, -9.07015, -28.2917, -22.0218, -18.8519, -34.4378, 0},
{0, -0.826823, -5.20145, -8.3843, -19.0111, -7.15699, -3.34332, -9.75121, 0},
{0, -0.577046, -6.28584, -11.4426, -28.0901, -21.3916, -12.5135, -61.76, 0},
{0, -0.956059, -5.00839, -7.22107, -25.7981, -25.7054, -15.0595, -40.2528, 0},
{0, -0.810131, -5.26933, -8.78012, -15.6279, -16.893, -9.72584, -17.8252, 0},
{0, -0.860119, -6.88744, -13.564, -35.2424, -35.6772, -14.8418, -77.2316, 0},
{0, -0.918947, -4.55685, -11.788, -23.1715, -26.2671, -25.1524, -32.8089, 0},
{0, -1.01825, -6.16033, -12.1071, -16.7393, -12.9732, -30.675, -67.521, 0},
{0, -2.63702, -7.23262, -12.3963, -29.1537, -26.1829, -24.3426, -35.0455, 0},
{0, -2.38232, -17.8376, -18.8801, -24.8636, -24.6509, -36.4698, -8.39844, 0},
{0, -1.96152, -8.79052, -11.6813, -4.72686, 4.81693, -4.2978, -8.39844, 0},
{0, -4.62994, -2.94981, -13.2937, -16.7482, -6.56193, 6.03485, -0.743217, 0},
{0, -2.06572, -7.28882, -11.7746, -5.47966, 0.813888, 10.4071, 20.0873, 0},
{0, -2.98167, -10.7673, -18.7865, -29.1646, -34.3681, -46.7229, -64.2386, 0},
{0, -3.09488, -12.6801, -23.2807, -50.5952, -50.3281, -77.6809, -127.885, 0},
{0, -2.6353, -11.327, -15.1071, -8.63713, -2.76153, -10.9598, -8.39844, 0},
{0, -2.69369, -14.0086, -23.0438, -20.243, -14.7684, -26.9053, 17.557, 0},
{0, -5.05491, -18.6246, -29.6528, -66.0253, -96.0606, -112.211, -87.435, 0},
{0, -5.6635, -19.449, -24.0238, -54.6138, -87.6223, -130.704, -149.534, 0},
{0, -5.14866, -17.6404, -30.0265, -53.7582, -74.4213, -88.8939, -107.031, 0},
{0, -5.1263, -17.4454, -28.9617, -43.0586, -71.6189, -103.028, -188.977, 0},
{0, -4.33081, -16.4979, -32.5261, -73.0454, -91.2755, -118.952, -83.7061, 0},
{0, -4.92134, -15.2835, -21.598, -48.8622, -43.3094, -40.5892, -69.4005, 0},
{0, -5.92968, -6.72971, -43.7034, -51.7821, -74.8294, -114.17, -209.667, 0},
{0, -5.65963, -17.484, -33.7998, -73.0621, -95.5599, -138.248, -182.989, 0},
{0, -5.45507, -20.3714, -29.0252, -69.4724, -120.52, -180.644, -117.081, 0},
{0, -5.67273, -20.3007, -32.4896, -74.9897, -109.782, -190.237, -146.269, 0},
{0, -6.15455, -18.331, -37.7733, -65.3947, -101.823, -146.711, -221.847, 0},
{0, -5.2493, -16.0076, -33.4883, -62.4233, -106.247, -139.708, -214.149, 0}};


Float_t A2t[48][9] = {
{0, 0.603703, 3.86157, 3.70388, 16.0228, 22.6954, 26.319, 44.3508, 0},
{0, 0.312779, 2.71821, 3.73851, 1.04468, 6.95408, 32.127, 46.5395, 0},
{0, 0.743361, 1.07616, 5.29201, 11.4779, 2.26256, 8.01082, 19.1997, 0},
{0, 0.534443, 1.40779, 4.55333, 17.9865, 18.9162, 5.49797, 36.8004, 0},
{0, 0.626433, 2.91524, 4.57075, 9.93735, 13.6735, 12.7599, 43.2477, 0},
{0, 0.403304, 2.5958, 3.37659, 9.36547, 4.78695, 3, 25.6807, 0},
{0, 0.554686, 2.54239, 4.54521, 8.81473, 3.42239, 1.55152, 10.102, 0},
{0, 0.40188, 1.53765, 4.26228, 17.2971, 13.2302, 3.44777, 1.44658, 0},
{0, 0.686001, 2.0432, 2.49348, 0, 3, 1.5756, 25.2157, 0},
{0, 0.123351, 2.27207, 8.01038, 3, 3, 12.2309, 3, 0},
{0, 0.73465, 1.47978, 4.24085, 13.1021, 6.44605, 0.681327, 8.34446, 0},
{0, 0.578203, 0.912111, 1.41144, 8.98856, 8.75212, 3, 7.95492, 0},
{0, 0.654411, 2.97054, 8.56266, 14.2585, 13.2056, 22.2557, 56.2898, 0},
{0, 0.194551, 2.5935, 6.94048, 9.31167, 24.9638, 11.8554, 46.4502, 0},
{0, 0.927312, 3.54272, 2, 0.557114, 3, 17.0684, 35.3889, 0},
{0, 0.528366, 3.31646, 10.7953, 18.5896, 11.9005, 27.3322, 43.338, 0},
{0, 0.912506, 5.8462, 7.66762, 25.6121, 29.5684, 42.9147, 38.2599, 0},
{0, -7.12979, 6.21087, 11.6741, 17.4818, 6.79753, 26.0391, 58.7369, 0},
{0, 0.665521, 2.95822, 6.47192, 7.05822, 20.8593, 18.9282, 65.3209, 0},
{0, 1.01894, 3.18477, 5.19886, 18.0916, 15.3677, 17.1022, 37.5697, 0},
{0, 0.721677, 2.71689, 4.22996, 11.1311, 6.35347, 8.49712, 27.3577, 0},
{0, 0.603829, 3.38297, 5.7368, 16.4073, 14.4175, 13.3666, 61.5047, 0},
{0, 0.707386, 2.73555, 3.78417, 15.7034, 17.5119, 14.2389, 42.0363, 0},
{0, 0.607098, 2.73084, 4.67754, 9.60779, 12.6869, 12.1505, 31.9974, 0},
{0, 0.661681, 3.79845, 7.40469, 21.1087, 23.2647, 14.5452, 68.5165, 0},
{0, 0.759887, 2.30697, 6.18542, 14.5154, 18.0736, 21.6141, 40.9744, 0},
{0, 0.3993, 3.20066, 6.95586, 10.3351, 10.6496, 25.789, 62.6339, 0},
{0, 1.11922, 4.03052, 7.15435, 19.0396, 19.5893, 21.629, 44.3572, 0},
{0, 1.67562, 9.80356, 7.57421, 11.7968, 17.3663, 33.0567, 28.6826, 0},
{0, 1.40668, 4.83038, 5.51678, 1.37774, 0.515603, 13.2881, 28.6826, 0},
{0, 3.26101, 0.0412805, 6.93063, 8.88441, 4.63011, 3, 25.3611, 0},
{0, 1.4764, 3.91391, 6.07754, 1.73516, 3, 3, 0, 0},
{0, 1.99777, 6.09319, 10.6805, 17.6838, 23.1259, 33.9526, 56.8642, 0},
{0, 2.09954, 7.17996, 13.001, 29.8411, 30.6333, 51.8831, 95.4856, 0},
{0, 1.80733, 6.5434, 7.91278, 5.34262, 8.17397, 18.3942, 28.6826, 0},
{0, 1.85376, 8.04532, 12.3798, 10.2491, 11.3718, 26.4686, 0, 0},
{0, 3.17369, 10.6479, 17.0112, 38.8841, 56.9463, 67.0329, 68.4495, 0},
{0, 3.57082, 11.0378, 12.2117, 29.7697, 49.3163, 74.7644, 111.901, 0},
{0, 3.23427, 10.1924, 17.1503, 31.2101, 44.171, 53.5184, 70.3691, 0},
{0, 3.23098, 10.1544, 16.4254, 24.8279, 43.4414, 64.6698, 122.023, 0},
{0, 2.73301, 9.65117, 18.883, 44.9873, 56.0859, 74.1037, 59.0439, 0},
{0, 3.13644, 8.83143, 12.0223, 28.0669, 24.5783, 24.7598, 68.4073, 0},
{0, 3.73914, 3.39684, 25.5464, 30.2677, 45.1122, 71.13, 143.934, 0},
{0, 3.47069, 10.3132, 20.0929, 43.0092, 57.7104, 85.0343, 123.773, 0},
{0, 3.45366, 12.0263, 15.9729, 39.5061, 71.0447, 107.589, 81.6047, 0},
{0, 3.54807, 12.0303, 19.0313, 44.4776, 66.7032, 116.232, 101.634, 0},
{0, 3.93093, 10.8482, 22.6813, 39.5255, 64.357, 93.1016, 147.004, 0},
{0, 3.30655, 9.37748, 20.0108, 37.5016, 65.6635, 88.695, 142.669, 0}};

// S115B failure: 1158700000 // 2006-09-19 21:06:40 UTC

if (time > 1158700000.) {     // if time > 2006-09-19 use special constants for PMT S115B
Float_t A0tx1[9] = {0, 1.1633, 6.24587, 9.51267, 15.1138, 14.404, 15.5928, 20.3819, 8100};
Float_t A1tx1[9] = {0, -0.348653, -4.64069, -3.15991, -2.33796, 6.02431, 14.7019, 26.2542, 0};
Float_t A2tx1[9] = {0, 0.123351, 2.27207, 3.22407, 3, 3, 0.0440895, 0.542518, 0};

for ( Int_t jj=0; jj<9; jj++ ){
A0t[9][jj] = A0tx1[jj] ; 
A1t[9][jj] = A1tx1[jj] ; 
A2t[9][jj] = A2tx1[jj] ; 
                              }
                         } // if time...

// 1167605999  31 dec 2006  S11A has problems with the dEdx

if (time > 1167605999 ) {     
Float_t A0tx2[9] = {0, 1.27778, 6.04128, 18.2621, 29.9833, 40, 48.6853, 70, 8100};  
Float_t A1tx2[9] = {0, -0.812946, -4.17262, -15.0648, -23.4309, -17.3787, -6.03927, 7.16296, 0};
Float_t A2tx2[9] = {0, 0.686001, 2.0432, 7.23452, 13.607, 10.2901, 5.99052, 15.045, 0};
for ( Int_t jj=0; jj<9; jj++ ){
A0t[8][jj] = A0tx2[jj] ; 
A1t[8][jj] = A1tx2[jj] ; 
A2t[8][jj] = A2tx2[jj] ; 
                              }
                         } // if time...


//============================================================================

//  2nd-order correction
Float_t corr_a[6][4] = {  
{0.977261,  0.353729,  -0.83314,  0.452883}, 
{1.27649,  3.53459, -5.41948,  2.67854}, 
{2.21926,  4.32797,  -7.24001,  3.77247},  
{2.4202,  7.99552,-12.4371,  6.08982}, 
{2.04984,  13.3375, -19.178,  8.86101},  
{2.63322,  14.8072,  -20.911,9.57905} };

//---------------------------------------------------------------------

Float_t xv[6],yv[6];
Int_t  iz;
Float_t x1[9],y1[9];
Int_t n1 = 9;

Float_t  Layercharge[6] = {1005., 1005., 1005., 1005., 1005., 1005. };

Int_t  Layerinfo[6] = {0, 0, 0, 0, 0, 0 };

//Float_t adc[48],adc_c[48],PMTcharge_arr[48];
Float_t adc[48],PMTcharge_arr[48];

//------------------------------------------------------------------

//PamTrack *track = L2->GetTrack(0);
 PamTrack *track = L2->GetTrack(0,trkAlg);
Float_t beta_mean = fabs(track->GetToFTrack()->beta[12]);

// New!! If "beta_in" != 100. then overwrite beta_mean with beta_in
if (beta_in != 100.) beta_mean=beta_in;

//Float_t def = track->GetTrkTrack()->al[4];
//Float_t rig = 1./def;

Int_t ncount=0;
Float_t charge_chi=1000.;

//----------------------------------------------------------------

for (Int_t jj=0; jj<6; jj++){
xv[jj]=track->GetToFTrack()->xtr_tof[jj];
yv[jj]=track->GetToFTrack()->ytr_tof[jj];
}

//----------------------------------------------------------------

Float_t theta,dx,dy,dr;

      theta=0.;
      dx=0.;
      dy=0.;
      dr=0.;

      if ((xv[0])<100.) {
         dx = (xv[0])-(xv[5]);
         dy = (yv[0])-(yv[5]);
         dr = sqrt(dx*dx+dy*dy);
         theta = atan(dr/77.5);
      }


//  ratio helium to proton ca. 4
    Float_t  hepratio = 4.;

//-------------------------------------------------------------
//------------------  ToF Charge ------------------------------
//-------------------------------------------------------------

// fill adc and tdc

   Int_t ich,jj,kk;

   for (ich=0; ich<48; ich++)   adc[ich]   = 4095.;
   //   for (ich=0; ich<48; ich++)   adc_c[ich] = 1000.;
   for (ich=0; ich<48; ich++)   PMTcharge_arr[ich] = 1000.;



if ((beta_mean < 0.05)  || (beta_mean > 2.0)) {
  for (Int_t jj=0; jj<48;jj++) charge_PMT_trk[jj] = 1001.;
  for (Int_t jj=0; jj<6;jj++) charge_layer_trk[jj] = 1001.; 
  for (Int_t jj=0; jj<6;jj++) charge_layer_trk_raw[jj] = 1001.; 
  for (Int_t jj=0; jj<6;jj++) layerinfo_trk[jj] = 1001; 
  charge_trk = 1001.;
  ncount_trk = 0;
                                              }


if ((beta_mean > 0.05)  && (beta_mean < 2.0)) {

   Float_t betahelp =   pow(beta_mean, 1.2);

//=======================================================================

   for (Int_t ipmt=0; ipmt<L2->GetToFLevel2()->npmt() ; ipmt++){
    ToFPMT *tofpmt = L2->GetToFLevel2()->GetToFPMT(ipmt);
    Int_t pmt_id = tofpmt->pmt_id;
    adc[pmt_id] = tofpmt->adc ;
                                                                       }

//=======================================================================
   Float_t yl, yh, ym, m, b, chelp, charge;

       for (Int_t ipmt=0; ipmt<track->GetToFTrack()->npmtadc; ipmt++){
          Float_t dEdx = track->GetToFTrack()->dedx[ipmt];
          Int_t pmtadc = track->GetToFTrack()->pmtadc[ipmt];    
    
//----------  S115B failure workaround  ---------------------------
         if ((pmtadc==9) && (time > 1158700000)) {
         Float_t x = adc[9];
         Float_t adcr = 28.0407 + 0.628929*x - 5.80901e-05*x*x + 3.14092e-08*x*x*x;
         adcr = adcr * cos(theta); 
         Float_t xpos=yv[0];
         Float_t xkorr = 3.91621*exp(-0.0129234*xpos) + 7.23454*exp(0.102039*xpos) ;
         xkorr=xkorr/hepratio;
         dEdx = adcr/xkorr;
                                              }
//------------------------------------------------------------------    
    
          ym = dEdx*betahelp;
          //          Int_t adcflag = track->GetToFTrack()->adcflag[ipmt];
          L2->GetToFLevel2()->GetPMTIndex(pmtadc,jj,kk);
          //          if (adcflag==0) adc_c[pmtadc] = dEdx;


          //         if (adc[pmtadc] == 4095) adc_c[pmtadc] = 0.;
         if (adc[pmtadc] == 4095) PMTcharge_arr[pmtadc] = 0.;

         //         if ((adc[pmtadc]>PMTsat[pmtadc]) && (adc[pmtadc]<4095)) adc_c[pmtadc] = 1000.;

         if (adc[pmtadc] < PMTsat[pmtadc]) {
//----------------------------------------------------------------------------------------
//if ((pmtadc!=1) && (pmtadc!=9) && (pmtadc!=28) && (pmtadc!=29) && (pmtadc!=34)) {

    for ( Int_t jj=0; jj<9; jj++ ){
    x1[jj] = A0t[pmtadc][jj]+A1t[pmtadc][jj]*beta_mean+A2t[pmtadc][jj]*beta_mean*beta_mean;
    y1[jj] = C0t[pmtadc][jj]*C0t[pmtadc][jj] ;
                                  }

    TGraph *gr2 = new TGraph(n1,x1,y1);
    TSpline3 *spl2 = new TSpline3("grs",gr2);    // use a cubic spline
    chelp = spl2->Eval(ym);
    if (chelp<0.) chelp=0.0001;  //WM new
    charge = TMath::Sqrt(chelp);
    gr2->Delete();
    spl2->Delete();
    PMTcharge_arr[pmtadc] = charge;

//                                                                         } // pmtadc != ...

//----------------------------------------------------------------------------------------
/*
if ((pmtadc==1) || (pmtadc==9) || (pmtadc==28) || (pmtadc==29) || (pmtadc==34)) {
         for ( Int_t jj=0; jj<8; jj++ ){
         yl = A0t[pmtadc][jj]+A1t[pmtadc][jj]*beta_mean+A2t[pmtadc][jj]*beta_mean*beta_mean;
         yh = A0t[pmtadc][jj+1]+A1t[pmtadc][jj+1]*beta_mean+A2t[pmtadc][jj+1]*beta_mean*beta_mean;
         if ((yl<ym)&&(ym<yh)){
         m = (C0t[pmtadc][jj+1]*C0t[pmtadc][jj+1] - C0t[pmtadc][jj]*C0t[pmtadc][jj]) / (yh - yl);
         b = (C0t[pmtadc][jj]*C0t[pmtadc][jj]) - m*yl;
         chelp = m*ym + b;
         if (chelp<0.) chelp=0.0001;  //WM new
         charge= TMath::Sqrt(chelp);
         PMTcharge_arr[pmtadc] = charge;
                             }
                                     } // jj=0..8
                                                                          } // if pmtadc ==....
*/
//----------------------------------------------------------------------------------------
                                       } // adc < sat

             }   // ipmt...

//====================================================================
//================  Find out the hitted ToF paddles    ===============
//====================================================================

Int_t PaddleIdOfTrack[6];

PaddleIdOfTrack[0] = L2->GetToFLevel2()->GetPaddleIdOfTrack(xv[0], yv[0], 0, 0.) ;
PaddleIdOfTrack[1] = L2->GetToFLevel2()->GetPaddleIdOfTrack(xv[1], yv[1], 1, 0.) ;
PaddleIdOfTrack[2] = L2->GetToFLevel2()->GetPaddleIdOfTrack(xv[2], yv[2], 2, 0.) ;
PaddleIdOfTrack[3] = L2->GetToFLevel2()->GetPaddleIdOfTrack(xv[3], yv[3], 3, 0.) ;
PaddleIdOfTrack[4] = L2->GetToFLevel2()->GetPaddleIdOfTrack(xv[4], yv[4], 4, 0.) ;
PaddleIdOfTrack[5] = L2->GetToFLevel2()->GetPaddleIdOfTrack(xv[5], yv[5], 5, 0.) ;

//==============================================================


Int_t ihelp_a[6] = {0,16,28,32,36,42};

for (Int_t ilay=0; ilay<6; ilay ++) {
Int_t jj = PaddleIdOfTrack[ilay] ;
//Float_t xhelp = 1000.;
Float_t xhelp = 1005.;

if (jj == -1) Layerinfo[ilay] = -1;

if (jj != -1) {
Int_t ih = ihelp_a[ilay] + 2*jj;

// New WM
Int_t igood1=0;
Int_t igood2=0;
if ((PMTcharge_arr[ih] < 1000.) && (PMTcharge_arr[ih] > 0.)) igood1=1;
if ((PMTcharge_arr[ih+1] < 1000.) && (PMTcharge_arr[ih+1] > 0.)) igood2=1;

if ((PMTcharge_arr[ih]==0.) && (PMTcharge_arr[ih+1]==0.))  xhelp = 0.;
if ((igood1==1)&&(igood2==1)) xhelp = 0.5*(PMTcharge_arr[ih]+PMTcharge_arr[ih+1]) ;
if ((igood1==1)&&(igood2==0)) xhelp = PMTcharge_arr[ih];
if ((igood2==1)&&(igood1==0)) xhelp = PMTcharge_arr[ih+1];


/*
if ((PMTcharge_arr[ih] < 1000.) && (PMTcharge_arr[ih+1] < 1000.)) xhelp = 0.5*(PMTcharge_arr[ih]+PMTcharge_arr[ih+1]) ;
if ((PMTcharge_arr[ih] < 1000.) && (PMTcharge_arr[ih+1]==1000.))  xhelp = PMTcharge_arr[ih] ;
if ((PMTcharge_arr[ih]==1000.) && (PMTcharge_arr[ih+1] < 1000.))  xhelp = PMTcharge_arr[ih+1] ;
*/
   if (adc[ih] ==  4095) Layerinfo[ilay] = Layerinfo[ilay] + 1;
   if (adc[ih+1] ==  4095) Layerinfo[ilay] = Layerinfo[ilay] + 1;
   if ((adc[ih] > PMTsat[ih]) && (adc[ih] < 4095)) Layerinfo[ilay] = Layerinfo[ilay] + 10;
   if ((adc[ih+1] > PMTsat[ih+1]) && (adc[ih+1] < 4095)) Layerinfo[ilay] = Layerinfo[ilay] + 10;
       }
Layercharge[ilay] = xhelp;
                                    } // ilay ...



//==========================================================================
//============= Layercharge without 2nd-order correction   =================
//==========================================================================

  for (Int_t jj=0; jj<6;jj++) charge_layer_trk_raw[jj] = Layercharge[jj]; 

//==========================================================================
//=========================   2nd-order correction   =======================
//==========================================================================

Float_t xb = beta_mean;

for (Int_t ii=0; ii<6;  ii++) {
Int_t ic = 0;
Float_t corr[4] = {1., 0., 0., 0.};
for (Int_t iii=1; iii<9;  iii++) if (fabs(Layercharge[ii] - iii)<0.50) ic = iii;
if ((ic>0)&&(ic<=5)) for (Int_t iii=0; iii<4;  iii++) corr[iii] = corr_a[ic-1][iii];
if (ic >5) for (Int_t iii=0; iii<4;  iii++) corr[iii] = corr_a[5][iii];

Float_t c_corr = corr[0]+corr[1]*xb+corr[2]*xb*xb+corr[3]*xb*xb*xb;

if ((ic>0)&&(ic<=5)) c_corr = c_corr/(1.*ic) ;
if (ic>5) c_corr = c_corr/6. ;

//cout<<"beta "<<xb<<" layer "<<ii<<" ic "<<ic<<" => "<<c_corr<<endl;
//cout<<xb<<"  "<<ic<<"  "<<c_corr<<endl;

Layercharge[ii] = Layercharge[ii]/c_corr;

                            }

//==========================================================================
//==========================================================================


Int_t  icount=0;
Float_t charge_mean = 0.;
for (Int_t ii=0; ii<6;  ii++) {
if (Layercharge[ii] < 1000) charge_mean = charge_mean + Layercharge[ii] ;
if (Layercharge[ii] < 1000) icount++;
                            }
if (icount>0) charge_mean = charge_mean/(1.*icount) ;

if (icount>0) {

//for (Int_t ii=0; ii<6;  ii++) cout<<Layercharge[ii]<<" ";
//cout<<endl;


  vector<float> charge_v;
  Float_t charge_median = 0.;

  ////////  get the median charge ////////////
  for(Int_t ii=0;ii<6;ii++) {
  if (Layercharge[ii] < 1000) charge_v.push_back(Layercharge[ii]);
                            }

  std::sort(charge_v.begin(),charge_v.end());
  charge_median = charge_v[(int)(charge_v.size()/2+charge_v.size()%2)-1];

//cout<<charge_mean<<" "<<charge_median<<endl;

  charge_mean = charge_median;

}  // icount>0


//--- Calculate charge for the second time, use residuals of the single
//--- measurements to get a chi2 value

  Float_t w_i_arr[8] = {0.1, 0.15, 0.2, 0.25, 0.30, 0.35, 0.45, 0.50};

  icount=0;
  Float_t sw=0.;
  Float_t sxw=0.;
  Float_t w_i = 1./0.5;
  Float_t resmax= 3.0;

  iz=1;
  if ((0.0<charge_mean) & (charge_mean<1.5)) iz=1;
  if ((1.5<charge_mean) & (charge_mean<2.5)) iz=2;
  if ((2.5<charge_mean) & (charge_mean<3.5)) iz=3;
  if ((3.5<charge_mean) & (charge_mean<4.5)) iz=4;
  if ((4.5<charge_mean) & (charge_mean<5.5)) iz=5;
  if ((5.5<charge_mean) & (charge_mean<6.5)) iz=6;
  if ((6.5<charge_mean) & (charge_mean<7.5)) iz=7;
  if ((7.5<charge_mean) & (charge_mean<90.)) iz=8;

  w_i = 1./w_i_arr[iz-1];

  for (Int_t jj=0; jj<6;jj++){
    if (Layercharge[jj] < 1000) {
      Float_t res = charge_mean - Layercharge[jj] ;
      if (fabs(res*w_i)<resmax) {
      icount= icount+1;
      sxw=sxw + Layercharge[jj]*w_i*w_i ;
      sw =sw + w_i*w_i ;
      }
    }
  }


  ncount =  icount;
  if (icount>0) charge_chi=sxw/sw;

//  cout<<" => "<<icount<<" "<<charge_chi<<endl;

//=============================================================
// changed 30-01-2009 WM

  for (Int_t jj=0; jj<48;jj++) charge_PMT_trk[jj] = PMTcharge_arr[jj]; 
  for (Int_t jj=0; jj<6;jj++) charge_layer_trk[jj] = Layercharge[jj]; 
  for (Int_t jj=0; jj<6;jj++) layerinfo_trk[jj] = Layerinfo[jj]; 
  charge_trk = charge_chi;
  ncount_trk = ncount;

} // 0.05 < beta < 2.0

} // if ntrk == 1

//================================================================
//==============          Siegen version 1           =============
//==============        standalone dEdx and beta     =============
//================================================================

  ToFTrkVar *track = L2->GetToFStoredTrack(-1);
//  if(!track)return;
  if(track) {

/*
Float_t  PMTsat[48] = {
3162.14, 3165.48, 3153.85, 3085.73, 3089.65, 3107.64, 3097.52, 3078.37, 3130.05, 3087.07, 3112.22, 
3102.92, 3080.58, 3092.55, 3087.94, 3125.03, 3094.09, 3143.16,
3125.51, 3181.27, 3092.09, 3124.98, 3069.3, 3095.53, 3097.11, 3133.53, 3114.73, 3113.01, 3091.19, 
3097.99, 3033.84, 3134.98, 3081.37, 3111.04, 3066.77, 3108.17,
3133, 3111.06, 3052.52, 3140.66, 3106.33, 3094.85, 3150.85, 3118.8, 3096.24, 3118.47,3111.36, 3117.11 
};
*/

// new values from Napoli dec 2008
Float_t  PMTsat[48] = {
3176.35,3178.19,3167.38,3099.73,3117.00,3126.29,3111.44,3092.27,
3146.48,3094.41,3132.13,3115.37,3099.32,3110.97,3111.80,3143.14,
3106.72,3153.44,3136.00,3188.96,3104.73,3140.45,3073.18,3106.62,
3112.48,3146.92,3127.24,3136.52,3109.59,3112.89,3045.15,3147.26,
3095.92,3121.05,3083.25,3123.62,3150.92,3125.30,3067.60,3160.18,
3119.36,3108.92,3164.77,3133.64,3111.47,3131.98,3128.87,3135.56 };

for (Int_t i=0; i<48;i++) PMTsat[i] = PMTsat[i] - 5.;  // safety margin

// calibration file "charge_fits_10th_std.dat"

Float_t C0t[48][9] = {
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90},
{0, 1, 2, 3, 4, 5, 6, 8, 90}} ;

Float_t A0t[48][9] = {
{0, 0.626244, 5.57075, 13.0829, 27.5278, 37.886, 46.0338, 59.9999, 8100},
{0, 0.669967, 6.13887, 12.0001, 16.3328, 26.9085, 48.9715, 60.0001, 8100},
{0, 1.07525, 4.95494, 14.0867, 23.9813, 26.9957, 37.3137, 57.6302, 8100},
{0, 0.991496, 6.14099, 12.9582, 29.0247, 36.1749, 34.2915, 62.8713, 8100},
{0, 0.894996, 5.84777, 13.8033, 23.4423, 32.5014, 35.5784, 49.9998, 8100},
{0, 1.1336, 5.59475, 12.6677, 23.5195, 27.4434, 29.3735, 50.6974, 8100},
{0, 0.691118, 5.26701, 14.351, 23.7506, 31.3431, 38.2434, 60.0001, 8100},
{0, 1.02012, 6.17243, 13.1588, 29.049, 33.8223, 34.5832, 42.4809, 8100},
{0, 0.0309897, 7.24482, 16.2425, 35.4099, 44.1408, 55.1932, 70, 8100},
{0, 0.65965, -0.44638, 9.51268, 15.1138, 14.4038, 15.5924, 20.3806, 8100},
{0, 0.681447, 4.88613, 13.0747, 25.7335, 28.0607, 29.3234, 43.1167, 8100},
{0, 1.21702, 6.06734, 11.333, 22.0258, 29.1518, 31.5672, 48.5941, 8100},
{0, 0.492463, 6.1436, 16.0077, 27.7831, 38.1067, 51.0924, 83.9029, 8100},
{0, 0.83562, 5.31218, 14.4727, 22.3348, 40.4978, 38.8456, 76.8416, 8100},
{0, 0.174564, 3.49201, 10.7861, 13.7616, 22.0316, 34.2567, 70.0001, 8100},
{0, 1.29945, 3.93137, 15.9999, 24.9999, 27.7403, 39.9996, 62.4627, 8100},
{0, 1.33337, 6.4213, 14.8368, 33.8082, 38.6969, 55.3044, 55.1627, 8100},
{0, -2.66314, 6.37651, 19.7608, 30.0002, 37.528, 60.0003, 79.9994, 8100},
{0, 0.974379, 5.37316, 14.438, 20.6792, 37.5899, 43.0121, 84.549, 8100},
{0, 0.980116, 6.6121, 12.6738, 26.5497, 32.5919, 39.7116, 66.775, 8100},
{0, 1.26686, 5.57744, 13.1284, 24.5904, 27.8926, 34.4098, 52.9463, 8100},
{0, 1.05908, 7.61144, 14.8745, 29.1849, 36.0101, 42.1412, 79.1916, 8100},
{0, 1.05984, 6.1965, 12.3971, 26.6773, 34.5123, 38.8503, 67.9607, 8100},
{0, 1.07406, 6.15097, 13.4108, 23.062, 31.6812, 37.3077, 57.2156, 8100},
{0, 1.27716, 6.50522, 15.266, 31.25, 40.2676, 40.9496, 83.4598, 8100},
{0, 1.09161, 6.2145, 14.6937, 25.726, 35.6646, 43.8102, 65.4656, 8100},
{0, 1.37356, 7.38264, 14.3081, 22.7334, 28.8201, 43.6075, 75.1262, 8100},
{0, 1.94333, 5.62022, 14.4537, 26.6738, 33.4061, 40.8853, 59.6588, 8100},
{0, 2.03697, 12.6287, 21.2366, 34.4342, 43.4967, 54.9998, 60, 8100},
{0, 1.86647, 7.13185, 15.1822, 20.2075, 22.4894, 30, 40, 8100},
{0, 2.18716, 7.9365, 15.6656, 25.0005, 29.7029, 31.755, 45, 8100},
{0, 1.79435, 7.24238, 14.6779, 20, 23.1849, 24.6151, 29.9996, 8100},
{0, 1.903, 8.60739, 17.3971, 28.7765, 39.0144, 52.9377, 80.1161, 8100},
{0, 2.0422, 9.39568, 19.6939, 38.8651, 49.4168, 70.8184, 117.258, 8100},
{0, 2.05025, 8.91676, 16.2662, 20.635, 22.7219, 30.0003, 50, 8100},
{0, 2.12015, 9.326, 20.0518, 28.0012, 34.1434, 45.5466, 49.9966, 8100},
{0, 2.58767, 9.7282, 22.1808, 44.8816, 67.8511, 87.5578, 99.9969, 8100},
{0, 3.10718, 11.1535, 21.186, 43.4575, 69.117, 102.142, 129.985, 8100},
{0, 3.26593, 12.3316, 22.175, 39.4979, 57.5949, 75.0662, 100, 8100},
{0, 2.48401, 7.28773, 19.7467, 30.0293, 45.2555, 60.7865, 100.537, 8100},
{0, 2.49424, 7.53288, 11.0001, 14, 18, 22, 30, 8100},
{0, 2.79325, 10.4782, 18.7078, 37.416, 44.8931, 53.9337, 69.9999, 8100},
{0, 3.34308, 10.753, 27.5373, 38.8625, 57.2984, 83.477, 139.082, 8100},
{0, 3.237, 10.5464, 23.0534, 46.944, 60.5023, 70.4994, 80.0005, 8100},
{0, 3.357, 10.8631, 22.0952, 47.3909, 78.6842, 117.079, 121.698, 8100},
{0, 3.2013, 10.6812, 19.9973, 30, 60.4706, 70.9416, 100.999, 8100},
{0, 3.05856, 10.4004, 24.423, 42.78, 65.1617, 93.6214, 147.771, 8100},
{0, 3.41072, 11.4925, 22.8054, 41.8445, 68.2985, 91.872, 145.849, 8100}};

Float_t A1t[48][9] = {
{0, 0.684247, -4.14554, -11.1094, -24.5129, -38.2756, -41.0959, -19.0501, 0},
{0, 0.623673, -4.60109, -4.78271, -0.884499, -3.96897, -46.489, -22.5088, 0},
{0, -0.249597, -3.13851, -15.133, -22.6693, -10.2338, -17.5215, -4.78572, 0},
{0, 0.0901572, -3.61432, -5.22235, -26.2322, -19.3028, 3.73157, 2.91283, 0},
{0, -0.135542, -4.18558, -10.3476, -19.8513, -21.8227, -2.345, -47.9072, 0},
{0, -0.528203, -2.78581, -5.76259, -14.4398, -4.65071, 6.9809, 10.9591, 0},
{0, 0.738867, -3.05266, -14.0515, -16.8378, -11.685, -2.39992, 9.73815, 0},
{0, 0.0240829, -4.06428, -7.4387, -25.1373, -15.4164, 4.23546, 30.1758, 0},
{0, 3.76114, -3.88903, -14.2742, -40.7931, -35.2381, -27.9355, 4.64363, 0},
{0, 2.53218, 10.3913, -4.60302, -5.07342, -4.32836, 6.74167, 10.6971, 0},
{0, 0.734454, -1.34727, -8.2282, -24.7553, -12.4964, -2.70228, 12.3389, 0},
{0, -0.607144, -4.88443, -3.87857, -14.385, -14.7764, -0.913896, 9.46744, 0},
{0, 1.18779, -4.14515, -13.6006, -23.6426, -21.2445, -30.6746, -14.5969, 0},
{0, 0.230348, -3.47996, -13.5497, -13.5157, -42.2043, -18.865, -44.7452, 0},
{0, 2.43646, 0.989512, -3.92652, -0.0489174, -5.7052, -13.7881, -34.5402, 0},
{0, 0.508931, -1.76998, -20.025, -27.0253, -19.5194, -42.5512, -18.2329, 0},
{0, -0.279749, -6.95578, -12.9933, -42.4332, -30.4393, -45.8736, -11.4401, 0},
{0, 14.1581, -3.80609, -22.8216, -18.5728, -11.4838, -29.6653, -63.3054, 0},
{0, 0.203648, -5.06129, -17.2442, -15.9798, -41.6745, -34.0871, -93.4167, 0},
{0, 0.527513, -5.25737, -5.07905, -19.5563, -9.89318, -0.0190118, -12.1354, 0},
{0, -0.987681, -4.45351, -10.8727, -22.6132, -11.9238, -8.73394, 0.495203, 0},
{0, -0.307322, -7.51125, -9.25771, -22.5488, -18.0782, -8.63964, -35.6685, 0},
{0, 0.12121, -5.91873, -8.14844, -26.6183, -27.4172, -17.9023, -45.658, 0},
{0, -0.0440127, -4.60804, -8.96586, -14.1608, -16.407, -9.45833, -13.0969, 0},
{0, -0.558487, -5.03789, -11.2856, -31.7381, -32.7836, -10.3583, -51.1819, 0},
{0, -0.319563, -6.28345, -13.2449, -26.0895, -32.1212, -34.8085, -35.3115, 0},
{0, -0.438421, -6.76323, -8.04215, -6.24, -3.51572, -3.26923, -14.7792, 0},
{0, -2.29529, -5.92959, -15.9074, -34.9195, -35.4169, -41.1345, -47.2614, 0},
{0, -2.8399, -19.7623, -14.4555, -22.8945, -25.4665, -22.8025, -8.39844, 0},
{0, -2.3361, -7.44249, -13.7837, -8.14004, -0.0720474, -7.50509, -8.39844, 0},
{0, -3.36944, -8.64707, -13.0254, -16.0805, -1.85751, 8.52888, 0.550509, 0},
{0, -2.18295, -7.89578, -12.4464, -7.93621, -6.2682, 6.24761, 14.8709, 0},
{0, -2.52134, -12.1011, -19.5864, -28.3525, -32.0441, -40.9499, -24.6919, 0},
{0, -2.64214, -12.1746, -21.8649, -49.8572, -50.3294, -77.8674, -91.4326, 0},
{0, -2.83382, -12.3206, -15.0702, -8.91696, -6.24556, -16.5196, -8.39844, 0},
{0, -3.0755, -12.831, -23.0945, -20.8865, -12.8932, -25.4243, 16.5467, 0},
{0, -3.89104, -13.1829, -31.3708, -71.7052, -107.284, -129.239, -108.794, 0},
{0, -5.37954, -17.0676, -24.4585, -49.6695, -89.3567, -134.941, -113.108, 0},
{0, -5.48076, -18.4538, -28.255, -48.7572, -56.4813, -46.2133, -13.7872, 0},
{0, -4.056, -9.7058, -32.0134, -48.0418, -82.316, -115.6, -185.798, 0},
{0, -3.58506, -8.95175, -7.02082, -3.72388, -8.31816, -3.70441, 8.91556, 0},
{0, -4.63926, -15.2966, -20.4456, -49.4273, -33.1319, -30.1151, -94.0042, 0},
{0, -5.9137, -16.5116, -42.9632, -48.5685, -70.2161, -104.088, -200.521, 0},
{0, -5.74618, -16.3597, -36.3691, -79.8407, -94.9335, -99.0161, -72.8879, 0},
{0, -6.07098, -17.0741, -30.4224, -69.8418, -120.675, -166.247, -88.5553, 0},
{0, -5.6577, -16.8394, -31.1924, -40.3184, -106.238, -112.924, -117.987, 0},
{0, -5.44442, -16.6105, -40.8566, -69.5939, -105.485, -143.841, -192.19, 0},
{0, -6.00784, -18.0034, -32.3969, -61.286, -107.103, -136.99, -213.381, 0}};


Float_t A2t[48][9] = {
{0, -0.256726, 2.27431, 6.43469, 11.8465, 25.5581, 31.0446, 22.2011, 0},
{0, -0.27314, 2.41841, 1.96977, 3, 4.1409, 37.5866, 31.8181, 0},
{0, 0.210304, 1.9894, 9.90239, 14.4125, 8.40969, 17.111, 11.0446, 0},
{0, -0.0897531, 1.31427, 1.38261, 13.8423, 10.7517, 3, 3, 0},
{0, 0.254985, 2.18331, 5.68851, 13.1542, 15.3206, 2.48479, 70.3325, 0},
{0, 0.403054, 1.07449, 2.02539, 7.53642, 3.53899, 3, 3, 0},
{0, -0.402018, 1.49429, 8.52411, 9.06339, 6.43531, 2.47331, 3, 0},
{0, 0.0163422, 1.90856, 3.94927, 13.6905, 10.1882, 3, 3, 0},
{0, -4.41759, 0.455884, 8.23305, 24.3049, 23.7324, 19.687, 9.34883, 0},
{0, -1.61978, -5.94435, 5.09418, 3.5784, 6.69858, 0, 3, 0},
{0, -0.406232, 0.261862, 3.87028, 14.9026, 9.15009, 8.09389, 3, 0},
{0, 0.450961, 2.79588, 1.475, 8.87168, 13.0701, 7.88346, 7.61453, 0},
{0, -0.642696, 1.93851, 7.07521, 13.7185, 12.8763, 23.2823, 7.88003, 0},
{0, -0.0242187, 1.90127, 7.67745, 6.80574, 27.4311, 16.7277, 32.2091, 0},
{0, -1.4328, -0.863971, 2, 4.44504, 10.57, 17.3062, 27.8696, 0},
{0, -0.670537, 1.49585, 12.8631, 16.2047, 14.9903, 36.0699, 16.2393, 0},
{0, -0.165185, 4.50708, 7.61307, 25.615, 20.0767, 33.8685, 31.1236, 0},
{0, -14.3972, -0.395553, 13.3646, 4.57022, 3, 7.13162, 73.4341, 0},
{0, -0.12435, 3.28793, 10.8198, 9.77605, 28.3088, 26.0003, 74.0148, 0},
{0, -0.249317, 2.71293, 1.75304, 10.5257, 5.86488, 1.62532, 21.8499, 0},
{0, 0.783725, 2.69445, 6.4717, 14.0212, 10.3889, 12.3235, 12.6467, 0},
{0, 0.251419, 3.89298, 3.7622, 11.1349, 11.8263, 8.91756, 30.8796, 0},
{0, -0.0967801, 3.47969, 4.14906, 15.4672, 17.9873, 14.6814, 40.101, 0},
{0, -0.0468341, 2.4518, 5.22592, 8.81726, 13.3005, 12.5642, 26.0746, 0},
{0, 0.361255, 2.6057, 5.68184, 18.9758, 23.0597, 13.4065, 42.9283, 0},
{0, 0.265048, 3.70422, 6.65035, 15.5919, 21.3769, 27.5229, 32.5316, 0},
{0, 0.131408, 3.54343, 4.01437, 3, 7.44138, 6.65706, 3, 0},
{0, 1.54593, 3.83503, 9.28219, 22.0934, 23.8119, 29.9772, 35.631, 0},
{0, 1.7922, 11.1126, 3.13824, 11.1306, 20.0004, 17.3392, 28.6826, 0},
{0, 1.47885, 4.12127, 7.4205, 3.88672, 3.32699, 12.5934, 28.6826, 0},
{0, 2.19139, 4.68991, 6.91045, 8.68486, 0.558281, 3, 25.2369, 0},
{0, 1.38324, 4.40549, 6.29519, 3.73961, 9.92062, 4.2368, 0, 0},
{0, 1.74636, 7.38456, 10.9205, 16.2117, 20.5563, 27.464, 10.1068, 0},
{0, 1.73446, 6.69344, 11.6119, 29.1162, 29.9171, 49.9535, 44.2648, 0},
{0, 1.77345, 7.29971, 7.85848, 5.92193, 11.9535, 23.2607, 28.6826, 0},
{0, 1.96343, 7.39403, 12.4053, 10.7737, 9.40302, 25.2622, 0, 0},
{0, 2.42414, 7.27305, 18.273, 42.3329, 63.3642, 76.6873, 68.8465, 0},
{0, 3.31282, 9.87776, 12.592, 23.6123, 51.5773, 79.4578, 70.7063, 0},
{0, 3.29838, 10.279, 15.8924, 26.8172, 26.854, 12.7043, 3, 0},
{0, 2.68146, 6.03602, 20.1606, 30.6191, 57.4667, 82.6249, 123.423, 0},
{0, 2.2217, 5.29813, 4.33261, 2.07538, 10.7653, 7.464, 0.349452, 0},
{0, 2.95636, 8.77044, 10.8631, 27.8588, 12.2192, 14.0908, 110.101, 0},
{0, 3.64161, 9.78281, 24.6906, 26.6128, 41.2993, 61.6105, 148.654, 0},
{0, 3.63967, 9.64793, 22.2672, 48.2125, 58.7374, 63.0135, 50.1782, 0},
{0, 3.82742, 10.2333, 17.375, 39.2687, 70.7411, 92.8927, 53.55, 0},
{0, 3.60427, 9.89206, 20.1062, 24.2838, 70.2791, 74.3114, 68.2462, 0},
{0, 3.49785, 10.0213, 25.4603, 42.8919, 66.5058, 87.2153, 118.037, 0},
{0, 3.7133, 10.5679, 19.2454, 36.5673, 67.1997, 85.7778, 144.19, 0}};

// S115B failure: 1158700000 // 2006-09-19 21:06:40 UTC

if (time < 1158700000.) {     // first workaround: just use calibration of PMT S115A
for ( Int_t jj=0; jj<9; jj++ ){
A0t[9][jj] = A0t[8][jj] ; 
A1t[9][jj] = A1t[8][jj] ; 
A2t[9][jj] = A2t[8][jj] ; 
                              }
                         } // if time...


//  2nd-order correction
Float_t corr_a[6][4] = {  
{0.551894 , 2.24353 , -3.35885 , 1.51401} , 
{0.915691 , 4.83967 , -6.85389 , 3.13679} , 
{1.84338 , 5.63517 , -8.41577 , 4.04055} , 
{1.51216 , 11.3233 , -16.2119 , 7.45008} , 
{1.50481 , 15.2538 , -21.2003 , 9.51127} , 
{1.82038 , 17.5336 , -23.5512 , 10.3036}  };

//---------------------------------------------------------------------

//Float_t xv[6],yv[6];
Int_t  iz;

Float_t x1[9],y1[9];
Int_t n1 = 9;

Float_t  Layercharge[6] = {1005., 1005., 1005., 1005., 1005., 1005. };

Int_t  Layerinfo[6] = {0, 0, 0, 0, 0, 0 };

//Float_t adc[48],adc_c[48],PMTcharge_arr[48];
Float_t adc[48],PMTcharge_arr[48];

Float_t beta_mean = fabs(track->beta[12]);

// New!! If "beta_in" != 100. then overwrite beta_mean with beta_in
if (beta_in != 100.) beta_mean=beta_in;

  Int_t ncount=0;
  Float_t charge_chi=1000.;

  
//-------------------------------------------------------------

// fill adc and tdc

   Int_t ich,jj,kk;

   for (ich=0; ich<48; ich++)   adc[ich]   = 4095.;
   //   for (ich=0; ich<48; ich++)   adc_c[ich] = 1000.;
   for (ich=0; ich<48; ich++)   PMTcharge_arr[ich] = 1000.;


if ((beta_mean < 0.05)  || (beta_mean > 2.0)) {
  for (Int_t jj=0; jj<48;jj++) charge_PMT_std[jj] = 1001.;
  for (Int_t jj=0; jj<6;jj++) charge_layer_std[jj] = 1001.; 
  for (Int_t jj=0; jj<6;jj++) charge_layer_std_raw[jj] = 1001.; 
  for (Int_t jj=0; jj<6;jj++) layerinfo_std[jj] = 1001; 
  charge_std = 1001.;
  ncount_std = 0;
                                              }


  if ((beta_mean > 0.05)  && (beta_mean < 2.0)) {

   Float_t betahelp =   pow(beta_mean, 1.2);

//=======================================================================

   for (Int_t ipmt=0; ipmt<L2->GetToFLevel2()->npmt() ; ipmt++){
    ToFPMT *tofpmt = L2->GetToFLevel2()->GetToFPMT(ipmt);
    Int_t pmt_id = tofpmt->pmt_id;
    adc[pmt_id] = tofpmt->adc ;
                                                                       }
       
//=======================================================================

  Float_t yl, yh, ym, m, b, chelp, charge;

              
       for (Int_t ipmt=0; ipmt<track->npmtadc; ipmt++){
          Float_t dEdx = track->dedx[ipmt];
          ym = dEdx*betahelp;
          Int_t pmtadc = track->pmtadc[ipmt];
          //          Int_t adcflag = track->adcflag[ipmt];
          L2->GetToFLevel2()->GetPMTIndex(pmtadc,jj,kk);
          //          if (adcflag==0) adc_c[pmtadc] = dEdx;

          //         if (adc[pmtadc] == 4095) adc_c[pmtadc] = 0.;
         if (adc[pmtadc] == 4095) PMTcharge_arr[pmtadc] = 0.;

         //         if ((adc[pmtadc]>PMTsat[pmtadc]) && (adc[pmtadc]<4095)) adc_c[pmtadc] = 1000.;
                                        
         if (adc[pmtadc] < PMTsat[pmtadc]) {

if ((pmtadc!=4) && (pmtadc!=8) && (pmtadc!=28) && (pmtadc!=31) && (pmtadc!=34) && (pmtadc!=39) && (pmtadc!=41) && (pmtadc!=46)) {

    for ( Int_t jj=0; jj<9; jj++ ){
    x1[jj] = A0t[pmtadc][jj]+A1t[pmtadc][jj]*beta_mean+A2t[pmtadc][jj]*beta_mean*beta_mean;
    y1[jj] = C0t[pmtadc][jj]*C0t[pmtadc][jj] ;
//    cout<<x1[jj]<<" "<<y1[jj]<<" ";
                                  }
//    cout<<endl;


    TGraph *gr2 = new TGraph(n1,x1,y1);
    TSpline3 *spl2 = new TSpline3("grs",gr2);    // use a cubic spline
    chelp = spl2->Eval(ym);
    if (chelp<0.) chelp=0.0001;  //WM new
    charge = TMath::Sqrt(chelp);

    gr2->Delete();
    spl2->Delete();
    PMTcharge_arr[pmtadc] = charge;
                                                                         } // pmtadc != ...

if ((pmtadc==4) || (pmtadc==8) || (pmtadc==28) || (pmtadc==31) || (pmtadc==34) || (pmtadc==39) || (pmtadc==41) || (pmtadc==46)) {
         for ( Int_t jj=0; jj<8; jj++ ){
         yl = A0t[pmtadc][jj]+A1t[pmtadc][jj]*beta_mean+A2t[pmtadc][jj]*beta_mean*beta_mean;
         yh = A0t[pmtadc][jj+1]+A1t[pmtadc][jj+1]*beta_mean+A2t[pmtadc][jj+1]*beta_mean*beta_mean;
         if ((yl<ym)&&(ym<yh)){
         m = (C0t[pmtadc][jj+1]*C0t[pmtadc][jj+1] - C0t[pmtadc][jj]*C0t[pmtadc][jj]) / (yh - yl);
         b = (C0t[pmtadc][jj]*C0t[pmtadc][jj]) - m*yl;
         chelp = m*ym + b;
         if (chelp<0.) chelp=0.0001;  //WM new
         charge= TMath::Sqrt(chelp);
         PMTcharge_arr[pmtadc] = charge;
                             }
                                     } // jj=0..8
                                                                          } // if pmtadc ==....

                                       } // adc < sat
             }   // ipmt...


//====================================================================
//=============  Check ToF standalone using HitPaddle ===================
//====================================================================

Int_t PaddleIdOfTrack[6] = {-1, -1, -1, -1, -1, -1};;

for(jj=0; jj<8; jj++){
Int_t HitPad = L2->GetToFLevel2()->HitPaddle(0,jj);
if (HitPad==1)  PaddleIdOfTrack[0] = jj;
                           }
for(jj=0; jj<6; jj++){
Int_t HitPad = L2->GetToFLevel2()->HitPaddle(1,jj);
if (HitPad==1)  PaddleIdOfTrack[1] = jj;
                           }


for(jj=0; jj<2; jj++){
Int_t HitPad = L2->GetToFLevel2()->HitPaddle(2,jj);
if (HitPad==1)   PaddleIdOfTrack[2] = jj;
                           }
for(jj=0; jj<2; jj++){
Int_t HitPad = L2->GetToFLevel2()->HitPaddle(3,jj);
if (HitPad==1)   PaddleIdOfTrack[3] = jj;
                           }

for(jj=0; jj<3; jj++){
Int_t HitPad = L2->GetToFLevel2()->HitPaddle(4,jj);
if (HitPad==1)   PaddleIdOfTrack[4] = jj;
                           }
for(jj=0; jj<3; jj++){
Int_t HitPad = L2->GetToFLevel2()->HitPaddle(5,jj);
if (HitPad==1)   PaddleIdOfTrack[5] = jj;
                           }

//==============================================================


Int_t ihelp_a[6] = {0,16,28,32,36,42};

for (Int_t ilay=0; ilay<6; ilay ++) {
Int_t jj = PaddleIdOfTrack[ilay] ;
//Float_t xhelp = 1000.;
Float_t xhelp = 1005.;

if (jj == -1) Layerinfo[ilay] = -1;

if (jj != -1) {
Int_t ih = ihelp_a[ilay] + 2*jj;

// New WM
Int_t igood1=0;
Int_t igood2=0;
if ((PMTcharge_arr[ih] < 1000.) && (PMTcharge_arr[ih] > 0.)) igood1=1;
if ((PMTcharge_arr[ih+1] < 1000.) && (PMTcharge_arr[ih+1] > 0.)) igood2=1;

if ((PMTcharge_arr[ih]==0.) && (PMTcharge_arr[ih+1]==0.))  xhelp = 0.;
if ((igood1==1)&&(igood2==1)) xhelp = 0.5*(PMTcharge_arr[ih]+PMTcharge_arr[ih+1]) ;
if ((igood1==1)&&(igood2==0)) xhelp = PMTcharge_arr[ih];
if ((igood2==1)&&(igood1==0)) xhelp = PMTcharge_arr[ih+1];


/*
if ((PMTcharge_arr[ih] < 1000.) && (PMTcharge_arr[ih+1] < 1000.)) xhelp = 0.5*(PMTcharge_arr[ih]+PMTcharge_arr[ih+1]) ;
if ((PMTcharge_arr[ih] < 1000.) && (PMTcharge_arr[ih+1]==1000.))  xhelp = PMTcharge_arr[ih] ;
if ((PMTcharge_arr[ih]==1000.) && (PMTcharge_arr[ih+1] < 1000.))  xhelp = PMTcharge_arr[ih+1] ;
*/

   if (adc[ih] ==  4095) Layerinfo[ilay] = Layerinfo[ilay] + 1;
   if (adc[ih+1] ==  4095) Layerinfo[ilay] = Layerinfo[ilay] + 1;
   if ((adc[ih] > PMTsat[ih]) && (adc[ih] < 4095)) Layerinfo[ilay] = Layerinfo[ilay] + 10;
   if ((adc[ih+1] > PMTsat[ih+1]) && (adc[ih+1] < 4095)) Layerinfo[ilay] = Layerinfo[ilay] + 10;


       }
Layercharge[ilay] = xhelp;
                                    } // ilay ...


//cout<<"Std: ";
//for (Int_t ilay=0; ilay<6; ilay ++) cout<<Layercharge[ilay]<<" ";

//==========================================================================
//============= Layercharge without 2nd-order correction   =================
//==========================================================================

  for (Int_t jj=0; jj<6;jj++) charge_layer_std_raw[jj] = Layercharge[jj];

//==========================================================================
//=========================   2nd-order correction   =======================
//==========================================================================

Float_t xb = beta_mean;

for (Int_t ii=0; ii<6;  ii++) {
Int_t ic = 0;
Float_t corr[4] = {1., 0., 0., 0.};
//for (Int_t iii=1; iii<9;  iii++) if (fabs(Layercharge[ii] - iii)<0.50) ic = iii;
for (Int_t iii=1; iii<10;  iii++) if (fabs(Layercharge[ii] - iii)<0.50) ic = iii;
if ((ic>0)&&(ic<=5)) for (Int_t iii=0; iii<4;  iii++) corr[iii] = corr_a[ic-1][iii];
if (ic >5) for (Int_t iii=0; iii<4;  iii++) corr[iii] = corr_a[5][iii];

Float_t c_corr = corr[0]+corr[1]*xb+corr[2]*xb*xb+corr[3]*xb*xb*xb;

if ((ic>0)&&(ic<=5)) c_corr = c_corr/(1.*ic) ;
if (ic>5) c_corr = c_corr/6. ;

//cout<<Layercharge[ii]<<" "<<ic<<" "<<c_corr<<" => ";
Layercharge[ii] = Layercharge[ii]/c_corr;
//cout<<Layercharge[ii]<<endl;

                            }


//==========================================================================
//cout<<"In ToFNuclei Std: ";
//for (Int_t ilay=0; ilay<6; ilay ++) cout<<Layercharge[ilay]<<" ";
//cout<<endl;
//==========================================================================


Int_t  icount=0;
Float_t charge_mean = 0.;
for (Int_t ii=0; ii<6;  ii++) {
if (Layercharge[ii] < 1000) charge_mean = charge_mean + Layercharge[ii] ;
if (Layercharge[ii] < 1000) icount++;
                            }
if (icount>0) charge_mean = charge_mean/(1.*icount) ;

if (icount>0) {

//for (Int_t ii=0; ii<6;  ii++) cout<<Layercharge[ii]<<" ";
//cout<<endl;


  vector<float> charge_v;
  Float_t charge_median = 0.;

  ////////  get the median charge ////////////
  for(Int_t ii=0;ii<6;ii++) {
  if (Layercharge[ii] < 1000) charge_v.push_back(Layercharge[ii]);
                            }

  std::sort(charge_v.begin(),charge_v.end());
  charge_median = charge_v[(int)(charge_v.size()/2+charge_v.size()%2)-1];

//cout<<charge_mean<<" "<<charge_median<<endl;

  charge_mean = charge_median;

}  // icount>0

//--- Calculate charge for the second time, use residuals of the single
//--- measurements to get a chi2 value

//  Float_t w_i_arr[8] = {0.1, 0.15, 0.2, 0.25, 0.30, 0.35, 0.45, 0.50};
  Float_t w_i_arr[8] = {0.1, 0.15, 0.2, 0.25, 0.35, 0.40, 0.50, 0.55};

  icount=0;
  Float_t sw=0.;
  Float_t sxw=0.;
  Float_t w_i = 1./0.5;
  Float_t resmax= 3.0;

  iz=1;
  if ((0.0<charge_mean) & (charge_mean<1.5)) iz=1;
  if ((1.5<charge_mean) & (charge_mean<2.5)) iz=2;
  if ((2.5<charge_mean) & (charge_mean<3.5)) iz=3;
  if ((3.5<charge_mean) & (charge_mean<4.5)) iz=4;
  if ((4.5<charge_mean) & (charge_mean<5.5)) iz=5;
  if ((5.5<charge_mean) & (charge_mean<6.5)) iz=6;
  if ((6.5<charge_mean) & (charge_mean<7.5)) iz=7;
  if ((7.5<charge_mean) & (charge_mean<90.)) iz=8;

  w_i = 1./w_i_arr[iz-1];


  for (Int_t jj=0; jj<6;jj++){
    if (Layercharge[jj] < 1000) {
      Float_t res = charge_mean - Layercharge[jj] ;
      if (fabs(res*w_i)<resmax) {;
      icount= icount+1;
      sxw=sxw + Layercharge[jj]*w_i*w_i ;
      sw =sw + w_i*w_i ;
      }
    }
  }

  if (icount>0) charge_chi=sxw/sw;

  ncount =  icount;

//cout<<" => "<<icount<<" "<<charge_chi<<endl;


//  charge_tof = charge_chi;

// changed 30-01-2009 WM

  for (Int_t jj=0; jj<48;jj++) charge_PMT_std[jj] = PMTcharge_arr[jj]; 
  for (Int_t jj=0; jj<6;jj++) charge_layer_std[jj] = Layercharge[jj]; 
  for (Int_t jj=0; jj<6;jj++) layerinfo_std[jj] = Layerinfo[jj]; 
  charge_std = charge_chi;
  ncount_std = ncount;

 } // 0.05 < fabs( beta) < 2.0


} // standalone track found

}
