/*
 * TriggerScanBasic
 * Author  Nagni
 * Modified by G. De Rosa
 * Version 1.2
 * Date 27 Apr 2006
 * Version 1.4
 * Date 26 Jun 2006: Inserted dependences on OrbitalTime and modified histogram ranges.
 *
 * Description:
 * Describe the performance of the Trigger.
 *
 * Parameters:
 * 	TString base            - the path to the root directory for the specific Pamela unpack session 
 * 	TString outDirectory    - the path where to save the output image (Default = base) 
 *      TString format          - the format which will be used for save the produced images (Default = "ps") 
 */

#include <TROOT.h>
#include <TFile.h>
#include <TString.h>
#include <TObjString.h>
#include <TTree.h>
#include <TBranch.h>
#include <TGraph.h>
#include <TStyle.h>
#include <TH1.h>
#include <TPaveText.h>
#include <TCanvas.h>
#include <TLatex.h>
#include <TGraph.h>
#include <physics/trigger/TriggerEvent.h>
#include <EventHeader.h>
#include <PscuHeader.h>
#include <iostream>

using namespace std;

char bit(int decimal, char pos)
{
  return( (decimal>>pos)%2 );
}

void TriggerScanBasic(TString base, TString outDirectory = "", TString format = ""){
    
  std::stringstream sst;
  if (outDirectory == "") outDirectory = base.Data();
  TString filename = ((TObjString*)base.Tokenize('/')->Last())->GetString();
  
  TFile *file  =new TFile(base.Data()) ;
  if (!file){
    printf("file not Found \n");
    return;
  }
  
  TTree *PhysicsTr = (TTree*)file->Get("Physics");
  TBranch *TrigBr = PhysicsTr->GetBranch("Trigger");
  TBranch *HeadBr = PhysicsTr->GetBranch("Header");

  pamela::EventHeader *eh = 0;
  pamela::PscuHeader  *ph = 0;
  
  pamela::trigger::TriggerEvent *triggerEvent = 0;
  
  PhysicsTr->SetBranchAddress("Trigger", &triggerEvent);
  PhysicsTr->SetBranchAddress("Header", &eh);
  
  Long64_t nevents    = TrigBr->GetEntries();
  if (nevents <= 0) {
    printf("nevents = %llu \n", nevents);
    file->Close();
    return;
  }

  /* Histo DEF */

  /*
   * S4 and Calorimeter Counters histograms
   */
  TCanvas *S4CalCanvas = new TCanvas("S4CalCanvas","S4CalCanvas", 1280, 1024);
  S4CalCanvas->Divide(1,2);
  //----------
  sst.str("");
  sst << "S4 Counters based on " << base.Data();
  //----------
  sst.str("");
  sst << "Calorimeter Rate Counters based on " << base.Data();
  TH1F *S4Cal2Histo      = new TH1F("S4Cal2Histo", sst.str().c_str(), 20, 0, 20);

  int i  = 0;
  Int_t size  = nevents;

  Double_t *time = new Double_t[size];
  Double_t *evCounter = new Double_t[size];
  Double_t *deadTime  = new Double_t[size];
  Double_t *liveTime  = new Double_t[size];
  //
  Double_t *trigrate0  = new Double_t[size];
  Double_t *trigrate1  = new Double_t[size];
  Double_t *trigrate2  = new Double_t[size];
  Double_t *trigrate3  = new Double_t[size];
  Double_t *trigrate4  = new Double_t[size];
  Double_t *trigrate5  = new Double_t[size];
  //
  Double_t *pmt1  = new Double_t[size];
  Double_t *pmt2  = new Double_t[size];
  Double_t *pmt3  = new Double_t[size];
  //
  Double_t *s4rate  = new Double_t[size];  
  //
  while(i < nevents){
    HeadBr->GetEntry(i);
    TrigBr->GetEntry(i);
    
    /* time vector fill*/
    ph           = eh->GetPscuHeader();
    time[i]      = ph->GetOrbitalTime();
    
    /* eventCounter vector fill*/
    evCounter[i] = triggerEvent->evcount;

    if( evCounter[i] > 1 ){
      
      /* live time vector fill*/
      liveTime[i] = 0.16 * triggerEvent->dltime[0];
      
      /* dead time vector fill*/
      deadTime[i] = 0.01 * triggerEvent->dltime[1];
      
    }

    gStyle->SetStatY(0.85);
    
    
    /* PMT Plane Counters */
    
    pmt1[i] = 1.*triggerEvent->pmtpl[0];
    pmt2[i] = 1.*triggerEvent->pmtpl[1];
    pmt3[i] = 1.*triggerEvent->pmtpl[2];

    /* Trigger Rate Counters*/

    trigrate0[i] = (1/4.)*triggerEvent->trigrate[0];
    trigrate1[i] = (1/4.)*triggerEvent->trigrate[1];
    trigrate2[i] = (1/4.)*triggerEvent->trigrate[2];
    trigrate3[i] = (1/4.)*triggerEvent->trigrate[3];
    trigrate4[i] = (1/4.)*triggerEvent->trigrate[4];
    trigrate5[i] = (1/4.)*triggerEvent->trigrate[5];
    
    /* S4 and Calorimeter Counters fill*/

    s4rate[i]  = triggerEvent->s4calcount[0];
    S4Cal2Histo->Fill(triggerEvent->s4calcount[1]);
 
    i++;
  }
  
  gStyle->SetOptStat(111111);
  gStyle->SetStatH(0.15);
  gStyle->SetStatW(0.15);

  /*
   * Event Counter graph
   */
  TCanvas *EvCounterCanvas = new TCanvas("EvCounterCanvas","EvCounterCanvas", 1280, 1024);
  //----------
  sst.str("");	
  sst << "Event Counter";
  TGraph *evCounterGraph   = new TGraph(nevents, time, evCounter);
  evCounterGraph->SetTitle(sst.str().c_str());
  evCounterGraph->GetXaxis()->SetTitle("OrbitalTime");
  evCounterGraph->GetXaxis()->CenterTitle();
  evCounterGraph->Draw("APL");

  /*
   * Dead and Live time graphs
   */
  TCanvas *DeadLiveCanvas = new TCanvas("DeadLiveCanvas","DeadLiveCanvas", 1280, 1024);
  DeadLiveCanvas->Divide(1,2);
  //----------
  sst.str("");
  sst << "Graph of Dead Time vs OBT based on " << base.Data();
  TGraph *DeadTimeGraph   = new TGraph(nevents, time, deadTime);
  DeadLiveCanvas->cd(1);
  sst.str("");
  sst << "Dead Time";
  DeadTimeGraph->SetTitle(sst.str().c_str());
  DeadTimeGraph->SetFillColor(2);
  DeadTimeGraph->GetXaxis()->SetTitle("OrbitalTime");
  DeadTimeGraph->GetXaxis()->CenterTitle();
  DeadTimeGraph->GetYaxis()->SetTitle("DeadTime(ms)");
  DeadTimeGraph->GetYaxis()->CenterTitle();
  DeadTimeGraph->SetMaximum(100);
  DeadTimeGraph->Draw("AP");
  //----------
  sst.str("");
  sst << "Graph of Live Time vs OBT based on " << base.Data();
  TGraph *LiveTimeGraph   = new TGraph(nevents, (const Double_t*)time, (const Double_t*)liveTime);
  DeadLiveCanvas->cd(2);
  sst.str("");
  sst << "Live Time";
  LiveTimeGraph->SetTitle(sst.str().c_str());
  LiveTimeGraph->SetFillColor(2);
  LiveTimeGraph->GetXaxis()->SetTitle("OrbitalTime");
  LiveTimeGraph->GetXaxis()->CenterTitle();
  LiveTimeGraph->GetYaxis()->SetTitle("LiveTime(ms)");
  LiveTimeGraph->GetYaxis()->CenterTitle();
  LiveTimeGraph->Draw("AP");

  /*	
   * PMT Plane Counters Graph
   */
  
  TCanvas *PMTCanvas = new TCanvas("PMTCanvas","PMTCanvas", 1280, 1024);
  PMTCanvas->Divide(1,3);
  //----------
  sst.str("");
  sst << "Graph of PMT Plane Counters (S11 AND S12) vs OBT based on " << base.Data();
  TGraph *PMT1Graph   = new TGraph(nevents, time, pmt1);
  PMTCanvas->cd(1);
  sst.str("");
  sst << "PMT Plane Counters (S11 AND S12)";
  PMT1Graph->SetTitle(sst.str().c_str());
  PMT1Graph->SetFillColor(2);
  PMT1Graph->GetXaxis()->SetTitle("OrbitalTime");
  PMT1Graph->GetXaxis()->CenterTitle();
  PMT1Graph->GetYaxis()->SetTitle("PMT Plane Counters");
  PMT1Graph->GetYaxis()->CenterTitle();
  PMT1Graph->Draw("AP");
  //----------
  sst.str("");
  sst << "Graph PMT Plane Counters (S21 AND S22) vs OBT based on " << base.Data();
  TGraph *PMT2Graph   = new TGraph(nevents, time, pmt2);
  PMTCanvas->cd(2);
  sst.str("");
  sst << "PMT Plane Counters (S21 AND S22)";
  PMT2Graph->SetTitle(sst.str().c_str());
  PMT2Graph->SetFillColor(2);
  PMT2Graph->GetXaxis()->SetTitle("OrbitalTime");
  PMT2Graph->GetXaxis()->CenterTitle();
  PMT2Graph->GetYaxis()->SetTitle("PMT Plane Counters");
  PMT2Graph->GetYaxis()->CenterTitle();
  PMT2Graph->Draw("AP");
  //----------
  sst.str("");
  sst << "Graph PMT Plane Counters (S31 AND S32) vs OBT based on " << base.Data();
  TGraph *PMT3Graph   = new TGraph(nevents, time, pmt3);
  PMTCanvas->cd(3);
  sst.str("");
  sst << "PMT Plane Counters (S31 AND S32)";
  PMT3Graph->SetTitle(sst.str().c_str());
  PMT3Graph->SetFillColor(2);
  PMT3Graph->GetXaxis()->SetTitle("OrbitalTime");
  PMT3Graph->GetXaxis()->CenterTitle();
  PMT3Graph->GetYaxis()->SetTitle("PMT Plane Rate Counters");
  PMT3Graph->GetYaxis()->CenterTitle();
  PMT3Graph->Draw("AP");
  //
  //----------

  /*	
   * Trigger Rate Counters draw
   */
  
  TCanvas *TrigRate1Canvas = new TCanvas("TrigRate1Canvas","TrigRate1Canvas", 1280, 1024);
  TrigRate1Canvas->Divide(1,3);
  //----------
  sst.str("");
  sst << "Graph of Trigger Rate Counters (S11 OR S12) AND (S21 OR S22) AND (S31 OR S32) vs OBT based on " << base.Data();

  TGraph *TrigRate0Graph   = new TGraph(nevents, time, trigrate0);
  TrigRate1Canvas->cd(1);
  sst.str("");
  sst << "Trigger Rate Counters (S11 OR S12) AND (S21 OR S22) AND (S31 OR S32)";
  TrigRate0Graph->SetTitle(sst.str().c_str());
  TrigRate0Graph->SetFillColor(2);
  TrigRate0Graph->GetXaxis()->SetTitle("OrbitalTime");
  TrigRate0Graph->GetXaxis()->CenterTitle();
  TrigRate0Graph->GetYaxis()->SetTitle("Trigger Rate Counters (Hz)");
  TrigRate0Graph->GetYaxis()->CenterTitle();
  TrigRate0Graph->Draw("AP");
  //----------
  sst.str("");
  sst << "Graph Trigger Rate Counters (S11 AND S12) AND (S21 AND S22) AND (S31 AND S32) vs OBT based on " << base.Data();
  TGraph *TrigRate1Graph   = new TGraph(nevents, time, trigrate1);
  TrigRate1Canvas->cd(2);
  sst.str("");
  sst << "Trigger Rate Counters (S11 AND S12) AND (S21 AND S22) AND (S31 AND S32)";
  TrigRate1Graph->SetTitle(sst.str().c_str());
  TrigRate1Graph->SetFillColor(2);
  TrigRate1Graph->GetXaxis()->SetTitle("OrbitalTime");
  TrigRate1Graph->GetXaxis()->CenterTitle();
  TrigRate1Graph->GetYaxis()->SetTitle("Trigger Rate Counters (Hz)");
  TrigRate1Graph->GetYaxis()->CenterTitle();
  TrigRate1Graph->Draw("AP");
  //----------
  sst.str("");
  sst << "Graph Trigger Rate Counters  (S21 OR S22) AND (S31 OR S32) vs OBT based on " << base.Data();
  TGraph *TrigRate2Graph   = new TGraph(nevents, time, trigrate2);
  TrigRate1Canvas->cd(3);
  sst.str("");
  sst << "Trigger Rate Counters (S21 OR S22) AND (S31 OR S32) ";
  TrigRate2Graph->SetTitle(sst.str().c_str());
  TrigRate2Graph->SetFillColor(2);
  TrigRate2Graph->GetXaxis()->SetTitle("OrbitalTime");
  TrigRate2Graph->GetXaxis()->CenterTitle();
  TrigRate2Graph->GetYaxis()->SetTitle("Trigger Rate Counters (Hz)");
  TrigRate2Graph->GetYaxis()->CenterTitle();
  TrigRate2Graph->Draw("AP");
  //
  //----------
  //
  TCanvas *TrigRate2Canvas = new TCanvas("TrigRate2Canvas","TrigRate2Canvas", 1280, 1024);
  TrigRate2Canvas->Divide(1,3);
  //----------
  sst.str("");
  sst << "Graph of Trigger Rate Counters (S21 AND S22) AND (S31 AND S32) vs OBT based on " << base.Data();

  TGraph *TrigRate3Graph   = new TGraph(nevents, time, trigrate3);
  TrigRate2Canvas->cd(1);
  sst.str("");
  sst << "Trigger Rate Counters (S21 AND S22) AND (S31 AND S32)";
  TrigRate3Graph->SetTitle(sst.str().c_str());
  TrigRate3Graph->SetFillColor(2);
  TrigRate3Graph->GetXaxis()->SetTitle("OrbitalTime");
  TrigRate3Graph->GetXaxis()->CenterTitle();
  TrigRate3Graph->GetYaxis()->SetTitle("Trigger Rate Counters (Hz)");
  TrigRate3Graph->GetYaxis()->CenterTitle();
  TrigRate3Graph->Draw("AP");
  //----------
  sst.str("");
  sst << "Graph Trigger Rate Counters S12 AND (S21 AND S22) vs OBT based on " << base.Data();
  TGraph *TrigRate4Graph   = new TGraph(nevents, time, trigrate4);
  TrigRate2Canvas->cd(2);
  sst.str("");
  sst << "Trigger Rate Counters S12 AND (S21 AND S22)";
  TrigRate4Graph->SetTitle(sst.str().c_str());
  TrigRate4Graph->SetFillColor(2);
  TrigRate4Graph->GetXaxis()->SetTitle("OrbitalTime");
  TrigRate4Graph->GetXaxis()->CenterTitle();
  TrigRate4Graph->GetYaxis()->SetTitle("Trigger Rate Counters (Hz)");
  TrigRate4Graph->GetYaxis()->CenterTitle();
  TrigRate4Graph->Draw("AP");
  //----------
  sst.str("");
  sst << "Graph Trigger Rate Counters (S11 OR S12) AND (S31 OR S32) vs OBT based on " << base.Data();
  TGraph *TrigRate5Graph   = new TGraph(nevents, time, trigrate5);
  TrigRate2Canvas->cd(3);
  sst.str("");
  sst << "Trigger Rate Counters (S11 OR S12) AND (S31 OR S32)";
  TrigRate5Graph->SetTitle(sst.str().c_str());
  TrigRate5Graph->SetFillColor(2);
  TrigRate5Graph->GetXaxis()->SetTitle("OrbitalTime");
  TrigRate5Graph->GetXaxis()->CenterTitle();
  TrigRate5Graph->GetYaxis()->SetTitle("Trigger Rate Counters (Hz)");
  TrigRate5Graph->GetYaxis()->CenterTitle();
  TrigRate5Graph->Draw("AP");
  //----------

  gStyle->SetOptStat("mr");

  /* S4 and Calorimeter Counters draw*/
  TGraph *s4RateGraph   = new TGraph(nevents, time, s4rate);
  S4CalCanvas->cd(1);
  sst.str("");
  sst << "S4 Trigger Rate Counters";
  s4RateGraph->SetTitle(sst.str().c_str());
  s4RateGraph->SetFillColor(2);
  s4RateGraph->GetXaxis()->SetTitle("OrbitalTime");
  s4RateGraph->GetXaxis()->CenterTitle();
  s4RateGraph->GetYaxis()->SetTitle("Trigger Rate Counters (Hz)");
  s4RateGraph->GetYaxis()->CenterTitle();
  s4RateGraph->Draw("AP");
  //----------
  S4CalCanvas->cd(2);
  gPad->SetLogy(1);
  S4Cal2Histo->GetXaxis()->SetTitle("Hz");
  S4Cal2Histo->Draw();
  //  
  //
  //  
  gStyle->SetOptStat("mr");
  gStyle->SetStatH(0.3);
  gStyle->SetStatW(0.3);
  
  TH1 *htemp;

  TCanvas *RatePMT1 = new TCanvas("RatePMT1", "Rate PMT S11A", 1);
  RatePMT1->Divide (3,3);
  //----------
  RatePMT1->cd(1);
  PhysicsTr->Draw("pmtcount1[0]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S111A");
  //----------
  RatePMT1->cd (2);
  PhysicsTr->Draw("pmtcount1[1]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S112A");
  //----------
  RatePMT1->cd (3);
  PhysicsTr->Draw("pmtcount1[2]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S113A");
  //----------
  RatePMT1->cd(4);
  PhysicsTr->Draw("pmtcount1[3]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S114A");
  //----------
  RatePMT1->cd(5);	
  PhysicsTr->Draw("pmtcount1[4]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S115A");
  //----------
  RatePMT1->cd(6);
  PhysicsTr->Draw("pmtcount1[5]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S116A");
  //----------
  RatePMT1->cd(7);
  PhysicsTr->Draw("pmtcount1[6]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S117A");
  //----------
  RatePMT1->cd(8);
  PhysicsTr->Draw("pmtcount1[7]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S118A");

  TCanvas *RatePMT2 = new TCanvas("RatePMT2", "Rate PMT S11B", 1);
  RatePMT2->Divide (3,3);
  //----------
  RatePMT2->cd(1);
  PhysicsTr-> Draw("pmtcount1[8]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S111B");
  //----------
  RatePMT2->cd(2);
  PhysicsTr->Draw("pmtcount1[9]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S112B");
  //----------
  RatePMT2->cd(3);
  PhysicsTr->Draw("pmtcount1[10]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S113B");
  //----------
  RatePMT2->cd(4);
  PhysicsTr->Draw("pmtcount1[11]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S114B");
  //----------
  RatePMT2->cd(5);
  PhysicsTr-> Draw("pmtcount1[12]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S115B");
  //----------
  RatePMT2->cd(6);
  PhysicsTr-> Draw("pmtcount1[13]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S116B");
  //----------
  RatePMT2->cd(7);
  PhysicsTr-> Draw("pmtcount1[14]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S117B");
  //----------
  RatePMT2->cd(8);
  PhysicsTr->Draw("pmtcount1[15]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S118B");
	    
  TCanvas *RatePMT3 = new TCanvas("RatePMT3", "Rate PMT S12", 1);
  RatePMT3->Divide (3,4);
  //----------
  RatePMT3->cd(1);
  PhysicsTr-> Draw("pmtcount1[16]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S121A");
  //----------
  RatePMT3->cd(2);
  PhysicsTr-> Draw("pmtcount1[17]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S122A");
  //----------
  RatePMT3->cd (3);	
  PhysicsTr-> Draw("pmtcount1[18]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S123A");
  //----------
  RatePMT3->cd(4);
  PhysicsTr->Draw("pmtcount1[19]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S124A");
  //----------
  RatePMT3->cd(5);
  PhysicsTr-> Draw("pmtcount1[20]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S125A");
  //----------
  RatePMT3->cd(6);
  PhysicsTr-> Draw("pmtcount1[21]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S126A");
  //----------
  RatePMT3->cd(7);
  PhysicsTr->Draw("pmtcount1[22]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S121B");
  //----------
  RatePMT3->cd(8);
  PhysicsTr-> Draw("pmtcount1[23]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S122B");  
  //----------
  RatePMT3->cd (9);
  PhysicsTr->Draw("pmtcount2[0]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S123B");  
  //----------
  RatePMT3->cd (10);
  PhysicsTr->Draw("pmtcount2[1]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S124B");    
  //----------
  RatePMT3->cd (11);
  PhysicsTr->Draw("pmtcount2[2]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S125B");   
  //----------
  RatePMT3->cd (12);
  PhysicsTr->Draw("pmtcount2[3]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S126B");  
	    
  TCanvas *RatePMT4 = new TCanvas("RatePMT4", "Rate PMT S2", 1);
  RatePMT4->Divide (3,3);
  //----------
  RatePMT4->cd (1);
  PhysicsTr-> Draw("pmtcount2[4]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S211A"); 
  //----------
  RatePMT4->cd (2);
  PhysicsTr-> Draw("pmtcount2[5]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S212A");
  //----------
  RatePMT4->cd (3);
  PhysicsTr-> Draw("pmtcount2[6]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S211B");  
  //----------
  RatePMT4->cd (4);
  PhysicsTr-> Draw("pmtcount2[7]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S212B"); 
  //----------
  RatePMT4->cd (5);
  PhysicsTr-> Draw("pmtcount2[8]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S221A");
  //----------
  RatePMT4->cd (6);
  PhysicsTr-> Draw("pmtcount2[9]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S222A");
  //----------
  RatePMT4->cd (7);	
  PhysicsTr-> Draw("pmtcount2[10]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S221B"); 
  //----------
  RatePMT4->cd (8);
  PhysicsTr->Draw("pmtcount2[11]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S222B");   
	
  TCanvas *RatePMT5 = new TCanvas("RatePMT5", "Rate PMT S3", 1);
  RatePMT5->Divide (3,4);
  //----------
  RatePMT5->cd (1);
  PhysicsTr-> Draw("pmtcount2[12]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S311A");   
  //----------
  RatePMT5->cd (2);
  PhysicsTr-> Draw("pmtcount2[13]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S312A");
  //----------
  RatePMT5->cd (3);
  PhysicsTr-> Draw("pmtcount2[14]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S313A"); 
  //----------
  RatePMT5->cd (4);
  PhysicsTr-> Draw("pmtcount2[15]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S311B"); 
  //----------
  RatePMT5->cd (5);
  PhysicsTr-> Draw("pmtcount2[16]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S312B");   
  //----------
  RatePMT5->cd (6);
  PhysicsTr-> Draw("pmtcount2[17]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S313B");    
  //----------
  RatePMT5->cd (7);
  PhysicsTr-> Draw("pmtcount2[18]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S321A");   
  //----------
  RatePMT5->cd (8);
  PhysicsTr-> Draw("pmtcount2[19]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S322A");  
  //----------
  RatePMT5->cd (9);
  PhysicsTr-> Draw("pmtcount2[20]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S323A"); 
  //----------
  RatePMT5->cd (10);
  PhysicsTr-> Draw("pmtcount2[21]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S321B"); 
  //----------
  RatePMT5->cd (11);
  PhysicsTr-> Draw("pmtcount2[22]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S322B"); 
  //----------
  gStyle->SetOptStat("mr");
  gStyle->SetStatH(0.3);
  gStyle->SetStatW(0.3);
  RatePMT5->cd (12);
  PhysicsTr-> Draw("pmtcount2[23]");
  htemp = (TH1*)gPad->GetPrimitive("htemp");
  htemp->SetTitle("S323B"); 
  
  // print the output
  const char *saveas = format;
  if ( !strcmp(saveas,"ps") ) {
    // ------print the ps
    
    gStyle->SetOptStat(111111);

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic." << saveas  << "(";
    EvCounterCanvas->Print(sst.str().c_str());
    DeadLiveCanvas->Print(sst.str().c_str());
    PMTCanvas->Print(sst.str().c_str());
    TrigRate1Canvas->Print(sst.str().c_str());
    TrigRate2Canvas->Print(sst.str().c_str());

    gStyle->SetOptStat("mr");
    S4CalCanvas->Print(sst.str().c_str());

    gStyle->SetOptStat("mr");
    RatePMT1->Print(sst.str().c_str());
    RatePMT2->Print(sst.str().c_str());
    RatePMT3->Print(sst.str().c_str());
    RatePMT4->Print(sst.str().c_str());
    //
    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic." << saveas << ")";    
    RatePMT5->Print(sst.str().c_str());

  } else {

    gStyle->SetOptStat(111111);

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic1." << saveas;
    EvCounterCanvas->SaveAs(sst.str().c_str());

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic2." << saveas;
    DeadLiveCanvas->SaveAs(sst.str().c_str());

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic3." << saveas;
    PMTCanvas->SaveAs(sst.str().c_str());
    
    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic4." << saveas;
    TrigRate1Canvas->SaveAs(sst.str().c_str());

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic5." << saveas;
    TrigRate2Canvas->SaveAs(sst.str().c_str());
    
    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic6." << saveas;
    S4CalCanvas->SaveAs(sst.str().c_str());

    gStyle->SetOptStat("mr");

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic13." << saveas;
    RatePMT1->SaveAs(sst.str().c_str());

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic14." << saveas;
    RatePMT2->SaveAs(sst.str().c_str());

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic15." << saveas;
    RatePMT3->SaveAs(sst.str().c_str());

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic16." << saveas;
    RatePMT4->SaveAs(sst.str().c_str());
    
    gStyle->SetOptStat("mr");

    sst.str("");
    sst << outDirectory.Data() << filename.Data() << "TriggerScanBasic17." << saveas;
    RatePMT5->SaveAs(sst.str().c_str());
    
  }

}


int main(int argc, char* argv[]){
  TString path;
  TString outDir        ="./";
  TString format        ="ps";
  
  if (argc < 2){
    printf("You have to insert at least the file to analyze \n");
    printf("Try '--help' for more information. \n");
    exit(1);
  }  
  
  if (!strcmp(argv[1], "--help")){
    printf( "Usage: TriggerScanBasic FILE [OPTION] \n");
    printf( "\t --help                  Print this help and exit \n");	
    printf( "\t -outDir[path]           Path where to put the output [default ./] \n");
    printf( "\t -format[ps]             Format for output files [default 'ps'] \n");
    exit(1);
  }
  
  
  path=argv[1];
  
  for (int i = 2; i < argc; i++){
    
    if (!strcmp(argv[i], "-outDir")){
      if (++i >= argc){
	printf( "-outDir needs arguments. \n");
	printf( "Try '--help' for more information. \n");
	exit(1);
      } 
      else{
	outDir = argv[i];
	continue;
      }
    }
    
    
    
    if (!strcmp(argv[i], "-format")){
      if (++i >= argc){
	printf( "-format needs arguments. \n");
	printf( "Try '--help' for more information. \n");
	exit(1);
      } 
      else{
	format = argv[i];
	continue;
      }
    }
  }

  TriggerScanBasic(argv[1], outDir, format);

}
