/**
 * FTrkQLook_EXPERT.cxx
 *
 * autor: D.Fedele
 * version v1r18
 * Parameters:
 * 	file - the data file to analyze
 *      fromevent - first event to analyze
 *      toevent - last event to analyze
 *      outdir - total path of output file 
 * 	outfile - extension of output file (pdf,ps,gif,jpg)
 * 
 * 
 */
//
#include <fstream>
//
#include <TPaveText.h>
#include <TLatex.h>
#include <TCanvas.h>
#include <TGraph.h>
#include <TFile.h>
#include <TTree.h>
#include <TStyle.h>
//
#include <physics/tracker/TrackerEvent.h>
#include <PscuHeader.h>
#include <EventHeader.h>
#include <RunHeaderEvent.h>
#include <TrkAlarmEvent.h>
#include <tsbt/TsbTRecord.h>
#include <tsbt/TsbTEvent.h>
#include <tsbb/TsbBRecord.h>
#include <tsbb/TsbBEvent.h>
#include <EventCounter.h>
#include <PacketType.h>
//
#define MAXSTORAGE 50000

void FTrkQLook_EXPERT(TString file,Int_t fromevent,Int_t toevent, TString outdir, TString outfile)
{
  // 
  //   obtain information about the data file and select the output dir
  Int_t dwpos = file.Last('/');
  Int_t dwpos1 = file.Last('.');
  TString base,ffile ;
  ffile=file(dwpos+1,dwpos1-(dwpos+1));
  if(dwpos>0) base=file(0,dwpos);

  TString out;
  if(outdir.Length()==0){
    out = base;
  }else{
    out = outdir;
  } 
  if(out.Last('/')+1<out.Length()) out+="/";

  //
  // inizialise the variables and open the file
  pamela::TsbTEvent  *event = 0;
  pamela::TsbTRecord *record = 0;
  pamela::TsbBEvent  *Bevent = 0;
  pamela::TsbBRecord *Brecord = 0;
  pamela::tracker::TrackerEvent *te=0;
  pamela::EventHeader *eh=0,*aleh=0,*eH=0,*eT=0,*eB=0;
  pamela::PscuHeader *ph=0,*pH=0;
  pamela::TrkAlarmEvent *al=0;
  pamela::RunHeaderEvent *reh=0;
  pamela::EventCounter *cod=0;

  pamela::PacketType *pctp=0;

  TFile *datafile = new TFile(file);
  if ( !datafile ){
    printf("No data file, exiting...\n");
    return;
  }

  TTree *tree = (TTree*)datafile->Get("TsbT");
  tree->SetBranchAddress("TsbT",&event);
  tree->SetBranchAddress("Header",&eT);

  TTree *Btree = (TTree*)datafile->Get("TsbB");
  Btree->SetBranchAddress("TsbB",&Bevent);
  Btree->SetBranchAddress("Header",&eB);

  TTree *altr = (TTree*)datafile->Get("TrkAlarm");
  altr->SetBranchAddress("TrkAlarm",&al);
  altr->SetBranchAddress("Header",&aleh);

  TTree *tr   = (TTree*)datafile->Get("Physics");
  tr->SetBranchAddress("Tracker",&te);
  tr->SetBranchAddress("Header",&eh);
   
  TTree *otr  = (TTree*)datafile->Get("RunHeader");
  otr->SetBranchAddress("Header",&eH);
  otr->SetBranchAddress("RunHeader",&reh);

  Long64_t nevent = tr->GetEntries();
  Long64_t neventH = otr->GetEntries();
  Long64_t tnevent = tree->GetEntries();
  Long64_t bnevent = Btree->GetEntries();
  Long64_t alnevent = altr->GetEntries();
  Int_t minevent=0;
  Int_t maxevent=0; 

  printf("Number of total events: %lld\n",nevent);
  printf("Number of header events: %lld\n",neventH);
  printf("Number of TsbT events: %lld\n",tnevent);
  printf("Number of TsbB events: %lld\n",bnevent);
  printf("Number of TrkAlarm packet events: %lld\n",alnevent);

  if (nevent<=0){
    datafile->Close();
    return;
  }
  if ( fromevent > toevent && toevent>0 ){
    printf("It must be fromevent < toevent \n");
    return;
  }
  if ( fromevent > nevent || fromevent < 0 ) {
    printf("You can choose fromevent between 0 (all) and %lld \n",nevent);
    return;
  }
  if ( toevent > nevent || toevent < 0 ) {
    printf("You can choose toevent between 0 (all) and %lld \n",nevent);
    return;
  }
  if ( fromevent == 0 ) {
    minevent = 0;
    maxevent = nevent;
  } else {
    minevent = fromevent;
    if ( toevent > 0 ){
      maxevent = toevent+1;
    } else if (toevent > nevent) {	   
      maxevent = nevent;
    } else {	   
      maxevent = nevent;  
    }
    nevent=maxevent-minevent ;
  }

  //
  // information about the RunHeader
  ULong64_t HOBT[neventH];
  Int_t trk_cal_us[neventH];
  ULong_t acq_info[neventH];   
  Int_t countnboot=1; 
  for (Int_t vi=0; vi<neventH;vi++){
    HOBT[vi]=0;
    trk_cal_us[vi]=0;
    acq_info[vi]=0;
  }
  for (Int_t ev=0; ev<neventH; ev++){
    otr->GetEntry(ev);
    pH = eH->GetPscuHeader();
    HOBT[ev]= pH->GetOrbitalTime();
    trk_cal_us[ev]=reh->TRK_CALIB_USED;
    acq_info[ev]=reh->ACQ_BUILD_INFO;
    if((HOBT[ev]<HOBT[ev-1]) && ev>0)
      countnboot+=1;
  }
  countnboot+=(Int_t)(2*nevent/MAXSTORAGE);
  //   printf("\ncountnboot=%d\n",countnboot);
  
  //
  // other variables definitions
  stringstream oss1,oss2,oss3,oss4,fromfile,isfile,tit;
  Int_t ALARM=0;
  Int_t WARNING[12];

  //
  // information about the entries for the temperatures
  Int_t tmpSize=0,siztmp=0,maxsize=0;
  for (Int_t ev=0; ev<tnevent; ev++){
    tree->GetEntry(ev);
    tmpSize = event->Records->GetEntries();
    maxsize+=tmpSize;
  }
  const Int_t tsize=maxsize;
  Int_t tempt[tsize][5];
  ULong64_t xt[tsize];

  Int_t cntpgtemp=1,cntpgdat=0,cntpgmag=0;
  //
  // information about the temperatures
  for (Int_t ev=0; ev<tnevent; ev++){
    tree->GetEntry(ev);
    tmpSize = event->Records->GetEntries();
    for (Int_t j = 0; j < tmpSize; j++){
      record = (pamela::TsbTRecord*)event->Records->At(j);
      xt[siztmp]=record->RECORD_OBT;
      if((xt[siztmp]<xt[siztmp-1]) && siztmp>0)
	cntpgtemp+=1;
      
      for (Int_t z = 0; z < 5; z++){
	tempt[siztmp][z] = record->TEMPERATURES[z+1];    
      }
      siztmp++;
    }
  }
  
  //
  // information about the entries for the magnetic field
  if(bnevent>0) cntpgmag=1;
  maxsize=0;
  for (Int_t ev=0; ev<bnevent; ev++){
    Btree->GetEntry(ev);
    tmpSize = Bevent->Records->GetEntries();
    maxsize+=tmpSize;
  }
  const Int_t bsize=maxsize;
  Int_t magt[bsize][5];
  ULong64_t xb[bsize];
  siztmp=0;
  //
  // information about the magnetic field
  for (Int_t ev=0; ev<bnevent; ev++){
    Btree->GetEntry(ev);
    tmpSize = Bevent->Records->GetEntries();
    for (Int_t j = 0; j < tmpSize; j++){
      Brecord = (pamela::TsbBRecord*)Bevent->Records->At(j);
      xb[siztmp]=Brecord->RECORD_OBT;
      if((xb[siztmp]<xb[siztmp-1]) && siztmp>0)
	cntpgmag+=1;
      for (Int_t z = 0; z < 5; z++){
	magt[siztmp][z] = Brecord->B_FIELD[z+1];    
      }
      siztmp++;
    }
  }
  
  //
  // open the output text files for the alarms
  TString fname1 = out+ffile; 
  TString fname = out+ffile;
  int slen=fname.Length();
  int last=fname.Last('_');
  if(last<(slen-1))fname.Append('_');   
  fname.Append("FTrk-DSP-report.txt");
  ofstream alarm(fname,ios::out);
  alarm << "TRACKER DSP REPORT - Downlink " << ffile<< endl;
  fname1.Append("_FTrkAlarm-pkt-report.txt");
  ofstream alarm1(fname1,ios::out);
  alarm1 << "TrkAlarm pkt REPORT - Downlink " << ffile<< endl;

  //
  // write the file for the TrkAlarm packet
  if(alnevent==0) alarm1 <<endl<< "------>  NO ALARM!!! <-------"<< endl;
  else{
    Long64_t obt=0;
    for (Int_t ev=0; ev<alnevent; ev++){
      altr->GetEntry(ev);
      ph = aleh->GetPscuHeader();

      if(ph->GetOrbitalTime()<obt && ev>0)
	alarm1<<endl<<"NEW CPU BOOT"<<endl;
      obt=ph->GetOrbitalTime();
	
      alarm1 << "================================================="<< endl;
      alarm1 << "PSCU-Pkt N. "<< ph->GetCounter() ;
      alarm1 << " - OBT "<< ph->GetOrbitalTime() << " ms"<<endl;
      alarm1 << "(ROOT-tree entry "<<ev<<")"<<endl;
      alarm1 << "================================================="<< endl<<endl;
	
      alarm1 << "Alarm variables (range 0-1)"<<endl;
      alarm1 << " ALARM[1]= 0x"<< hex << (Int_t)al->ALARM[0]<<endl;
      alarm1 << " ALARM[2]= 0x"<< (Int_t)al->ALARM[1]<<endl;
      alarm1 << " CmdDuringTrig= 0x"<< (Int_t)al->CmdDuringTrig<<endl;
      alarm1 << " FinalCheck= 0x"<< (Int_t)al->FinalCheck<<endl;
      alarm1 << " FlashData= 0x"<< (Int_t)al->FlashData<<endl;
      alarm1 << " FlashShutdown= 0x"<< (Int_t)al->FlashShutdown<<endl;
      alarm1 << " FlashUpset= 0x"<< (Int_t)al->FlashUpset<<endl;
      alarm1 << " InterCheck= 0x"<< (Int_t)al->InterCheck<<endl;
      alarm1 << " UnknownCmd= 0x"<<(Int_t)al->UnknownCmd<<endl<<endl;

      alarm1 << "Alarm variables (range 0-3F)"<<endl;
      alarm1 << " CmdIDMA= 0x"<< (Int_t)al->CmdIDMA<<endl;
      alarm1 << " DSPSoft= 0x"<< (Int_t)al->DSPSoft<<endl;
      alarm1 << " TrigIDMA= 0x"<< (Int_t)al->TrigIDMA<<endl<<endl;
      
      alarm1 << "Control variables (range 0-3F)"<<endl;
      alarm1 << " DSPBusy= 0x"<< (Int_t)al->DSPBusy<<endl;
      alarm1 << " DSPMask= 0x"<< (Int_t)al->DSPMask<<endl<<endl;

      alarm1 << "Control variables (range 0-1)"<<endl;
      alarm1 << " FlashOn= 0x"<< (Int_t)al->FlashOn<<endl<<endl;

      alarm1 << "Control variables (range 0-3)"<<endl;
      alarm1 << " TrigMask= 0x"<< (Int_t)al->TrigMask<<endl<<endl;

      alarm1 << "Control bits fixed"<<endl;
      alarm1 << " Aswr= 0x"<< (Int_t)al->Aswr<<endl;
      alarm1 << " BID[1]= 0x"<< (Int_t)al->BID[0]<<endl;
      alarm1 << " BID[2]= 0x"<< (Int_t)al->BID[1]<<endl;
      alarm1 << " BID[3]= 0x"<< (Int_t)al->BID[2]<<endl;
      alarm1 << " BID[4]= 0x"<< (Int_t)al->BID[3]<<endl;
      alarm1 << " BID[5]= 0x"<< (Int_t)al->BID[4]<<endl;
      alarm1 << " BID[6]= 0x"<< (Int_t)al->BID[5]<<endl;
      alarm1 << " BID[7]= 0x"<< (Int_t)al->BID[6]<<endl;
      alarm1 << " CmdNum= 0x"<< (Int_t)al->CmdNum<<endl;
      alarm1 << " PNum= 0x"<< (Int_t)al->PNum<< dec <<endl;

      alarm1 << "================================================="<< endl<<endl;
      
    }
  }

  //
  // Set Style options
  gStyle->SetLabelSize(0.05,"x");
  gStyle->SetLabelSize(0.06,"y");
  gStyle->SetStatFontSize(0.075);
  gStyle->SetOptStat(1110);
  gStyle->SetFillColor(10);  
  gStyle->SetStatColor(10);
  gStyle->SetTitleFillColor(10);  
  gStyle->SetTitleFontSize(0.1);
  gStyle->SetTitleOffset(0.8,"y");
  gStyle->SetTitleOffset(0.9,"x");
  gStyle->SetTitleSize(0.06,"y");
  gStyle->SetTitleSize(0.055,"x");

  //
  // Define output canvas, histos and graphs
  TCanvas *CompTimeCanv[countnboot],*EventNumCanv[countnboot],*TempCanv[cntpgtemp],*BfieldCanv[cntpgmag];
  TCanvas *IlluminaCanv,*LandauCanv1,*LandauCanv2,*LandauCanv3;

  TH1F *landau1[12],*landau2[12],*landau3[12],*illuminazione[12];
  TGraph *comprtime[12][countnboot],*temp[5][cntpgtemp],*tempw[5][cntpgtemp],*eventnumb[countnboot],*mag[5][cntpgmag];
  TPad *pad1[12],*pad2[12],*pad3[12],*pad4[12],*pad5[12],*pad6[5],*pad7[5]; 
 
  for(Int_t n = 0; n<12; n++) {

    /* -----------> HISTOGRAMS          */
 
    tit<<"DSP "<<n+1;
    oss1<<"DSP  "<<n+1;
    oss2<<"DSPh  "<<n+1;
    oss3<<"DSP h "<<n+1;
    oss4<<"DSP   h"<<n+1;
     
    landau1[n]= new TH1F(oss1.str().c_str(),tit.str().c_str(),401,-0.5,1200.5);
    landau2[n]= new TH1F(oss2.str().c_str(),tit.str().c_str(),401,-0.5,1200.5);
    landau3[n]= new TH1F(oss3.str().c_str(),tit.str().c_str(),401,-0.5,1200.5);

    illuminazione[n]= new TH1F(oss4.str().c_str(),tit.str().c_str(),3073,-0.5,3072.5);

    tit.str("");
    oss1.str("");
    oss2.str("");
    oss3.str("");
    oss4.str("");
  };            
   

  //***************************************************************************************
  // LOOP on each event
  //***************************************************************************************

  if (fromevent!=0)
    printf("\n Scan of events from %i to %i ... \n",minevent,maxevent-1);
  else 
    printf("\n Scan of events from %i to %i ... \n",minevent+1,maxevent);
  
  TLatex *t=new TLatex();
  TLatex *t1=new TLatex();
  
  TLine li,liva1;

  Int_t minev=minevent,maxev=maxevent,countTEMP=0,countMAG=0;
  ULong_t countALARMx=0,countALARMy=0,countFC[12],countFL1[12],countFL2[12],countFL3[12],countFL4[12],countFL5[12],countFL6[12];

  for(int i=0; i<12; i++){
    countFC[i]=0;
    countFL1[i]=0;
    countFL2[i]=0;
    countFL3[i]=0;
    countFL4[i]=0;
    countFL5[i]=0;
    countFL6[i]=0;
  }

  //
  // Fill temperature graphs
  for(Int_t ii=0; ii<cntpgtemp;ii++){
    //****************************************************************************************
    //Temperature Output Pages
    //****************************************************************************************
    fromfile.str("");
    fromfile<<"FTrkQLook_EXPERT      File: "<<ffile;
    isfile<<"Temperatures vs OBT   pag"<<ii+1;
    TempCanv[ii]=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
    TempCanv[ii]->SetFillColor(10);
    TempCanv[ii]->Range(0,0,100,100);

    t->SetTextFont(32);
    t->SetTextColor(1);
    t->SetTextAlign(12);
    t->SetTextSize(0.02);
    t->DrawLatex(2.,98.7,fromfile.str().c_str());
    t1->SetTextFont(32);
    t1->SetTextColor(1);
    t1->SetTextAlign(12);
    t1->SetTextSize(0.02);
    t1->DrawLatex(78.,98.7,isfile.str().c_str());
    isfile.str("");
    t1->SetTextSize(0.025);
    t1->SetTextColor(196);
    isfile<<"--> Values at 28.5^{o}C (if presents) correspond";
    t1->DrawLatex(50.,22.,isfile.str().c_str());
    isfile.str("");
    isfile<<"    to wrong readings ( > 100^{o}C) of the Tsb";
    t1->DrawLatex(50.,19.,isfile.str().c_str());
    isfile.str("");
    isfile<<"--> Values at 25.5^{o}C (if presents) correspond";
    t1->DrawLatex(50.,12.,isfile.str().c_str());
    isfile.str("");
    isfile<<"    to wrong readings ( < 0^{o}C) of the Tsb";
    t1->DrawLatex(50.,9.,isfile.str().c_str());
    isfile.str("");
  
    Float_t tposy = 0.95;    // up y-coord - top pads
    Float_t thpad = 0;   // pad height
    Float_t tposx1=0;          // left  x-coord - pad column
    Float_t tposx0=0;          //       x-coord - column division
    Float_t twrel = 0;     // relative x size of first sub-column
    Float_t tmarg = 0.004;   // margin among pads
   
    thpad = (tposy-tmarg*5)/3;
    twrel = (1-tmarg*4)/2;
    
    for(Int_t i=0; i<5; i++){
      if ( (i+1)%2==1 ) {
	if(i>1) tposy = tposy-(tmarg*2+thpad);
	tposx1 = tmarg;
	tposx0 = tposx1 + twrel;
      }
      else {
	tposx1 = tposx0 + 2*tmarg;
	tposx0 = tposx1 + twrel;
      }
     
      /* -----------> pad for histograms  */
      pad6[i] = new TPad("pad6"," ",tposx1,tposy-thpad,tposx0,tposy,18,0,0);

      Int_t v=0,vv=0;
      Double_t xt_tmp[tsize], tempt_tmp[tsize];
      Double_t xt_tmpw[tsize], tempt_tmpw[tsize];
      for (Int_t ev=countTEMP; ev<tsize; ev++){
       
	if(ev==tsize-1 && i==4 && ii==cntpgtemp-1) countTEMP=tsize-1;
	if(xt[ev]<xt[ev-1] && ev>countTEMP){
	  if(i==4) countTEMP=ev;
	  break;
	}
	else{
	  tempt_tmp[v]=(Double_t)((Int_t)(((3.3*tempt[ev][i]/4096)-0.586)/0.0231));
	  if(tempt_tmp[v]>100) {
	    tempt_tmp[v]=28.5;
	    tempt_tmpw[vv]=28.5;
	    xt_tmpw[vv++]=(Double_t)xt[ev];
	  }
	  else if(tempt_tmp[v]<0){
	    tempt_tmp[v]=25.5;
	    tempt_tmpw[vv]=25.5;
	    xt_tmpw[vv++]=(Double_t)xt[ev];
	  }
	  xt_tmp[v++]=(Double_t)xt[ev];
	}
      }
      TempCanv[ii]->cd();
      tit<<"T"<<i+5<<" (magnetic module "<<i+1<<")";
      pad6[i]->SetFillColor(10);
      pad6[i]->Draw();
      pad6[i]->cd();
      temp[i][ii]= new TGraph(v,xt_tmp,tempt_tmp);
      temp[i][ii]->SetTitle(tit.str().c_str());
      if(v==1) temp[i][ii]->GetXaxis()->SetNdivisions(-501);
      temp[i][ii]->GetXaxis()->SetLabelSize(0.04);
      temp[i][ii]->GetXaxis()->SetTitleSize(0.04);
      temp[i][ii]->GetXaxis()->SetTitle("OBT (ms)");
      temp[i][ii]->GetXaxis()->CenterTitle();
      temp[i][ii]->GetXaxis()->SetTitleOffset(0.85);
      temp[i][ii]->GetYaxis()->SetTitleOffset(1.2);
      temp[i][ii]->GetYaxis()->SetLabelOffset(0.001);
      temp[i][ii]->GetYaxis()->SetLabelSize(0.04);
      temp[i][ii]->GetYaxis()->SetTitleSize(0.04);
      temp[i][ii]->GetYaxis()->SetTitle("Temperatures ( ^{o}C)");
      temp[i][ii]->GetYaxis()->CenterTitle();
      temp[i][ii]->SetMarkerStyle(21);
      temp[i][ii]->SetMarkerSize(0.2);
      temp[i][ii]->Draw("ap");
      tit.str("");
      if(vv>0){
	tempw[i][ii]= new TGraph(vv,xt_tmpw,tempt_tmpw);
	tempw[i][ii]->SetMarkerStyle(21);
	tempw[i][ii]->SetMarkerSize(0.2);
	tempw[i][ii]->SetMarkerColor(196);
	tempw[i][ii]->Draw("samep");
      }
      TempCanv[ii]->Update();
	
      
    }
    if(countTEMP==tsize-1 && ii==cntpgtemp-1){
      break;
    }
  } 

  //
  // Fill magnetic field graphs
  for(Int_t ii=0; ii<cntpgmag;ii++){
    //****************************************************************************************
    //Magnetic Field Output Pages
    //****************************************************************************************
    fromfile.str("");
    fromfile<<"FTrkQLook_EXPERT      File: "<<ffile;
    isfile<<"Magnetic Axial Field vs OBT   pag"<<ii+1;
    BfieldCanv[ii]=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
    BfieldCanv[ii]->SetFillColor(10);
    BfieldCanv[ii]->Range(0,0,100,100);
    
    t->SetTextFont(32);
    t->SetTextColor(1);
    t->SetTextAlign(12);
    t->SetTextSize(0.02);
    t->DrawLatex(2.,98.7,fromfile.str().c_str());
    t1->SetTextFont(32);
    t1->SetTextColor(1);
    t1->SetTextAlign(12);
    t1->SetTextSize(0.02);
    t1->DrawLatex(70.,98.7,isfile.str().c_str());
    isfile.str("");
  
    li.SetLineStyle(1);
    li.SetLineWidth(1);
    li.SetLineColor(190);
    t1->SetTextColor(190);
    t1->SetTextSize(0.04);
    
    Float_t tposy = 0.95;    // up y-coord - top pads
    Float_t thpad = 0;   // pad height
    Float_t tposx1=0;          // left  x-coord - pad column
    Float_t tposx0=0;          //       x-coord - column division
    Float_t twrel = 0;     // relative x size of first sub-column
    Float_t tmarg = 0.004;   // margin among pads
   
    thpad = (tposy-tmarg*5)/3;
    twrel = (1-tmarg*4)/2;
    
    for(Int_t i=0; i<5; i++){
      if ( (i+1)%2==1 ) {
	if(i>1) tposy = tposy-(tmarg*2+thpad);
	tposx1 = tmarg;
	tposx0 = tposx1 + twrel;
      }
      else {
	tposx1 = tposx0 + 2*tmarg;
	tposx0 = tposx1 + twrel;
      }
      
      /* -----------> pad for histograms  */
      pad7[i] = new TPad("pad7"," ",tposx1,tposy-thpad,tposx0,tposy,18,0,0);
      
      Int_t v=0;
      Double_t xb_tmp[bsize], magt_tmp[bsize];
      for (Int_t ev=countMAG; ev<bsize; ev++){
	
	if(ev==bsize-1 && i==4 && ii==cntpgmag-1) countMAG=bsize-1;
	if(xb[ev]<xb[ev-1] && ev>countMAG){
	  if(i==4) countMAG=ev;
	  break;
	}
	else{
	  magt_tmp[v]=(Double_t)magt[ev][i];
	  xb_tmp[v++]=(Double_t)xb[ev];
	}
      }
      BfieldCanv[ii]->cd();
      tit<<"B"<<i+5<<" (magnetic module "<<i+1<<")";
      pad7[i]->SetFillColor(10);
      pad7[i]->Draw();
      pad7[i]->cd();
      mag[i][ii]= new TGraph(v,xb_tmp,magt_tmp);
      mag[i][ii]->SetTitle(tit.str().c_str());
      if(v==1) mag[i][ii]->GetXaxis()->SetNdivisions(-501);
      mag[i][ii]->GetXaxis()->SetLabelSize(0.04);
      mag[i][ii]->GetXaxis()->SetTitleSize(0.04);
      mag[i][ii]->GetXaxis()->SetTitle("OBT (ms)");
      mag[i][ii]->GetXaxis()->CenterTitle();
      mag[i][ii]->GetXaxis()->SetTitleOffset(0.85);
      if(i!=4) mag[i][ii]->GetYaxis()->SetRangeUser(2047,2058);
      else mag[i][ii]->GetYaxis()->SetRangeUser(2059,2067);
      mag[i][ii]->GetYaxis()->SetTitleOffset(1.2);
      mag[i][ii]->GetYaxis()->SetLabelOffset(0.001);
      mag[i][ii]->GetYaxis()->SetLabelSize(0.03);
      mag[i][ii]->GetYaxis()->SetTitleSize(0.04);
      mag[i][ii]->GetYaxis()->SetTitle("Magnetic Field (au)");
      mag[i][ii]->GetYaxis()->CenterTitle();
      mag[i][ii]->SetMarkerStyle(21);
      mag[i][ii]->SetMarkerSize(0.4);
      mag[i][ii]->Draw("ap");
      Double_t min=0.,max=0,xlat=0.;
      min=mag[i][ii]->GetXaxis()->GetXmin();
      max=mag[i][ii]->GetXaxis()->GetXmax();
      xlat=min+(max-min)/2;
      stringstream mi,ma;
      mi<<"minimum recorded value";
      ma<<"maximum recorded value";
      t->SetTextSize(0.1);
      switch(i){
      case 0:
	li.DrawLine(min,2054.,max,2054.);
	li.DrawLine(min,2048.,max,2048.);
	t1->DrawLatex(xlat,2054.3,ma.str().c_str());
	t1->DrawLatex(xlat,2047.7,mi.str().c_str());	
	break;
 
      case 1:
	li.DrawLine(min,2055.,max,2055.);
	li.DrawLine(min,2050.,max,2050.);
	t1->DrawLatex(xlat,2055.3,ma.str().c_str());
	t1->DrawLatex(xlat,2049.7,mi.str().c_str());
	break;

      case 2:
	li.DrawLine(min,2056.,max,2056.);
	li.DrawLine(min,2051.,max,2051.);
	t1->DrawLatex(xlat,2056.3,ma.str().c_str());
	t1->DrawLatex(xlat,2050.7,mi.str().c_str());
	break;
      
      case 3:
	li.DrawLine(min,2057.,max,2057.);
	li.DrawLine(min,2052.,max,2052.);
	t1->DrawLatex(xlat,2057.3,ma.str().c_str());
	t1->DrawLatex(xlat,2051.7,mi.str().c_str());
	break;

      case 4:
	li.DrawLine(min,2066.,max,2066.);
	li.DrawLine(min,2060.,max,2060.);
	t1->DrawLatex(xlat,2066.15,ma.str().c_str());
	t1->DrawLatex(xlat,2059.85,mi.str().c_str());
	break;

      }
      BfieldCanv[ii]->Update();
      tit.str("");
    }
    if(countMAG==bsize-1 && ii==cntpgmag-1){
      break;
    }
  }
  

  //
  // Fill compressiontime and eventnumber graphs and DSP warnings
  for(Int_t ii=0; ii<countnboot;ii++){
    TPad *pt;         //pad for histos
  
    //****************************************************************************************
    //COMPRESSIONTIME vs. OBT Output Pages
    //****************************************************************************************
    fromfile.str("");
    fromfile<<"FTrkQLook_EXPERT      File: "<<ffile;
    isfile<<"COMPRESSIONTIME vs. OBT   pag"<<ii+1;
    CompTimeCanv[ii]=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
    CompTimeCanv[ii]->SetFillColor(10);
    CompTimeCanv[ii]->Range(0,0,100,100);
    t->SetTextFont(32);
    t->SetTextColor(1);
    t->SetTextAlign(12);
    t->SetTextSize(0.02);
    t->DrawLatex(2.,98.7,fromfile.str().c_str());
    t1->SetTextFont(32);
    t1->SetTextColor(1);
    t1->SetTextAlign(12);
    t1->SetTextSize(0.02);
    t1->DrawLatex(62.,98.7,isfile.str().c_str());
    isfile.str("");

    //****************************************************************************************
    //DSP EVENT NUMBER Output Pages
    //****************************************************************************************

    isfile<<"DSP EVENT NUMBER vs. OBT   pag"<<ii+1;
    EventNumCanv[ii]=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
    EventNumCanv[ii]->SetFillColor(10);
    EventNumCanv[ii]->Range(0,0,100,100);
    t->SetTextFont(32);
    t->SetTextColor(1);
    t->SetTextAlign(12);
    t->SetTextSize(0.02);
    t->DrawLatex(2.,98.7,fromfile.str().c_str());
    t1->SetTextFont(32);
    t1->SetTextColor(1);
    t1->SetTextAlign(12);
    t1->SetTextSize(0.02);
    t1->DrawLatex(62.,98.7,isfile.str().c_str());
    isfile.str("");

    Float_t posy = 0.95;    // up y-coord - top pads
    Float_t hpad = 0;   // pad height
    Float_t posx1=0;          // left  x-coord - pad column
    Float_t posx0=0;          //       x-coord - column division
    Float_t wrel = 0;     // relative x size of first sub-column
    Float_t marg = 0.004;   // margin among pads
   
    hpad = (posy-marg*11)/6;
    wrel = (1-marg*4)/2;
   
    for(Int_t n = 0; n<12; n++) {
     
      if ( (n+1)%2==1 ) {
	if(n>1) posy = posy-(marg*2+hpad);
	posx1 = marg;
	posx0 = posx1 + wrel;
      }
      else {
	posx1 = posx0 + 2*marg;
	posx0 = posx1 + wrel;
      }     
     
      /* -----------> pad for histograms  */
      pad1[n] = new TPad("pad1"," ",posx1,posy-hpad,posx0,posy,18,0,0);
      pad2[n] = new TPad("pad2"," ",posx1,posy-hpad,posx0,posy,18,0,0);
      pad3[n] = new TPad("pad3"," ",posx1,posy-hpad,posx0,posy,18,0,0);
      pad4[n] = new TPad("pad4"," ",posx1,posy-hpad,posx0,posy,18,0,0);
      pad5[n] = new TPad("pad5"," ",posx1,posy-hpad,posx0,posy,18,0,0);
    }
   
    //
    // Obtain information about the tracker data
    // and fill graphs and histos

    Int_t al=0;
    Double_t x[MAXSTORAGE];
    Float_t yc[MAXSTORAGE][12];
    Double_t eventint[MAXSTORAGE];
    Int_t eventn[12];
   
    for (Int_t ev=minev; ev<maxevent; ev++){
      tr->GetEntry(ev);
      ph = eh->GetPscuHeader();
      cod = eh->GetCounter();


      if(ev==maxevent-1) maxev=maxevent-1;

      if((ph->GetOrbitalTime()<x[ev-minev-1] && ev-minev!=0) || ev-minev==MAXSTORAGE){
	maxev=ev;
	break;
      }
      else{
	x[(ev-minev)]= ph->GetOrbitalTime();
	ALARM=0;
	Int_t dsp=0;
	Int_t noacq=0;
	Int_t acq_dsp[12];
	ULong_t acq_info_tmp=0;
	if(cod->Get(pctp->RunHeader)>0) acq_info_tmp=acq_info[cod->Get(pctp->RunHeader)-1];
	else acq_info_tmp=acq_info[cod->Get(pctp->RunHeader)];
	for(Int_t io=0;io<12;io++){
	  acq_dsp[dsp]=(acq_info_tmp>>(12+io))&(~(~0<<1));
	  if(((acq_info_tmp>>(12+io))&(~(~0<<1)))==0)
	    noacq+=1;
	  WARNING[io]=0;
	}

	for(Int_t i=0; i<12; i++){

	  if(i<12-noacq){
	    if(te->DSPnumber[i]<1 || te->DSPnumber[i]>12) {
	      ALARM=1;
	      break;
	    } 

	    dsp=te->DSPnumber[i]-1;
	    yc[(ev-minev)][dsp]= 0.051*te->compressiontime[i];

	    //
	    // Fill Cluster Signal and Lighting of the view histos
	    for(Int_t j=0;j<3;j++){
	      if(te->signcluster[i][j]!=0){
		if((te->addrcluster[i][j]>6 && te->addrcluster[i][j]<505) ||(te->addrcluster[i][j]>518 && te->addrcluster[i][j]<1018)){
		  illuminazione[dsp]->Fill((Float_t)((j*1024.)+te->addrcluster[i][j]));
		}
	      }
	    }
	    if(trk_cal_us[cod->Get(pctp->RunHeader)]!=104){
	      if(!(dsp%2)){
		if(te->signcluster[i][0]!=0) landau1[dsp]->Fill((Float_t)(1024.-te->signcluster[i][0]));
		if(te->signcluster[i][1]!=0) landau2[dsp]->Fill((Float_t)(1024.-te->signcluster[i][1]));
		if(te->signcluster[i][2]!=0) landau3[dsp]->Fill((Float_t)(1024.-te->signcluster[i][2]));
	      }
	      else{
		if(te->signcluster[i][0]!=0) landau1[dsp]->Fill((Float_t)te->signcluster[i][0]);
		if(te->signcluster[i][1]!=0) landau2[dsp]->Fill((Float_t)te->signcluster[i][1]);
		if(te->signcluster[i][2]!=0) landau3[dsp]->Fill((Float_t)te->signcluster[i][2]);
	      }
	    }

	    WARNING[i]=0;
	    if(te->fc[i]!=0){ 
	      if(countFC[dsp]==0){
		ALARM = 1;      //general alarm
		WARNING[i] = 1; // DSP warning
	      }
	      countFC[dsp]+=1;
	    };

	    if(te->fl1[i]!=0){ 
	      if(countFL1[dsp]==0){
		ALARM = 1;      //general alarm
		WARNING[i] = 1; // DSP warning
	      }
	      countFL1[dsp]+=1;
	    };

	    if(te->fl2[i]!=0){ 
	      if(countFL2[dsp]==0){
		ALARM = 1;      //general alarm
		WARNING[i] = 1; // DSP warning
	      }
	      countFL2[dsp]+=1;
	    };

	    if(te->fl3[i]!=0){ 
	      if(countFL3[dsp]==0){
		ALARM = 1;      //general alarm
		WARNING[i] = 1; // DSP warning
	      }
	      countFL3[dsp]+=1;
	    };

	    if(te->fl4[i]!=0){ 
	      if(countFL4[dsp]==0){
		ALARM = 1;      //general alarm
		WARNING[i] = 1; // DSP warning
	      }
	      countFL4[dsp]+=1;
	    };

	    if(te->fl5[i]!=0){ 
	      if(countFL5[dsp]==0){
		ALARM = 1;      //general alarm
		WARNING[i] = 1; // DSP warning
	      }
	      countFL5[dsp]+=1;
	    };

	    if(te->fl6[i]!=0){ 
	      if(countFL6[dsp]==0){
		ALARM = 1;      //general alarm
		WARNING[i] = 1; // DSP warning
	      }
	      countFL6[dsp]+=1;
	    };

	    if(dsp%2==0 && te->alarm[i]!=0){ // hardware alarm
	      if(countALARMy==0)
		ALARM = 1;
	      countALARMy+=1;
	    };

	    if(dsp%2==1 && te->alarm[i]!=0){ // hardware alarm
	      if(countALARMx==0)
		ALARM = 1;
	      countALARMx+=1;
	    };
      
	    //**************************************************************************************
	    // warning for internal number
	    //**************************************************************************************
	    if(ev==minev && i<=10 && te->eventn[i]!=te->eventn[i+1]) ALARM=1;
	    if((ev-minev)>=1 && eventn[i]+1!=te->eventn[i] && te->eventn[i]!=1) ALARM=1;
	  } 
	}
	eventint[(ev-minev)]=te->eventn[0];

	//********************************************************************************************
	// file DSP warning
	//********************************************************************************************

	if(ALARM==1) {	
	  al=1;
	  alarm <<endl<< "================================================="<< endl;
	  alarm << "PSCU-Pkt N. "<< ph->GetCounter() ;
	  alarm << " - OBT "<< ph->GetOrbitalTime() << " ms"<<endl;
	  alarm << "Total events "<<nevent<<endl;
	  alarm << "(ROOT-tree entry "<<ev<<" in page "<<ii+1<<" )"<<endl;
	  alarm << "================================================="<< endl;
	  
	  alarm << " DSPn";
	  alarm << " Counter";
	  alarm << " previous";
	  alarm << " Words";
	  alarm << " crc";
	  alarm << "  FC";
	  alarm << " FL1";
	  alarm << " FL2";
	  alarm << " FL3";
	  alarm << " FL4";
	  alarm << " FL5";
	  alarm << " FL6" << endl;
	  
	  for(Int_t i=0; i<12 ; i++){
	    alarm.width(5); alarm << te->DSPnumber[i];
	    alarm.width(8); alarm << te->eventn[i];
	    alarm.width(8); alarm << eventn[i];
	    alarm.width(6); alarm << te->DATAlength[i];
	    alarm.width(4); alarm << te->crc[i];
	    alarm.width(4); alarm << te->fc[i];
	    alarm.width(4); alarm << te->fl1[i];
	    alarm.width(4); alarm << te->fl2[i];
	    alarm.width(4); alarm << te->fl3[i];
	    alarm.width(4); alarm << te->fl4[i];
	    alarm.width(4); alarm << te->fl5[i];
	    alarm.width(4); alarm << te->fl6[i];
	    if(te->alarm[i]!=0)alarm<<" >> ALARM ";
	    if(WARNING[i]!=0)alarm<<" (DSP warning) ";
	    alarm    << endl;
	  }
	}
	for(Int_t i=0; i<12 ; i++)
	  eventn[i]=te->eventn[i];
      }
    }
    if(al==0) alarm << endl<< "Page "<<ii+1<< ": ------>  NO ALARM!!! <-------"<<endl;
    if(maxev==maxevent-1) {
      if(countALARMy!=0){
	if(countALARMy%6!=0) alarm << endl<<endl<< "***** "<<countALARMy/6+1 <<" events with hardware alarm in Y-view *****"<<endl;
	else alarm << endl<<endl<< "***** "<<countALARMy/6 <<" events with hardware alarm in Y-view *****"<<endl;
      }
      if(countALARMx!=0){
	if(countALARMx%6!=0) alarm << endl<<endl<< "***** "<<countALARMx/6+1 <<" events with hardware alarm in X-view *****"<<endl;
	else alarm << endl<<endl<< "***** "<<countALARMx/6 <<" events with hardware alarm in X-view *****"<<endl;
      }
      for(int p=0; p<12; p++){
	if(countFC[p]!=0) alarm << endl<<endl<< "***** "<<countFC[p] <<" events with fc!=0 in DSP "<<p+1<<" *****"<<endl;
	if(countFL1[p]!=0) alarm << endl<<endl<< "***** "<<countFL1[p] <<" events with fl1!=0 in DSP "<<p+1<<" *****"<<endl;
	if(countFL2[p]!=0) alarm << endl<<endl<< "***** "<<countFL2[p] <<" events with fl2!=0 in DSP "<<p+1<<" *****"<<endl;
	if(countFL3[p]!=0) alarm << endl<<endl<< "***** "<<countFL3[p] <<" events with fl3!=0 in DSP "<<p+1<<" *****"<<endl;
	if(countFL4[p]!=0) alarm << endl<<endl<< "***** "<<countFL4[p] <<" events with fl4!=0 in DSP "<<p+1<<" *****"<<endl;
	if(countFL5[p]!=0) alarm << endl<<endl<< "***** "<<countFL5[p] <<" events with fl5!=0 in DSP "<<p+1<<" *****"<<endl;
	if(countFL6[p]!=0) alarm << endl<<endl<< "***** "<<countFL6[p] <<" events with fl6!=0 in DSP "<<p+1<<" *****"<<endl;
      }
    }

    //
    // Draw the graphs
    for (Int_t i=0; i<12 ; i++){ 
      Double_t yyc[maxev-minev];
      for (Int_t v=0; v<maxev-minev; v++){
	yyc[v]=yc[v][i];
      }

      CompTimeCanv[ii]->cd();
      pad3[i]->SetFillColor(10);
      pad3[i]->SetFrameFillColor(10);
      pad3[i]->Draw();
      pad3[i]->cd();
      comprtime[i][ii]= new TGraph(maxev-minev,x,yyc);
      oss1<<"DSP  "<<i+1;
      comprtime[i][ii]->SetTitle(oss1.str().c_str());
      comprtime[i][ii]->GetXaxis()->SetTitle("OBT (ms)");
      comprtime[i][ii]->GetXaxis()->CenterTitle();
      comprtime[i][ii]->GetYaxis()->SetTitle("compressiontime (ms)");
      comprtime[i][ii]->GetYaxis()->CenterTitle();
      // comprtime[i][ii]->GetYaxis()->SetRangeUser(0,3);
      comprtime[i][ii]->Draw("ap"); 
      oss1.str(""); 
      CompTimeCanv[ii]->Update();
    }
   
    
    EventNumCanv[ii]->cd();
    pt = new TPad("pt"," ",0.02,0.01,1.,0.98,18,0,0);
    pt->SetFillColor(10);
    pt->SetFrameFillColor(10);
    pt->Draw();
    pt->cd();
    eventnumb[ii]=new TGraph(maxev-minev,x,eventint);
    eventnumb[ii]->SetTitle("");
    eventnumb[ii]->GetXaxis()->SetLabelSize(0.02);
    eventnumb[ii]->GetXaxis()->SetTitleSize(0.03);
    eventnumb[ii]->GetXaxis()->SetTitle("OBT (ms)");
    eventnumb[ii]->GetXaxis()->CenterTitle();
    eventnumb[ii]->GetXaxis()->SetTickLength(0.01);
    eventnumb[ii]->GetXaxis()->SetTitleOffset(1.1);
    eventnumb[ii]->GetYaxis()->SetLabelSize(0.02);
    eventnumb[ii]->GetYaxis()->SetTitleSize(0.03);
    eventnumb[ii]->GetYaxis()->SetTitle("DSP event-number");
    eventnumb[ii]->GetYaxis()->CenterTitle();
    eventnumb[ii]->GetYaxis()->SetTitleOffset(2.);
    eventnumb[ii]->SetMarkerStyle(21);
    eventnumb[ii]->SetMarkerColor(kBlue);
    eventnumb[ii]->SetMarkerSize(0.2);
    oss1.str("");
    oss1<<"number of events in this graph: "<<maxev-minev;
    t->SetTextColor(kBlue);
    t->SetTextSize(0.02);
    double xtex=0,ytex=0;
    xtex=x[maxev-minev-1]-(x[maxev-minev-1]-x[0])/3;
    ytex=1000.+eventnumb[ii]->GetYaxis()->GetXmax();
    t->DrawLatex(xtex,ytex,oss1.str().c_str()); 
    oss1.str("");   
    eventnumb[ii]->Draw("ap");
    EventNumCanv[ii]->Update();
    
    minev=maxev;
    if(maxev==maxevent-1) {
      cntpgdat=ii+1;
      break;
    }
    
  }

  alarm.close();

  //****************************************************************************************
  //Cluster Signal ladder1 Output Pages
  //****************************************************************************************

  fromfile.str("");
  fromfile<<"FTrkQLook_EXPERT      File: "<<ffile;
  isfile<<"Cluster Signal ladder1";
  LandauCanv1=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
  LandauCanv1->SetFillColor(10);
  LandauCanv1->Range(0,0,100,100);

  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");
  
  //****************************************************************************************
  //Cluster Signal ladder2 Output Pages 
  //****************************************************************************************

  isfile<<"Cluster Signal ladder2";
  LandauCanv2=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
  LandauCanv2->SetFillColor(10);
  LandauCanv2->Range(0,0,100,100);

  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");

  //****************************************************************************************
  //Cluster Signal ladder3 Output Pages
  //****************************************************************************************

  isfile<<"Cluster Signal ladder3";
  LandauCanv3=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
  LandauCanv3->SetFillColor(10);
  LandauCanv3->Range(0,0,100,100);

  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");
  
  //****************************************************************************************
  //Lighting of the views Output Pages
  //****************************************************************************************

  isfile<<"lighting of the views";
  IlluminaCanv=new TCanvas(isfile.str().c_str(),isfile.str().c_str(),900,1200);
  IlluminaCanv->SetFillColor(10);
  IlluminaCanv->Range(0,0,100,100);
  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");
  
  //
  // Draw the histos
  for (Int_t i=0; i<12 ; i++){ 
        
    TBox b;

    Float_t maxhist=0,va1x=0;
    LandauCanv1->cd();
    pad1[i]->SetFillColor(10);
    pad1[i]->SetFrameFillColor(10);
    pad1[i]->Draw();
    pad1[i]->cd();
    landau1[i]->GetXaxis()->SetTitle("max signal");
    landau1[i]->GetXaxis()->CenterTitle();
    landau1[i]->Draw("");
    LandauCanv1->Update();

    LandauCanv2->cd();
    pad4[i]->SetFillColor(10);
    pad4[i]->SetFrameFillColor(10);
    pad4[i]->Draw();
    pad4[i]->cd();
    landau2[i]->GetXaxis()->SetTitle("max signal");
    landau2[i]->GetXaxis()->CenterTitle();
    landau2[i]->Draw("");
    LandauCanv2->Update();

    LandauCanv3->cd();
    pad5[i]->SetFillColor(10);
    pad5[i]->SetFrameFillColor(10);
    pad5[i]->Draw();
    pad5[i]->cd();
    landau3[i]->GetXaxis()->SetTitle("max signal");
    landau3[i]->GetXaxis()->CenterTitle();
    landau3[i]->Draw("");
    LandauCanv3->Update();


    li.SetLineColor(38);
    li.SetLineStyle(4);
    li.SetLineWidth(2);
    liva1.SetLineColor(42);
    liva1.SetLineStyle(3);
    liva1.SetLineWidth(1);

    IlluminaCanv->cd();
    pad2[i]->SetFillColor(10);
    pad2[i]->SetFrameFillColor(10);
    pad2[i]->SetLogy();
    pad2[i]->Draw();
    pad2[i]->cd();
    illuminazione[i]->GetXaxis()->SetTitle("strip with max signal");
    illuminazione[i]->GetXaxis()->CenterTitle();
    illuminazione[i]->Draw("");
    maxhist= illuminazione[i]->GetMaximum();
    if(i==0){
      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(768.,0.,2047.,maxhist);
    }
    else if(i==1){
      b.SetFillColor(6);
      b.SetFillStyle(3945);
      b.DrawBox(2944.,0.,3070.,maxhist);

      b.SetFillColor(107);
      b.SetFillStyle(3954);
      //b.DrawBox(384.,0.,512.,maxhist);
      b.DrawBox(2816.,0.,2944.,maxhist);
      b.DrawBox(2048.,0.,2176.,maxhist);
    }
    else if(i==4){
      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(384.,0.,512.,maxhist);
    }
    else if(i==6){
      b.SetFillColor(6);
      b.SetFillStyle(3945);
      b.DrawBox(2560.,0.,2816.,maxhist);
      b.DrawBox(1024.,0.,1535.,maxhist);

      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(512.,0.,768.,maxhist);
      b.DrawBox(1536.,0.,1792.,maxhist); 
   }
    else if(i==7){
      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(512.,0.,768.,maxhist);
    }
    else if(i==8){
      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(512.,0.,768.,maxhist);
    }
    else if(i==9){
      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(256.,0.,384.,maxhist);
      //b.DrawBox(1280.,0.,1408.,maxhist);
      //b.DrawBox(1792.,0.,1920.,maxhist);
    }
    else if(i==10){
      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(2048.,0.,3070.,maxhist);
    }
    else if(i==11){
      b.SetFillColor(6);
      b.SetFillStyle(3945);
      b.DrawBox(768.,0.,1024.,maxhist);

      b.SetFillColor(107);
      b.SetFillStyle(3954);
      b.DrawBox(0.,0.,512.,maxhist);
      b.DrawBox(1920.,0.,2560.,maxhist);
    }
    for(int va=1; va<24; va++){
      va1x=128*va;
      liva1.DrawLine(va1x,0.,va1x,maxhist);
    }
    li.DrawLine(1024.5,0,1024.5,maxhist);
    li.DrawLine(2048.5,0,2048.5,maxhist);
    IlluminaCanv->Update();
  }

  printf("... end of packets. \n");

  //*************************************************************************
  // Save output Files
  //*************************************************************************
  stringstream out1,out2,out3,command;
  Int_t totpg=0;
  if(cntpgtemp>cntpgdat) totpg=cntpgtemp;
  else if(cntpgtemp<=cntpgdat) totpg=cntpgdat;

  for(Int_t fl=0;fl<totpg;fl++){

    if(!strcmp(outfile.Data(),"ps")||!strcmp(outfile.Data(),"pdf")){
      out1.str("");
      out2.str("");
      out3.str("");
      
      out1<<ffile<<"_FTrkQLook_EXPERT.ps(";
      out2<<ffile<<"_FTrkQLook_EXPERT.ps";
      out3<<ffile<<"_FTrkQLook_EXPERT.ps)";
    
      if(totpg==1){
	CompTimeCanv[fl]->Print(out+out1.str().c_str(),"Portrait");
	EventNumCanv[fl]->Print(out+out2.str().c_str(),"Portrait");
	TempCanv[fl]->Print(out+out2.str().c_str(),"Portrait");
	BfieldCanv[fl]->Print(out+out2.str().c_str(),"Portrait");
	LandauCanv1->Print(out+out2.str().c_str(),"Portrait");
	LandauCanv2->Print(out+out2.str().c_str(),"Portrait");
	LandauCanv3->Print(out+out2.str().c_str(),"Portrait");
	IlluminaCanv->Print(out+out3.str().c_str(),"Portrait");
      }
      else if(totpg>1){
	if(fl==0) CompTimeCanv[fl]->Print(out+out1.str().c_str(),"Portrait");
	if(fl>0 && fl<cntpgdat) CompTimeCanv[fl]->Print(out+out2.str().c_str(),"Portrait");
	if(fl<cntpgdat) EventNumCanv[fl]->Print(out+out2.str().c_str(),"Portrait"); 
	if(fl<cntpgmag) BfieldCanv[fl]->Print(out+out2.str().c_str(),"Portrait"); 
	if(fl<cntpgtemp) TempCanv[fl]->Print(out+out2.str().c_str(),"Portrait");
	if(fl==totpg-1){
	  LandauCanv1->Print(out+out2.str().c_str(),"Portrait");
	  LandauCanv2->Print(out+out2.str().c_str(),"Portrait");
	  LandauCanv3->Print(out+out2.str().c_str(),"Portrait");
	  IlluminaCanv->Print(out+out3.str().c_str(),"Portrait");
	}
      }
    }
    else{
      out1.str("");
      out1<<ffile<<"_FTrkQLook_EXPERT-CompTime-pag"<<fl+1<<"."<<outfile.Data();
      if(fl<cntpgdat) CompTimeCanv[fl]->Print(out+out1.str().c_str());
      out1.str("");
      out1<<ffile<<"_FTrkQLook_EXPERT-BField-pag"<<fl+1<<"."<<outfile.Data();
      if(fl<cntpgmag) BfieldCanv[fl]->Print(out+out1.str().c_str());
      out1.str("");
      out1<<ffile<<"_FTrkQLook_EXPERT-Temp-pag"<<fl+1<<"."<<outfile.Data();
      if(fl<cntpgtemp) TempCanv[fl]->Print(out+out1.str().c_str());
      out1.str("");
      out1<<ffile<<"_FTrkQLook_EXPERT-EvNum-pag"<<fl+1<<"."<<outfile.Data();
      if(fl<cntpgdat) EventNumCanv[fl]->Print(out+out1.str().c_str());
      if(fl==totpg-1){
	out1.str("");
	out1<<ffile<<"_FTrkQLook_EXPERT-LandauL1."<<outfile.Data();
	LandauCanv1->Print(out+out1.str().c_str());
	out1.str("");
	out1<<ffile<<"_FTrkQLook_EXPERT-LandauL2."<<outfile.Data();
	LandauCanv2->Print(out+out1.str().c_str());
	out1.str("");
	out1<<ffile<<"_FTrkQLook_EXPERT-LandauL3."<<outfile.Data();
	LandauCanv3->Print(out+out1.str().c_str());
	out1.str("");
	out1<<ffile<<"_FTrkQLook_EXPERT-Lighting."<<outfile.Data();
	IlluminaCanv->Print(out+out1.str().c_str());
      }
    }
  }  

  //
  // Convert ps to pdf if required
  if(!strcmp(outfile.Data(),"pdf")){
    stringstream com;
    com<<"ps2pdf13 "<<out<<ffile<<"_FTrkQLook_EXPERT.ps "<<out<<ffile<<"_FTrkQLook_EXPERT.pdf";
    system(com.str().c_str());
    printf("\n---> ps file converted in pdf format!\n");
    com.str("");
    com<<"rm -f "<<out<<ffile<<"_FTrkQLook_EXPERT.ps ";
    system(com.str().c_str());
    printf("---> ps file removed!\n\n");
    com.str("");
  }

  gROOT->Reset();
  return;
}
