#include <utils/yodaUtility.h>
#include "TrkFunctions.cpp"
#include <iostream>
#include <fstream>
#include <TPaveText.h>
#include <TLatex.h>
#include <TCanvas.h>
#include <TGraph.h>
#include <PscuHeader.h>
#include <EventHeader.h>
#include <TrkAlarmEvent.h>
#include <TStyle.h>
#include <TObjString.h>
#include <physics/tracker/TrackerEvent.h>
#include <tsbt/TsbTRecord.h>
#include <tsbt/TsbTEvent.h>


/**
 * TrkQLook_EXPERT
 *
 * autor: D.Fedele
 * version 1.0
 * Parameters:
 * 	base - the path to the root directory for the specific
 *	       Pamela unpack session
 *      fromevent - first event to analyse
 *      toevent - last event to analyse
 *      outdir - total path of output file 
 * 	outfile - extension of output file (pdf,ps,gif,jpg)
 * 
 * 
 */

void TrkQLook_EXPERT(TString file,Int_t fromevent,Int_t toevent, TString outdir, TString outfile)
{
  // 
  //   obtain information about the data file and select the output dir
  const string filepath=file.Data(); 
  Int_t dwpos = filepath.find("DW_");
  Int_t dwpos1 = filepath.find(".root");
  TString fpath=(filepath.c_str());
  TString base,ffile ;
  stringcopy(ffile,fpath,dwpos,dwpos1);
  stringcopy(base,fpath,0,dwpos);

  TString out;
  if(outdir.Length()==0){
    out = base;
  }else{
    out = outdir;
  } 

  //
  // inizialise the variables and open the file
  pamela::TsbTEvent  *event = 0;
  pamela::TsbTRecord *record = 0;
  pamela::tracker::TrackerEvent *te=0;
  pamela::EventHeader *eh=0,*aleh=0;
  pamela::PscuHeader *ph=0;
  pamela::TrkAlarmEvent *al=0;

  TFile *datafile = new TFile(file);

  TTree *tree = (TTree*)datafile->Get("TsbT");
  tree->SetBranchAddress("TsbT",&event);

  TTree *altr = (TTree*)datafile->Get("TrkAlarm");
  altr->SetBranchAddress("TrkAlarm",&al);
  altr->SetBranchAddress("Header",&aleh);

  TTree *tr   = (TTree*)datafile->Get("Physics");
  tr->SetBranchAddress("Tracker",&te);
  tr->SetBranchAddress("Header",&eh);
   
  Long64_t nevent = tr->GetEntries();
  Long64_t tnevent = tree->GetEntries();
  Long64_t alnevent = altr->GetEntries();
  Int_t minevent=0;
  Int_t maxevent=0; 

//   cout<<"Total n.events  "<<nevent <<endl;

  if (nevent<=0){
    datafile->Close();
    return;
  }
  if ( fromevent > toevent && toevent>0 ){
    printf("It must be fromevent < toevent \n");
    return;
  }
  if ( fromevent > nevent || fromevent < 0 ) {
    printf("You can choose fromevent between 0 (all) and %lld \n",nevent);
    return;
  }
  if ( toevent > nevent || toevent < 0 ) {
    printf("You can choose toevent between 0 (all) and %lld \n",nevent);
    return;
  }
  if ( fromevent == 0 ) {
    minevent = 0;
    maxevent = nevent;
  } else {
    minevent = fromevent;
    if ( toevent > 0 ){
      maxevent = toevent+1;
    } else if (toevent > nevent) {	   
      maxevent = nevent;
    } else {	   
      maxevent = toevent+1;   
    }
    nevent=maxevent-minevent ;
  }


  //if (fromevent!=0)cout << "Scan of events from "<<minevent << " to "<< maxevent-1<< endl << endl;
  //else cout << "Scan of events from "<<minevent+1 << " to "<< maxevent<< endl << endl;

  //
  // other variables definitions
  stringstream oss1,oss2,oss3,oss4,fromfile,isfile,tit;
  Int_t warning_dspnumber=0;
  Int_t ALARM=0;
  Int_t WARNING[12];
  const Int_t size=nevent;
  Int_t dsp=0;
  Double_t yd[size*12], yc[size*12], x[size], eventint[size];
  Double_t yyc[size], hi=0, land[size*12][3], illum[size*12][3] ;

  TGraph *comprtime,*temp[5];
  TGraph *eventnumb;
  TH1F *landau1[12],*landau2[12],*landau3[12],*illuminazione[12];
  TPaveText *pt1;
  TPad *pt;
  
  //
  // information about the entries for the temperatures
  Int_t tmpSize=0,siztmp=0,maxsize=0;
  for (Int_t ev=0; ev<tnevent; ev++){
    tree->GetEntry(ev);
    tmpSize = event->Records->GetEntries();
    maxsize+=tmpSize;
  }
  const Int_t tsize=maxsize;
  Int_t tempt[tsize][5],xt[tsize], tempt_tmp[tsize];

  
  //
  // open the output text files for the alarms 
  ofstream warning(out + "warning.txt",ios::out);
  TString fname = out+ffile;
  TString fname1 = out+ffile;
  int slen=fname.Length();
  int last=fname.Last('_');
  if(last<(slen-1))fname.Append('_');   
  fname.Append("Trk-ALARM-report.txt");
  ofstream alarm(fname,ios::out);
  alarm << "TRACKER ALARM REPORT - Downlink " << ffile<< endl;
  fname1.Append("_TrkAlarm-pkt-report.txt");
  ofstream alarm1(fname1,ios::out);
  alarm1 << "TrkAlarm pkt REPORT - Downlink " << ffile<< endl;

  //
  // write the file for the TrkAlarm packet
  if(alnevent==0) alarm1 <<endl<< "------>  NO ALARM!!! <-------"<< endl;
  else{
    for (Int_t ev=0; ev<alnevent; ev++){
      altr->GetEntry(ev);
      ph = aleh->GetPscuHeader();

      alarm1 << "================================================="<< endl;
      alarm1 << "PSCU-Pkt N. "<< ph->GetCounter() ;
      alarm1 << " - OBT "<< ph->GetOrbitalTime() << " ms"<<endl;
      alarm1 << "(ROOT-tree entry "<<ev<<")"<<endl;
      alarm1 << "================================================="<< endl;
    
      alarm1 << " ALARM[1]= "<<(Int_t)al->ALARM[1]<<endl;
      alarm1 << " ALARM[2]= "<<(Int_t)al->ALARM[2]<<endl;
      alarm1 << " Aswr= "<<(Int_t)al->Aswr<<endl;
      alarm1 << " BID[1]= "<<(Int_t)al->BID[0]<<endl;
      alarm1 << " BID[2]= "<<(Int_t)al->BID[1]<<endl;
      alarm1 << " BID[3]= "<<(Int_t)al->BID[2]<<endl;
      alarm1 << " BID[4]= "<<(Int_t)al->BID[3]<<endl;
      alarm1 << " BID[5]= "<<(Int_t)al->BID[4]<<endl;
      alarm1 << " BID[6]= "<<(Int_t)al->BID[5]<<endl;
      alarm1 << " BID[7]= "<<(Int_t)al->BID[6]<<endl;
      alarm1 << " CmdDuringTrig= "<<(Int_t)al->CmdDuringTrig<<endl;
      alarm1 << " CmdIDMA= "<<(Int_t)al->CmdIDMA<<endl;
      alarm1 << " CmdNum= "<<(Int_t)al->CmdNum<<endl;
      alarm1 << " DSPBusy= "<<(Int_t)al->DSPBusy<<endl;
      alarm1 << " DSPMask= "<<(Int_t)al->DSPMask<<endl;
      alarm1 << " DSPSoft= "<<(Int_t)al->DSPSoft<<endl;
      alarm1 << " FinalCheck= "<<(Int_t)al->FinalCheck<<endl;
      alarm1 << " FlashData= "<<(Int_t)al->FlashData<<endl;
      alarm1 << " FlashOn= "<<(Int_t)al->FlashOn<<endl;
      alarm1 << " FlashShutdown= "<<(Int_t)al->FlashShutdown<<endl;
      alarm1 << " FlashUpset= "<<(Int_t)al->FlashUpset<<endl;
      alarm1 << " InterCheck= "<<(Int_t)al->InterCheck<<endl;
      alarm1 << " PNum= "<<(Int_t)al->PNum<<endl;
      alarm1 << " TrigIDMA= "<<(Int_t)al->TrigIDMA<<endl;
      alarm1 << " TrigMask= "<<(Int_t)al->TrigMask<<endl;
      alarm1 << " UnknownCmd= "<<(Int_t)al->UnknownCmd<<endl;
    }
  }

  //***************************************************************************************
  // LOOP on each event
  //***************************************************************************************

  //
  // information about the temperatures
  for (Int_t ev=0; ev<tnevent; ev++){
    tree->GetEntry(ev);
    tmpSize = event->Records->GetEntries();
    for (Int_t j = 0; j < tmpSize; j++){
      record = (pamela::TsbTRecord*)event->Records->At(j);
      xt[siztmp]=record->RECORD_OBT;  //0.051*record->RECORD_OBT;   
      for (Int_t z = 0; z < 5; z++){
	tempt[siztmp][z] = record->TEMPERATURES[z+1];    
      }
      siztmp++;
    }
  }

  //
  // information about the tracker data
  for (Int_t ev=minevent; ev<maxevent; ev++){
    tr->GetEntry(ev);
    ph = eh->GetPscuHeader();
    dsp=0;
    ALARM=0;
    for(Int_t i=0; i<12; i++){
      dsp=te->DSPnumber[i]-1;
      yd[(ev-minevent)*12+dsp]= te->DATAlength[i];
      yc[(ev-minevent)*12+dsp]= 0.051*te->compressiontime[i];
      for(Int_t j=0;j<3;j++){
	illum[(ev-minevent)*12+dsp][j]=te->addrcluster[i][j];
	if(!(dsp%2)&&(te->signcluster[i][j])!=0)
	  land[(ev-minevent)*12+dsp][j]=1024-(te->signcluster[i][j]);
	else land[(ev-minevent)*12+dsp][j]=te->signcluster[i][j];
      }
      x[(ev-minevent)]= ph->GetOrbitalTime();
       
      WARNING[i]=0;
      if(te->fc[i]!=0 || te->fl1[i]!=0 || te->fl2[i]!=0 || te->fl3[i]!=0 || te->fl4[i]!=0 || te->fl5[i]!=0 || te->fl6[i]!=0){
	ALARM = 1;      //general alarm
	WARNING[i] = 1; // DSP warning
      };

      if(te->alarm[i]!=0){ // hardware alarm
	ALARM = 1;
      };
      
      //**************************************************************************************
      // warning for internal number
      //**************************************************************************************
      if(i<=10){
	if(te->eventn[i]!=te->eventn[i+1]){
	  warning_dspnumber++;
	  warning<< "==> WARNING!!    Check entry "<< ev<<" (DSP "<<dsp<<")" <<endl<< 
	            "eventn["<<i<<"]= "<<te->eventn[i]<<"!= eventn["<<i+1<<"]= "<< te->eventn[i+1]<<endl;
	}
	else 
	  eventint[(ev-minevent)]=te->eventn[0];
      }
     }

    if(ev<=maxevent-1){

      if((ev-minevent)>=1 && eventint[(ev-minevent)]!=eventint[(ev-minevent)-1]+1 && eventint[(ev-minevent)]!=1){
	warning_dspnumber++;
	warning<< "==> WARNING!!    Check entry "<< ev<<endl<<
	          " DSP event num.= "<< eventint[(ev-minevent)]<<
	          " is different from (previus+1) ="<<eventint[(ev-minevent)-1]+1 <<"\n"<<endl;
      }
    };
      
    //********************************************************************************************
    // file ALARM
    //********************************************************************************************
    if(ALARM==1) {

      alarm << "================================================="<< endl;
      alarm << "PSCU-Pkt N. "<< ph->GetCounter() ;
      alarm << " - OBT "<< ph->GetOrbitalTime() << " ms"<<endl;
      alarm << "(ROOT-tree entry "<<ev<<")"<<endl;
      alarm << "================================================="<< endl;
      
      alarm << " DSPn";
      alarm << " Event";
      alarm << " Words";
      alarm << " crc";
      alarm << "  FC";
      alarm << " FL1";
      alarm << " FL2";
      alarm << " FL3";
      alarm << " FL4";
      alarm << " FL5";
      alarm << " FL6" << endl;
      
      for(Int_t i=0; i<12 ; i++){
	alarm.width(5); alarm << te->DSPnumber[i];
	alarm.width(6); alarm << te->eventn[i];
	alarm.width(6); alarm << te->DATAlength[i];
	alarm.width(4); alarm << te->crc[i];
	alarm.width(4); alarm << te->fc[i];
	alarm.width(4); alarm << te->fl1[i];
	alarm.width(4); alarm << te->fl2[i];
	alarm.width(4); alarm << te->fl3[i];
	alarm.width(4); alarm << te->fl4[i];
	alarm.width(4); alarm << te->fl5[i];
	alarm.width(4); alarm << te->fl6[i];
	if(te->alarm[i]!=0)alarm<<" >> ALARM ";
	if(WARNING[i]!=0)alarm<<" (DSP warning) ";
	alarm    << endl;
      }
    }
  }
  if(ALARM==0) alarm << endl<< "------>  NO ALARM!!! <-------"<<endl;
  
  if(warning_dspnumber==0) warning<<"NONE"<<endl;

  //*****************************************
  //close all files
  //*****************************************   
  warning.close();
  alarm.close();
  datafile->Close();


  gStyle->SetLabelSize(0.05,"x");
  gStyle->SetLabelSize(0.06,"y");
  gStyle->SetStatFontSize(0.075);
  gStyle->SetOptStat(1110);
  gStyle->SetFillColor(10);  
  gStyle->SetTitleFillColor(10);  
  gStyle->SetTitleFontSize(0.1);
  gStyle->SetTitleOffset(0.8,"y");
  gStyle->SetTitleOffset(0.9,"x");
  gStyle->SetTitleSize(0.06,"y");
  gStyle->SetTitleSize(0.055,"x");

  //****************************************************************************************
  //Output figures --- First sheet:
  //****************************************************************************************
  
  TCanvas *CompTimeCanv=new TCanvas("Tracker_Detector_Report_4/6","",900,1200);
  CompTimeCanv->SetFillColor(10);
  CompTimeCanv->Range(0,0,100,100);
  fromfile<<"TrackerQLookExpert      File: "<<ffile;
  isfile<<"COMPRESSIONTIME vs. OBT";
  TLatex *t=new TLatex();
  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  TLatex *t1=new TLatex();
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(64.,98.7,isfile.str().c_str());
  isfile.str("");

  //****************************************************************************************
  //Output figures --- Second sheet:
  //****************************************************************************************

  TCanvas *EventNumCanv=new TCanvas("Tracker_Detector_Report_5/6","",900,1200);
  EventNumCanv->SetFillColor(10);
  EventNumCanv->Range(0,0,100,100);
  isfile<<"WARNINGS on DSP EVENT NUMBER ";
  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(68.4,98.7,isfile.str().c_str());
  isfile.str("");

  //****************************************************************************************
  //Output figures --- Third sheet:
  //****************************************************************************************

  TCanvas *LandauCanv1=new TCanvas("Landau ladder1","",900,1200);
  LandauCanv1->SetFillColor(10);
  LandauCanv1->Range(0,0,100,100);
  isfile<<"Landau ladder1";

  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");
  //   //****************************************************************************************
//   //Output figures --- Third sheet:
//   //****************************************************************************************

  TCanvas *LandauCanv2=new TCanvas("Landau ladder2","",900,1200);
  LandauCanv2->SetFillColor(10);
  LandauCanv2->Range(0,0,100,100);
  isfile<<"Landau ladder2";

  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");

  //   //****************************************************************************************
//   //Output figures --- Third sheet:
//   //****************************************************************************************

  TCanvas *LandauCanv3=new TCanvas("Landau ladder3","",900,1200);
  LandauCanv3->SetFillColor(10);
  LandauCanv3->Range(0,0,100,100);
  isfile<<"Landau ladder3";

  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");
  
//   //****************************************************************************************
//   //Output figures --- Fourth sheet:
//   //****************************************************************************************

  TCanvas *IlluminaCanv=new TCanvas("Illuminazione","",900,1200);
  IlluminaCanv->SetFillColor(10);
  IlluminaCanv->Range(0,0,100,100);
  isfile<<"lighting of the views";
  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(70.,98.7,isfile.str().c_str());
  isfile.str("");

 //****************************************************************************************
  //Output figures --- Fifth sheet:
  //****************************************************************************************
 
  TCanvas *TempCanv=new TCanvas("Tracker_Detector_Temperaturt","",900,1200);
  TempCanv->SetFillColor(10);
  TempCanv->Range(0,0,100,100);
  isfile<<"Temperatures vs OBT";

  t->SetTextFont(32);
  t->SetTextColor(1);
  t->SetTextAlign(12);
  t->SetTextSize(0.02);
  t->DrawLatex(2.,98.7,fromfile.str().c_str());
  t1->SetTextFont(32);
  t1->SetTextColor(1);
  t1->SetTextAlign(12);
  t1->SetTextSize(0.02);
  t1->DrawLatex(80.,98.7,isfile.str().c_str());
  isfile.str("");
  
  //*************************************************************************************
  //book pads and histos
  //***************************************************************************************

  TPad *pad1[12],*pad2[12],*pad3[12],*pad4[12],*pad5[12],*pad6[5];          //pad for histos
  Double_t posy = 0.95;    // up y-coord - top pads
  Double_t hpad = 0;   // pad height
  Double_t posx1=0;          // left  x-coord - pad column
  Double_t posx0=0;          //       x-coord - column division
  Double_t wrel = 0;     // relative x size of first sub-column
  Double_t marg = 0.004;   // margin among pads
  Double_t tposy = 0.95;    // up y-coord - top pads
  Double_t thpad = 0;   // pad height
  Double_t tposx1=0;          // left  x-coord - pad column
  Double_t tposx0=0;          //       x-coord - column division
  Double_t twrel = 0;     // relative x size of first sub-column
  Double_t tmarg = 0.004;   // margin among pads

  hpad = (posy-marg*11)/6;
  wrel = (1-marg*4)/2;
  thpad = (tposy-tmarg*5)/3;
  twrel = (1-tmarg*4)/2;
  stringstream title;
  stringstream hid;

  for(Int_t n = 0; n<12; n++) {
    
    if ( (n+1)%2==1 ) {
      if(n>1) posy = posy-(marg*2+hpad);
      posx1 = marg;
      posx0 = posx1 + wrel;
    }
    else {
      posx1 = posx0 + 2*marg;
      posx0 = posx1 + wrel;
    }
    
    if ( (n+1)%2==1 ) {
      if(n>1) tposy = tposy-(tmarg*2+thpad);
      tposx1 = tmarg;
      tposx0 = tposx1 + twrel;
    }
    else {
      tposx1 = tposx0 + 2*tmarg;
      tposx0 = tposx1 + twrel;
    }


    /* -----------> pad for histograms  */
    pad1[n] = new TPad("pad1"," ",posx1,posy-hpad,posx0,posy,18,0,0);
    pad2[n] = new TPad("pad2"," ",posx1,posy-hpad,posx0,posy,18,0,0);
    pad3[n] = new TPad("pad3"," ",posx1,posy-hpad,posx0,posy,18,0,0);
    pad4[n] = new TPad("pad4"," ",posx1,posy-hpad,posx0,posy,18,0,0);
    pad5[n] = new TPad("pad5"," ",posx1,posy-hpad,posx0,posy,18,0,0);
    if(n<5) pad6[n] = new TPad("pad6"," ",tposx1,tposy-thpad,tposx0,tposy,18,0,0);


    /* -----------> HISTOGRAMS          */
 
    oss1<<"DSP  "<<n+1;
    oss2<<"DSPh  "<<n+1;
    oss3<<"DSP h "<<n+1;
    oss4<<"DSP   h"<<n+1;
     
    landau1[n]=new TH1F(oss1.str().c_str(),oss1.str().c_str(),1200,0.5,1200.5);
    landau1[n]->GetXaxis()->SetTitle("max signal");
    landau1[n]->GetXaxis()->CenterTitle();
    landau2[n]=new TH1F(oss2.str().c_str(),oss1.str().c_str(),1200,0.5,1200.5);
    landau2[n]->GetXaxis()->SetTitle("max signal");
    landau2[n]->GetXaxis()->CenterTitle();
    landau3[n]=new TH1F(oss3.str().c_str(),oss1.str().c_str(),1200,0.5,1200.5);
    landau3[n]->GetXaxis()->SetTitle("max signal");
    landau3[n]->GetXaxis()->CenterTitle();

    illuminazione[n]=new TH1F(oss4.str().c_str(),oss1.str().c_str(),3072,0.5,3072.5);
    illuminazione[n]->GetXaxis()->SetTitle("strip with max signal");
    illuminazione[n]->GetXaxis()->CenterTitle();

    oss1.str("");
    oss2.str("");
    oss3.str("");
    oss4.str("");
  };            
   

  //**********************************************************************************
  // Fill Graphs and Histos
  //**********************************************************************************

  for (Int_t i=0; i<12 ; i++){ 
 
    for (Int_t ev=minevent; ev<maxevent; ev++){
      yyc[(ev-minevent)]=yc[12*(ev-minevent)+i];
      landau1[i]->Fill(land[12*(ev-minevent)+i][1]);
      landau2[i]->Fill(land[12*(ev-minevent)+i][2]);
      landau3[i]->Fill(land[12*(ev-minevent)+i][3]);
      for(Int_t j=0;j<3;j++){
	hi=(j*1024)+illum[12*(ev-minevent)+i][j];
	illuminazione[i]->Fill(hi);
      }
    }
    
    LandauCanv1->cd();
    pad1[i]->SetFillColor(10);
    pad1[i]->SetFrameFillColor(10);
    pad1[i]->Draw();
    pad1[i]->cd();
    landau1[i]->Draw("");
    LandauCanv2->cd();
    pad4[i]->SetFillColor(10);
    pad4[i]->SetFrameFillColor(10);
    pad4[i]->Draw();
    pad4[i]->cd();
    landau2[i]->Draw("");
    LandauCanv3->cd();
    pad5[i]->SetFillColor(10);
    pad5[i]->SetFrameFillColor(10);
    pad5[i]->Draw();
    pad5[i]->cd();
    landau3[i]->Draw("");

    IlluminaCanv->cd();
    pad2[i]->SetFillColor(10);
    pad2[i]->SetFrameFillColor(10);
    pad2[i]->Draw();
    pad2[i]->cd();
    pad2[i]->SetLogy();
    illuminazione [i]->SetStats(kFALSE);
    illuminazione[i]->Draw("");

    CompTimeCanv->cd();
    pad3[i]->SetFillColor(10);
    pad3[i]->SetFrameFillColor(10);
    pad3[i]->Draw();
    pad3[i]->cd();
    comprtime= new TGraph(nevent,x,yyc);
    oss1<<"DSP  "<<i+1;
    comprtime->SetTitle(oss1.str().c_str());
    comprtime->GetXaxis()->SetTitle("OBT (ms)");
    comprtime->GetXaxis()->CenterTitle();
    comprtime->GetYaxis()->SetTitle("compressiontime (ms)");
    comprtime->GetYaxis()->CenterTitle();
    comprtime->GetYaxis()->SetRangeUser(0,2);
    comprtime->Draw("ap"); 
    oss1.str(""); 
  };
   
  EventNumCanv->cd();
  TPad *pt0=new TPad("pt0"," ",0.704,0.004,0.996,0.98);
  pt1 = new TPaveText(0.004,0.004,0.986,0.996);
  pt = new TPad("pt"," ",0.004,0.004,0.7,0.98,18,0,0);
  pt1->SetFillColor(10);
  pt1->ReadFile(out + "warning.txt");
  pt1->SetTextAlign(22);
  pt1->SetTextSize(0.035);
  pt0->SetFillColor(10);
  pt0->SetFrameFillColor(10);
  pt0->Draw();
  pt0->cd();
  pt1->Draw();
  EventNumCanv->cd();
  pt->SetFillColor(10);
  pt->SetFrameFillColor(10);
  pt->Draw();
  pt->cd();
  eventnumb=new TGraph(nevent,x,eventint);
  eventnumb->SetTitle("");
  eventnumb->GetXaxis()->SetLabelSize(0.04);
  eventnumb->GetXaxis()->SetTitleSize(0.04);
  eventnumb->GetXaxis()->SetTitle("OBT (ms)");
  eventnumb->GetXaxis()->CenterTitle();
  eventnumb->GetXaxis()->SetTickLength(0.01);
  eventnumb->GetXaxis()->SetTitleOffset(1.2);
  eventnumb->GetYaxis()->SetLabelSize(0.04);
  eventnumb->GetYaxis()->SetTitleSize(0.04);
  eventnumb->GetYaxis()->SetTitle("DSP event-number");
  eventnumb->GetYaxis()->CenterTitle();
  eventnumb->GetYaxis()->SetTitleOffset(1.5);
  eventnumb->SetMarkerStyle(21);
  eventnumb->SetMarkerColor(kBlue);
  eventnumb->SetMarkerSize(0.3);
  eventnumb->Draw("ap");

  for(Int_t i=0; i<5; i++){
    for (Int_t ev=0; ev<tsize; ev++){
      tempt_tmp[ev]=(Int_t)(((3.3*tempt[ev][i]/4096)-0.586)/0.0231);
    }
    TempCanv->cd();
    tit<<"T"<<i+5<<" (magnetic module "<<i+1<<")";
    pad6[i]->SetFillColor(10);
    pad6[i]->Draw();
    pad6[i]->cd();
    temp[i]= new TGraph(tsize,xt,tempt_tmp);
    temp[i]->SetTitle(tit.str().c_str());
    temp[i]->GetXaxis()->SetLabelSize(0.04);
    temp[i]->GetXaxis()->SetTitleSize(0.04);
    temp[i]->GetXaxis()->SetTitle("OBT (ms)");
    temp[i]->GetXaxis()->CenterTitle();
    temp[i]->GetXaxis()->SetTitleOffset(0.85);
    temp[i]->GetYaxis()->SetTitleOffset(1.2);
    temp[i]->GetYaxis()->SetLabelOffset(0.001);
    temp[i]->GetYaxis()->SetLabelSize(0.04);
    temp[i]->GetYaxis()->SetTitleSize(0.04);
    temp[i]->GetYaxis()->SetTitle("Temperatures ( ^{o}C)");
    temp[i]->GetYaxis()->CenterTitle();
    temp[i]->SetMarkerStyle(21);
    temp[i]->SetMarkerSize(0.2);
    temp[i]->Draw("ap");
    tit.str("");
  }
  CompTimeCanv->Update();
  LandauCanv1->Update();
  LandauCanv2->Update();
  LandauCanv3->Update();
  IlluminaCanv->Update();
  TempCanv->Update();
  
  //*************************************************************************
  // Save output Files
  //*************************************************************************
  stringstream out1,out2,out3,command;
  
  if(!strcmp(outfile.Data(),"ps")||!strcmp(outfile.Data(),"pdf")){
    out1.str("");
    out2.str("");
    out3.str("");
    
    out1<<ffile<<"_TrkQLook_EXPERT.ps(";
    out2<<ffile<<"_TrkQLook_EXPERT.ps";
    out3<<ffile<<"_TrkQLook_EXPERT.ps)";
    
    CompTimeCanv->Print(out+out1.str().c_str(),"Portrait");
    EventNumCanv->Print(out+out2.str().c_str(),"Portrait");
    LandauCanv1->Print(out+out2.str().c_str(),"Portrait");
    LandauCanv2->Print(out+out2.str().c_str(),"Portrait");
    LandauCanv3->Print(out+out2.str().c_str(),"Portrait");
    IlluminaCanv->Print(out+out2.str().c_str(),"Portrait");
    TempCanv->Print(out+out3.str().c_str(),"Portrait");
    
    if(!strcmp(outfile.Data(),"pdf")){
      stringstream com;
      com<<"ps2pdf13 "<<out<<ffile<<"_TrkQLook_EXPERT.ps "<<out<<ffile<<"_TrkQLook_EXPERT.pdf";
      system(com.str().c_str());
      com.str("");
      com<<"rm -f "<<out<<ffile<<"_TrkQLook_EXPERT.ps ";
      system(com.str().c_str());
      com.str("");
    }
  }
  else{
    out1.str("");
    out1<<ffile<<"_TrkQLook_EXPERT-CompTime."<<outfile.Data();
    CompTimeCanv->Print(out+out1.str().c_str());
    out1.str("");
    out1<<ffile<<"_TrkQLook_EXPERT-Temp."<<outfile.Data();
    TempCanv->Print(out+out1.str().c_str());
    out1.str("");
    out1<<ffile<<"_TrkQLook_EXPERT-EvNum."<<outfile.Data();
    EventNumCanv->Print(out+out1.str().c_str());
    out1.str("");
    out1<<ffile<<"_TrkQLook_EXPERT-LandauL1."<<outfile.Data();
    LandauCanv1->Print(out+out1.str().c_str());
    out1.str("");
    out1<<ffile<<"_TrkQLook_EXPERT-LandauL2."<<outfile.Data();
    LandauCanv2->Print(out+out1.str().c_str());
    out1.str("");
    out1<<ffile<<"_TrkQLook_EXPERT-LandauL3."<<outfile.Data();
    LandauCanv3->Print(out+out1.str().c_str());
    out1.str("");
    out1<<ffile<<"_TrkQLook_EXPERT-Illum."<<outfile.Data();
    IlluminaCanv->Print(out+out1.str().c_str());
  }
 
  for (Int_t i=0; i<12 ; i++){
    delete landau1[i];
    delete landau2[i];
    delete landau3[i];
    delete illuminazione[i];
    if(i<5) temp[i]->Delete();
  }
  eventnumb->Delete();
  comprtime->Delete();
  delete CompTimeCanv;
  delete EventNumCanv;
  delete LandauCanv1;
  delete LandauCanv2;
  delete LandauCanv3;
  delete IlluminaCanv;
  delete TempCanv;
  
  command.str("");
  command<<"rm -f "<<out<<"warning.txt";
  system(command.str().c_str());

  gROOT->Reset();
//   printf("... end of packets. \n");
  return;
}
