/****************************************************************************
/*  F i l e   D a t a                                                        
/*                                                                           
/*  Module       : FileManager                                                   
/*  C.I. No.     :                                                           
/*  $Revision: 1.4 $
/*  $Date: 2005/02/21 08:58:28 $
/*  Belonging to :                                                           
/*               :                                                           
/*  $RCSfile: FD_MMSUDriver_op.c,v $
/*  Program Type :                                                           
/*  Sub-modules  :                                                           
/*                                                                           
/****************************************************************************
/*  S W   D e v e l o p m e n t   E n v i r o n m e n t                      
/*                                                                           
/*  Host system  :                                                           
/*  SW Compiler  :                                                           
/*  $Author: sebastiani $
/*               :                                                           
/****************************************************************************
/*  U p d a t i n g                                                          
/*                                                                           
/*  $Log: FD_MMSUDriver_op.c,v $
/*  Revision 1.4  2005/02/21 08:58:28  sebastiani
/*  all log comments completed
/*
/*  Revision 1.3  2003/10/21 16:09:12  alfarano
/*  LU_LOG_INFN replacement for all remaining original log functions
/*
/*  Revision 1.2  2003/08/07 08:44:14  wizard
/*  SIMULATOR bugs ifdef/ifndef corrected
/*  laben bug "GetFileInfo" for FS_ERC32 id accepted in order to fix SM_pi_GetPacket
/*
/*  Revision 1.1.1.1  2003/08/04 09:40:21  sebastiani
/*  Imported sources laben rel. 19.06.2003 integrated with pam2
/*
/*  Revision 1.5  2002/11/08 11:10:36  zulia
/*  fixed down link in case of the last sector stored
/*
/*  Revision 1.4  2002/10/17 10:03:54  zulia
/*  removed FD_PIFReset
/*
/*  Revision 1.3  2002/05/09 08:16:34  zulia
/*  *  acceptance release
/*
/*                                                                           
/*****************************************************************************/

 
/*============================= Include File ================================*/
  
#include <src/FileManager/MMSUDriver/FD_MMSUDriver_op.h>
#include <src/FileManager/MMSUDriver/FD_MMSUDriver_int.h>

#include <src/INFN/LU_SourceFileID_INFN.h>
#define __FILEID__ _FD_MMSUDriver_op__c
#include <src/INFN/PRH_ParamHandler_INFN_auto.h>
#include <src/INFN/PRH_ParamHandler_INFN.h>
#include <src/INFN/LU_LogUtility_INFN.h>

LU_DECL_MASK();


/*============================== local define ===============================*/

/*======================== Object Internal variables ========================*/


/*****************************************************************************/
/* @Constant: FD_DrammaRegMask                                               */
/* @Purpose :                                                                */
/*  Array of FD_DRAMMA_REG type.                                             */
/*  DRAMMA registers mask.                                                   */
/* @@                                                                        */
/*****************************************************************************/
const FD_DRAMMA_REG FD_DrammaRegMask= {{0x00,
                                     0x80000000,
                                     0xc0000000,
                                     0xc8000000,
                                     0xd0000000,
                                     0xd8000000,
                                     0xe0000000,
                                     0xf0000000,
                                     0xf8000000,
                                     0xfc000000,
                                     0xfe000000},
                                    {0x00,
                                     0x01,
                                     0x02,
                                     0x03,
                                     0x04,
                                     0x05, 
                                     0x06, 
                                     0x07}
                                    };



/*****************************************************************************/

/*===== M M S U D r i v e r   O P E R A T I O N A L   F U N C T I O N S =====*/

/*****************************************************************************/
/* @Function: FD_opInitMMSUDriver                                            */
/* @Purpose :                                                                */
/*  The function performs the MMSUDriver object initialization.              */
/*  The BOX ID register is read to get the MMSU boxes composition. If the    */
/*  box ID returned is 0 the addresses of the Box X are Local and the        */
/*  addresses of the Box Y are Remote. Otherwise if the value returned is 1  */
/*  the  addresses of the  Box Y are Local and the addresses of the Box X    */
/*  are Remote.                                                              */
/*  The Wideband registers are  also initialized with the default value.     */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/

status_code FD_opInitMMSUDriver (void)
{
status_code   status;


    status =SUCCESSFUL;
    return (status);
}



/*****************************************************************************/
/* @Function: FD_opDrammaWrWPBUSReg                                          */
/* @Purpose :                                                                */
/*  The function programs a writing into the Dramma WPBUS register.          */
/*  The value is written into the Dramma register specified through the      */
/*  Box, Board address, and register parameters.                             */
/*  An invalid parameter detection, causes  an internal error notified       */
/*  with a log into the History area object.                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  BoardId              IN      Board address   (0 to FD_MAX_BOARD_ADDR)    */
/*  Reg                  IN      Register indentifier                        */
/*                                               (FD_DRAMMA_REG_WPBUS type)  */
/*  Value                IN      Value to write into the Dramma register     */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/

status_code FD_opDrammaWrWPBUSReg (unsigned int BoardId,
                                   unsigned int Reg, 
                                   unsigned int Value)
{
status_code     status;

    /* Check box, board ID, and register parameter       */
    if (BoardId <= FD_MAX_BOARD_ADDR && Reg < FD_MAX_DRAMMA_WPBUS_REG)
    {      
        /* Write DRAMMA WPBUS register                   */
        FD_ifDrammaWrWPBUSReg ((void*)&FD_DrammaRegMask,BoardId,Reg,Value);
        status =SUCCESSFUL;
    }
    else
    {

      // LOG_INFN HA_piLogHistoryEntry10(HA_E10_SW_FD,HA_E10_MMSU_REG_ADDRESS_ERR,HA_E10_REP2_NULL);
      /*@LOG Invalid MMSU register address */
      LU_INFN_LOG(LU_CRITICAL,LU_MASK(__FILEID__),__FILEID__,__LINE__,HA_E10_REP2_NULL);
      status =HA_E2_INTERNAL_ERR;
    }

    return (status);

}



/*****************************************************************************/
/* @Function: FD_opDrammaRdRPBUSReg                                          */
/* @Purpose :                                                                */
/*  The function programs a reading from the Dramma WPBUS register.          */
/*  The value is read from the Dramma register specified  through the Box,   */
/*  Board address, and register parameters.                                  */
/*  An invalid parameter detection, causes  an internal error notified       */
/*  with a log into the History area object.                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  BoardId              IN      Board address   (0 to FD_MAX_BOARD_ADDR)    */
/*  Reg                  IN      Register indentifier                        */
/*                                               (FD_DRAMMA_REG_RPBUS type)  */

/*  Value                OUT     Value read from the Dramma register         */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/
            
status_code FD_opDrammaRdRPBUSReg (unsigned int  BoardId,
                                   unsigned int  Reg, 
                                   unsigned int* Value)
{
status_code status;

    /* Check box, board ID, and register parameter       */
    if (BoardId<=FD_MAX_BOARD_ADDR && Reg<FD_MAX_DRAMMA_RPBUS_REG)
    {      
        /* Read DRAMMA RPBUS register                    */
        FD_ifDrammaRdRPBUSReg ((void*)&FD_DrammaRegMask,BoardId,Reg,Value);
        status =SUCCESSFUL;
    }
    else
    {

        // LOG_INFN HA_piLogHistoryEntry10(HA_E10_SW_FD,HA_E10_MMSU_REG_ADDRESS_ERR,HA_E10_REP2_NULL);
        /*@LOG Invalid MMSU register address                 */
       	LU_INFN_LOG(LU_CRITICAL,LU_MASK(__FILEID__),__FILEID__,__LINE__,HA_E10_REP2_NULL);
        status =HA_E2_INTERNAL_ERR;
    }

    return (status);

}

/*****************************************************************************/
/* @Function: FD_opStartTransferToMemory                                     */
/* @Purpose :                                                                */
/*  Provided Interface to start the WPBUS writing into the memory module.    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  Module               IN      Module ID to write to                       */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/

status_code FD_opStartTransferToMemory(FD_MODID Module)
{
status_code status = SUCCESSFUL;
BYTE  *reg;

  reg = (BYTE *) FD_WPBUS_DBID;
  /* Start the data transfer to the selected module */
  switch(Module) {
    case FD_PRIMARY: {
#ifndef SIMULATOR
      *reg = (BYTE)(0x00);
#endif
    } break;
    case FD_BACKUP: {
#ifndef SIMULATOR
      *reg = (BYTE)(0x01);
#endif
    } break;
    default:{
      status = UNSATISFIED;
    }
  }
  return (status);
}

/*****************************************************************************/
/* @Function: FD_opStartTransferFromMemory                                   */
/* @Purpose :                                                                */
/*  Provided Interface to start the RPBUS reading from the memory module.    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  Module               IN      Module ID to write to                       */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/

status_code FD_opStartTransferFromMemory(FD_MODID Module, BOOL LastSector)
{
#ifdef SIMULATOR
  return SUCCESSFUL;
#else 
  status_code status = SUCCESSFUL;
  BYTE  *reg;
  BYTE Lsf = 0x00;

  reg = (BYTE *) FD_RPBUS_DBID;
  if(LastSector)
  {
    Lsf = FD_LAST_SECTOR_FLAG;
  }
  switch(Module) {
    case FD_PRIMARY: {
      *reg = (BYTE)(0x00 | Lsf);
    } break;
    case FD_BACKUP: {
      *reg = (BYTE)(0x01 | Lsf);
    } break;
    default:{
      status = UNSATISFIED;
    }
  }
  return (status);
#endif
}

/*****************************************************************************/
/* @Function: FD_opAbortTransfer                                             */
/* @Purpose :                                                                */
/*  Provided Interface to start the WPBUS writing into the memory module.    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  Module               IN      Module ID to write to                       */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/

status_code FD_opAbortTransfer(FD_ABT_TYPE Aborttype)
{
#ifdef SIMULATOR
  return SUCCESSFUL;
#else
  status_code status = SUCCESSFUL;
  BYTE  *reg;

  reg = (BYTE *) FD_RWPB_ABORT;
  /* Abort the selected operation */
  switch(Aborttype) {
    case FD_ABT_WRITE: {
      *reg = (BYTE)(0x01);
    } break;
    case FD_ABT_READ: {
      *reg = (BYTE)(0x02);
    } break;
  }
  return (status);
#endif
}

/*****************************************************************************/
/* @Function: FD_opSetWPBUSWaitState                                         */
/* @Purpose :                                                                */
/*  Provided Interface to set the WPBUS wait state counter                   */
/*  packets.                                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  Counter              IN      Number of WPBUS wait states                 */
/* @@                                                                        */
/*****************************************************************************/

void FD_opSetWPBUSWaitState(UINT32 Counter)
{
#ifdef SIMULATOR
  return;
#else
  BYTE  *reg;

  reg = (BYTE *) FD_WPB_WSC;
  *reg = (BYTE)(Counter & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: FD_opSetRPBUSWaitState                                         */
/* @Purpose :                                                                */
/*  Provided Interface to set the RPBUS wait state counter                   */
/*  packets.                                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  Counter              IN      Number of RPBUS wait states                 */
/* @@                                                                        */
/*****************************************************************************/

void FD_opSetRPBUSWaitState(UINT32 Counter)
{
#ifdef SIMULATOR
  return;
#else
  BYTE  *reg;

  reg = (BYTE *) FD_RPB_WSC;
  *reg = (BYTE)(Counter & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: FD_opSetBlockLength                                            */
/* @Purpose :                                                                */
/*  Provided Interface to set the last block length for the RPBUS            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  Length               IN      block length                                */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/

status_code FD_opSetBlockLength(UINT32 Length)
{
#ifdef SIMULATOR
  return SUCCESSFUL;
#else
  status_code status = SUCCESSFUL;
  BYTE  *reg;
  
  /* Program the Block length for the RPBUS*/
  reg = (BYTE *) FD_RPB_RBL1;
  *reg = (BYTE)(Length & 0x000000FF);
  reg = (BYTE *) FD_RPB_RBL2;
  *reg = (BYTE)((Length>>8) & 0x000000FF);
  reg = (BYTE *) FD_RPB_RBL3;
  *reg = (BYTE)((Length>>16) & 0x000000FF);

  return (status);
#endif
}




