/****************************************************************************
/*  F i l e   D a t a                                                        
/*                                                                           
/*  Module       : FileManager                                                   
/*  C.I. No.     :                                                           
/*  $Revision: 1.17 $
/*  $Date: 2005/02/21 08:58:28 $
/*  Belonging to :                                                           
/*               :                                                           
/*  $RCSfile: FS_FileSystem_op.c,v $
/*  Program Type :                                                           
/*  Sub-modules  :                                                           
/*                                                                           
/****************************************************************************
/*  S W   D e v e l o p m e n t   E n v i r o n m e n t                      
/*                                                                           
/*  Host system  :                                                           
/*  SW Compiler  :                                                           
/*  $Author: sebastiani $
/*               :                                                           
/****************************************************************************
/*  U p d a t i n g                                                          
/*                                                                           
/*  $Log: FS_FileSystem_op.c,v $
/*  Revision 1.17  2005/02/21 08:58:28  sebastiani
/*  all log comments completed
/*
/*  Revision 1.16  2004/09/17 15:01:00  faber
/*  LU_INFN_LOG flags fixing
/*
/*  Revision 1.15  2004/07/28 17:18:45  sebastiani
/*  better loggin status
/*
/*  Revision 1.14  2004/07/08 16:23:37  sebastiani
/*  *** empty log message ***
/*
/*  Revision 1.13  2004/06/15 08:12:25  sebastiani
/*  no warning MaxSize any more
/*
/*  Revision 1.12  2004/01/29 18:07:06  sebastiani
/*  important bug fix (no mass memory store for big packet):
/*  // fix: commented out due a BIG for long packet
/*  >      // FS_WrittenFile = FileId;
/*  >      // FS_CurrentSaveSize = WriteSize;
/*  >      *WriteAddr = FS_Files[FS_ERC32].WriteAddr;
/*
/*  Revision 1.11  2003/11/10 12:11:52  sebastiani
/*  more complete log info
/*
/*  Revision 1.10  2003/11/05 08:16:36  sebastiani
/*  new filesystem %size in 100*100 instead 100 units
/*
/*  Revision 1.9  2003/11/04 11:56:53  sebastiani
/*  new log status sequence
/*
/*  Revision 1.8  2003/10/30 10:09:30  sebastiani
/*  MM size definitions fixed
/*
/*  Revision 1.7  2003/10/29 17:41:48  sebastiani
/*  *** empty log message ***
/*
/*  Revision 1.6  2003/10/29 08:32:26  sebastiani
/*  *** empty log message ***
/*
/*  Revision 1.5  2003/10/17 08:18:48  sebastiani
/*  FileID <-- FILE1 bugfix
/*
/*  Revision 1.4  2003/09/10 13:38:13  laben
/*  New File System Model:
/*  jusy one file used like a circular buffer.
/*
/*  Revision 1.3  2003/08/27 09:49:17  wizard
/*  Fix last sector in donwload.
/*  Set Sector size to 1024*1015
/*  Fix Donwload procedure... ( set size set lastsector flag.....)
/*  bye
/*
/*  Revision 1.2  2003/08/07 08:44:14  wizard
/*  SIMULATOR bugs ifdef/ifndef corrected
/*  laben bug "GetFileInfo" for FS_ERC32 id accepted in order to fix SM_pi_GetPacket
/*
/*  Revision 1.1.1.1  2003/08/04 09:40:21  sebastiani
/*  Imported sources laben rel. 19.06.2003 integrated with pam2
/*
/*  Revision 1.8  2003/06/09 10:22:10  aurora
/*  introduced a new file called FS_ERC32.
/*  It stores the data in ERC32 memory
/*
/*  Revision 1.7  2003/05/09 08:51:10  aurora
/*  correct race on write of regular file and, after, cmd
/*
/*  Revision 1.6  2002/05/09 08:16:34  zulia
/*  *  acceptance release
/*
/*                                                                           
/*****************************************************************************/

/*============================= Include File ================================*/

#include <src/INFN/LU_SourceFileID_INFN.h>
#define __FILEID__ _FS_FileSystem_op__c
#include <src/INFN/PRH_ParamHandler_INFN.h>
#include <src/INFN/LU_LogUtility_INFN.h>
#include <src/INFN/PRH_ParamHandler_INFN_auto.h>
LU_DECL_MASK();

  
#include <src/FileManager/FileSystem/FS_FileSystem_op.h>
#include <src/FileManager/MMSUManager/FT_MMSUManager_p.h>

/*============================== global define ==============================*/

/*======================== Object Internal variables ========================*/

/*****************************************************************************/
/* @Variable: FS_Files                                                       */
/* @Purpose :                                                                */
/*  Mass memory files                                                        */
/* @@                                                                        */
/*****************************************************************************/
static FS_FILE_DESCR FS_Files[FS_MAXID+1];

/*****************************************************************************/
/* @Variable: FS_CurrentSaveSize                                             */
/* @Purpose :                                                                */
/*  Size of the last store operation                                         */
/* @@                                                                        */
/*****************************************************************************/
static UINT32 FS_CurrentSaveSize;

/*****************************************************************************/
/* @Variable: FS_WrittenFile                                                 */
/* @Purpose :                                                                */
/*  Last Written file                                                        */
/* @@                                                                        */
/*****************************************************************************/
static UINT32 FS_WrittenFile;

/*****************************************************************************/
/* @Variable: FS_SecList                                                     */
/* @Purpose :                                                                */
/*  Sector chain used for download operation                                 */
/* @@                                                                        */
/*****************************************************************************/
static FS_SECT_DESCR FS_SecList[FT_MAXSECTORS];

/*****************************************************************************/
/* @Variable: FS_ReadSectorCount                                             */
/* @Purpose :                                                                */
/*  Read sector counter                                                      */
/* @@                                                                        */
/*****************************************************************************/
static UINT32 FS_ReadSectorCount;

/*****************************************************************************/
/* @Variable: FS_CurrentImageSector                                          */
/* @Purpose :                                                                */
/*  Index of the current first sector of the chain                           */
/* @@                                                                        */
/*****************************************************************************/
static UINT32 FS_CurrentImageSector;

/*****************************************************************************/
/* @Variable: FS_LastImageSector                                             */
/* @Purpose :                                                                */
/*  Index of the last sector in the chain                                    */
/* @@                                                                        */
/*****************************************************************************/
static UINT32 FS_LastImageSector;

/*****************************************************************************/
/* @Variable: FS_PktBuf                                                      */
/* @Purpose :                                                                */
/*  Buffer used to tranfer packets from SRAM to microprocessor RAM           */
/*  and viceversa                                                            */
/* @@                                                                        */
/*****************************************************************************/
static BYTE FS_PktBuf[SM_DAQPKTSIZE];


/*****************************************************************************/
/* @Variable: FS_LastSectorDwn                                               */
/* @Purpose :                                                                */
/*  last sector downlinked pointer                                           */
/*                                                                           */
/* @@                                                                        */
/*****************************************************************************/
static UINT32 FS_LastSectorDwn;


/*****************************************************************************/

/*=== F S _ F i l e S y s t e m   O P E R A T I O N A L   F U N C T I O N S ==*/

/*****************************************************************************/
/* @Function: FS_opInitFileSystem                                            */
/* @Purpose :                                                                */
/*  The function initializes the File System structures.                     */
/*  The -Fat- parameter specifies which FAT initialize Local, Remote, or     */
/*  both (FAT_LOCAL,FAT_REMOTE,FAT_ALL). In FAT_ALL modality is also         */
/*  initialized the File Description Table                                   */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*                      IN                                                   */
/* status_code          OUT     Return code                                  */
/* @@                                                                        */
/*****************************************************************************/

FS_EXITCODE FS_opInitFileSystem(void)
{
FS_EXITCODE status = FS_SUCCESS;

  // Hardcoded file setup
  FS_Files[FS_FILE1].Size = 0;
  FS_Files[FS_FILE1].WriteAddr = 0;         // 64byte gran  addres
  FS_Files[FS_FILE1].MaxSize = (FT_MAXSECTORS*FT_SECTSIZE)/FT_SAMPLESIZE;
  FS_Files[FS_FILE1].MaxAddr = (FT_MAXSECTORS*FT_SECTSIZE)/FT_MEMGRANUL;
 
  // dummy values; the FS_HK is FS_FILE1
  FS_Files[FS_HK].Size = 0;
  FS_Files[FS_HK].WriteAddr = 0;
  FS_Files[FS_HK].MaxSize = 0;
  FS_Files[FS_HK].MaxAddr = 0;
  
  // Special file used for the ERC32
  FS_Files[FS_ERC32].Size = 0;
  FS_Files[FS_ERC32].WriteAddr = (unsigned int) FS_PktBuf;
  FS_Files[FS_ERC32].MaxSize = SM_DAQPKTSIZE;
  // unused
  FS_Files[FS_ERC32].MaxAddr = FS_Files[FS_ERC32].WriteAddr + SM_DAQPKTSIZE;  
  // Special file used for the CMD I/F
  FS_Files[FS_CMD].Size = 0;
  FS_Files[FS_CMD].WriteAddr = 0;
  FS_Files[FS_CMD].MaxSize = 0xFFFFFFFF;
  // unused
  FS_Files[FS_CMD].MaxAddr = FS_Files[FS_CMD].WriteAddr + FS_Files[FS_CMD].MaxSize; 
  // Dummy files
  FS_Files[FS_REGFILES].Size = 0;
  FS_Files[FS_REGFILES].WriteAddr = 0;
  FS_Files[FS_REGFILES].MaxSize = 0x00000000;
  // unused
  FS_Files[FS_REGFILES].MaxAddr = FS_Files[FS_REGFILES].WriteAddr + FS_Files[FS_REGFILES].MaxSize; 

  FS_Files[FS_MAXID].Size = 0;
  FS_Files[FS_MAXID].WriteAddr = 0;
  FS_Files[FS_MAXID].MaxSize = 0x00000000;
  // unused
  FS_Files[FS_MAXID].MaxAddr = FS_Files[FS_MAXID].WriteAddr + FS_Files[FS_MAXID].MaxSize; 

  FS_LastSectorDwn = 0;
  
  return (status);
}

/*****************************************************************************/
/* @Function: FS_opGetFileInfo                                               */
/* @Purpose :                                                                */
/*  Provided Interface to get the status of the selected file                */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  FileId              IN      File ID                                      */
/*  *FileInfo           OUT     File status                                  */
/*  FS_EXITCODE         OUT     Return code                                  */
/* @@                                                                        */
/*****************************************************************************/

FS_EXITCODE FS_opGetFileInfo(FS_FILEID FileId, FS_FILE_DESCR* FileInfo)
{
  FS_EXITCODE status;
 unsigned int intLevel;

 /* File info are available only for regular files */
 /* INFN BugFix: FS_ERC32 is needed by PM_pi_GetPacket(...) */
 if(FileId >= FS_REGFILES && FileId != FS_ERC32) 
   status = FS_WRONGFILEID;
 else {
   OS_piInterDisable(&intLevel);
   /* Retrive selected file info */
   FileInfo->Size = FS_Files[FileId].Size;
   FileInfo->WriteAddr = FS_Files[FileId].WriteAddr;
   FileInfo->MaxSize = FS_Files[FileId].MaxSize;
   OS_piInterEnable(intLevel);
   status = FS_SUCCESS;
  }
  return (status);
}

void FS_opLogInfo() {
  unsigned short used     = (FS_Files[FS_FILE1].Size*100) / FS_Files[FS_FILE1].MaxSize;
  unsigned short size_mega = (unsigned short)((FS_Files[FS_FILE1].Size*FT_SAMPLESIZE)/(1024*1024));
  /*@LOG Size */
  LU_INFN_LOG(LU_NORMAL_TRACE,LU_MASK(__FILEID__),__FILEID__,__LINE__,FS_Files[FS_FILE1].Size);
  /*@LOG WriteAddress */
  LU_INFN_LOG(LU_NORMAL_TRACE,LU_MASK(__FILEID__),__FILEID__,__LINE__,FS_Files[FS_FILE1].WriteAddr);
  /*@LOG Used Ratio % */
  LU_INFN_LOG(LU_NORMAL_TRACE,LU_MASK(__FILEID__),__FILEID__,__LINE__,used);
  /*@LOG Size in MB */
  LU_INFN_LOG(LU_NORMAL_TRACE,LU_MASK(__FILEID__),__FILEID__,__LINE__,size_mega);
  /*@LOG Last Sector Downlink */
  LU_INFN_LOG(LU_NORMAL_TRACE,LU_MASK(__FILEID__),__FILEID__,__LINE__,FS_LastSectorDwn);

}

/*****************************************************************************/
/* @Function: FS_opWriteFile                                                 */
/* @Purpose :                                                                */
/*  Provided Interface to check if a file could be written. The status of the*/
/*  selected file is not updated until the UpdateFile operation is called    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  FileId              IN      File ID                                      */
/*  WriteSize           IN      Size of the data to be saved (in samples)    */
/*  WriteAddr           OUT     Address where to store data in the file      */
/*  FS_EXITCODE         OUT     Return code                                  */
/* @@                                                                        */
/*****************************************************************************/

FS_EXITCODE FS_opWriteFile(FS_FILEID FileId, UINT32 WriteSize,
                           UINT32 *WriteAddr)
{
FS_EXITCODE status = FS_SUCCESS;
UINT32 algn;
UINT32 StartNextDownlinkAddr;
UINT32 granInc;
UINT32 granWordInc; 
 
  /*
   * if(FileId < FS_REGFILES) 
   * dobbiamo controllare che: 
   * ci si spazio nel file (che WriteAddress+WriteSize non si maggiore di StartDonwlinkAddress)
   * che WriteAddress + WriteSize non oltrepassi la fine del file 
   * non bisogna fare reset del file
   *  
   */

 if (FileId == FS_CMD)
   return(status);
 
 if ((FileId == FS_REGFILES) || (FileId == FS_MAXID))
   return(FS_WRONGFILEID);
 
 if(FileId < FS_REGFILES) // PUO' ESSERE SOLO FILE1
   {
     FileId = FS_FILE1;

     algn = WriteSize % FT_SAMPLESIZE;
     if(algn)
       {
	 WriteSize += (FT_SAMPLESIZE - algn);
       }

     // graninc: write size in blocchi da 64byte!     
     granInc =  WriteSize / FT_MEMGRANUL;
     if( (WriteSize % FT_MEMGRANUL) ) granInc++;
     // granwordinc: size realmente occupata nel file in bloacchi da 4 byte!
     granWordInc = ( granInc *  FT_MEMGRANUL ) / FT_SAMPLESIZE; 

     // WriteSize: size in blocchi da 4 byte realmente da downloadare!
     WriteSize /= FT_SAMPLESIZE; 

     // controllo se vado oltre la fine del file!  
     if((FS_Files[FileId].WriteAddr + granInc) >  FS_Files[FileId].MaxAddr) 
       {
	 // spazio non sufficiente ( alla fine del file) mi metto all'inizio del file e aggiorno la size 
	 UINT32 LostByte = ( ( FS_Files[FileId].MaxAddr - FS_Files[FileId].WriteAddr) * FT_MEMGRANUL );
	 FS_Files[FileId].Size += ( LostByte / FT_SAMPLESIZE);
	 FS_Files[FileId].WriteAddr = 0;
       }
     
     if(FS_Files[FileId].Size + granWordInc > FS_Files[FileId].MaxSize )
       {
	 /* There is not enough space on the device                  */
	 FS_WrittenFile = FS_MAXID;
	 FS_CurrentSaveSize = 0;
	 return(FS_FILEISFULL);
       }

     /* Puo' bastare il controllo precedente

     // se sono arrivato dove ho gia' scritto il file e' pieno!!!!
      StartNextDownlinkAddr = (FS_Files[FileId].StartDwnlnkSec*FT_SECTSIZE/FT_MEMGRANUL );
      if((FS_Files[FileId].WriteAddr+granInc) > StartNextDownlinkAddr )
	{
	  FS_WrittenFile = FS_MAXID;
	  FS_CurrentSaveSize = 0;
	  return(FS_FILEISFULL);
	}
     */
       
     FS_WrittenFile = FileId;
     FS_CurrentSaveSize = WriteSize;
     *WriteAddr = FS_Files[FS_WrittenFile].WriteAddr;
   }
 
 if ( FileId == FS_ERC32)
   { 
     // fix: commented out due a BIG for long packet
     // FS_WrittenFile = FileId;
     // FS_CurrentSaveSize = WriteSize;
     *WriteAddr = FS_Files[FS_ERC32].WriteAddr;
   }
 
 return (status);
} 

/*****************************************************************************/
/* @Function: FS_opUpdateFile                                                */
/* @Purpose :                                                                */
/*  Provided Interface to update file size after a store operation           */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  FS_EXITCODE         OUT     Return code                                  */
/* @@                                                                        */
/*****************************************************************************/

FS_EXITCODE FS_opUpdateFile(void)
{
FS_EXITCODE status = FS_SUCCESS;
UINT32 granInc;

  /* Update the size/appendAddress info for regular files */
  if(FS_WrittenFile < FS_REGFILES) {
    /* Evaluate the minimum necessary FT_MEMGRANUL sized chunks of memory */
    /* to hold the new data stored in the file                            */
    granInc = (FS_CurrentSaveSize * FT_SAMPLESIZE) / FT_MEMGRANUL;
    if(((FS_CurrentSaveSize * FT_SAMPLESIZE) % FT_MEMGRANUL)) granInc++;
    /* Evaluate the new filesize in samples */

    FS_Files[FS_FILE1].Size += (granInc * FT_MEMGRANUL) / FT_SAMPLESIZE;
    /* Increment the file append address */
    FS_Files[FS_FILE1].WriteAddr += granInc;
  }
  return (status);
}

/*****************************************************************************/
/* @Function: FS_opMakeFileImage                                             */
/* @Purpose :                                                                */
/*  Provided Interface to build the sector chain for the download operation  */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  StartSector         IN      First sector of the partition to add to the  */
/*                              sector chain                                 */
/*  Size                IN      Size of the partition (in sample)            */
/*  New                 IN      This flag is used to build a new chain or to */
/*                              append a partition to an existing chain      */
/*  nsect               OUT     Number of sectors added to the chain         */
/*  FS_EXITCODE         OUT     Return code                                  */
/* @@                                                                        */
/*****************************************************************************/

FS_EXITCODE FS_opMakeFileImage(UINT32 StartSector, UINT32 Size, UINT32* nsect, BOOL New)
{
  FS_EXITCODE status = FS_SUCCESS;
  UINT32 sectors;
  UINT32 lastSectSize;
  UINT32 halfSector = 0;
  UINT32 sectorAddress;
  UINT32 idx;
  
  // Find the number entierly data filled sectors
  sectors = Size / (FT_SECTSIZE / FT_SAMPLESIZE); 
  // Find the last sector size
  lastSectSize = Size % (FT_SECTSIZE / FT_SAMPLESIZE);
  if(lastSectSize > 0)  halfSector = 1;
  *nsect = sectors + halfSector;

  /* Check if the sectors exceeds the maximum length of the sector chain */
  if(Size == 0 || (StartSector + sectors + halfSector) > FT_MAXSECTORS ) {
    status = FS_WRONGPARAMS;
  }
  else {
    /* Create a new sector chain */
    if(New) {
      FS_CurrentImageSector = StartSector;
      FS_ReadSectorCount = 0;
    }
    else {
      /* Append another sector chain to an exsisting one */
      FS_SecList[FS_LastImageSector].LastSectLength = FS_ONEMB_SECTOR_SIZE;
      FS_SecList[FS_LastImageSector].NextSect = StartSector;
      FS_SecList[FS_LastImageSector].LastSectFlag = FALSE;
    }
    /* Evaluate the address of the fist sector of the chain */
    
    sectorAddress = StartSector * (FT_SECTSIZE / FT_MEMGRANUL);
    
    for(idx = 0; idx < sectors; idx++) {
      /* Create a chain record                    */
      /* for all the entierly data filled sectors */
      FS_SecList[StartSector + idx].LastSectLength = FS_ONEMB_SECTOR_SIZE;
      FS_SecList[StartSector + idx].NextSect = StartSector + idx + 1;
      FS_SecList[StartSector + idx].LastSectFlag = FALSE;
      FS_SecList[StartSector + idx].SectAddr = sectorAddress;

      sectorAddress += (FT_SECTSIZE / FT_MEMGRANUL);
    }
    if(lastSectSize > 0) {
      /* Create a chain record for the last half filled sector */
      /* and close the chain tail by wrapping it ovet itself   */
      FS_SecList[StartSector + sectors].LastSectLength = lastSectSize;
      FS_SecList[StartSector + sectors].NextSect = StartSector + sectors;
      FS_SecList[StartSector + sectors].LastSectFlag = TRUE;
      FS_SecList[StartSector + sectors].SectAddr = sectorAddress;
      FS_LastImageSector = StartSector + sectors;
    }
    else {
      /* Close the chain tail by wrapping it ovet itself   */
      FS_SecList[StartSector + sectors - 1].NextSect = StartSector + sectors - 1;
      FS_SecList[StartSector + sectors - 1].LastSectFlag = TRUE;
      FS_LastImageSector = StartSector + sectors - 1;
    }
  }
  return (status);
}


/*****************************************************************************/
/* @Function: FS_opGetNextSector                                             */
/* @Purpose :                                                                */
/*  Provided Interface to retrive the next sector information to be used     */
/*  in a downlink sequence                                                   */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  *SectorInfo         OUT     Next sector information                      */
/*  FS_EXITCODE         OUT     Return code                                  */
/* @@                                                                        */
/*****************************************************************************/

FS_EXITCODE FS_opGetCurrentSector(FS_SECT_DESCR *SectorInfo)
{
  /* Retrive the selected sector info */ 
  
  //  SectorInfo->SectAddr =  FS_ONEMB_SECTOR_SIZE*FS_LastSectorDwn;
  // The Sector Address is in 64byte unit
  SectorInfo->SectAddr =  FS_LastSectorDwn * (FT_SECTSIZE / FT_MEMGRANUL);
  /* If a minimum number of sectors have not been read yet, then      */
  /* return the last sector info but with the last sector flag=FALSE  */
  SectorInfo->LastSectLength = FS_ONEMB_SECTOR_SIZE;
  SectorInfo->LastSectFlag = FALSE;

  return(FS_SUCCESS);
}

FS_EXITCODE FS_opGetNextSector(FS_SECT_DESCR *SectorInfo)
{
  FS_EXITCODE status;
  
  /* Increment the sector chain index and retrive the next sector information */
  //  FS_CurrentImageSector = FS_SecList[FS_CurrentImageSector].NextSect;
  
  if(FS_Files[FS_FILE1].Size>=FS_ONEMB_SECTOR_SIZE)
    FS_Files[FS_FILE1].Size -= FS_ONEMB_SECTOR_SIZE;
  else{
    FS_Files[FS_FILE1].Size = 0 ;
    // Ho donwloadato tutta la memoria scritta!!!
  }
  FS_LastSectorDwn++;
  if(FS_LastSectorDwn >= FT_MAXSECTORS){
    FS_LastSectorDwn=0;
  }
  
  
  status = FS_piGetCurrentSector(SectorInfo);
     
  return (status);
}

FS_EXITCODE FS_opUpdateSectorPointer()
{
  UINT32 FileWriteSector;
  UINT32 LastSectorDwnl;
  FS_EXITCODE status = FS_SUCCESS;
  
  if(! FS_Files[FS_FILE1].Size){
    // l'ultimo settore dove ho scritto!
    FileWriteSector = 1 + (FS_Files[FS_FILE1].WriteAddr * FT_MEMGRANUL / FT_SECTSIZE);
    FS_LastSectorDwn = FileWriteSector;
    FS_Files[FS_FILE1].WriteAddr = FileWriteSector * (FT_SECTSIZE / FT_MEMGRANUL);
  }
  return (status);
}


/*****************************************************************************/
/* @Function: FS_opResetFile                                                 */
/* @Purpose :                                                                */
/*  Provided Interface to reset the status of a file. By now it means that   */
/*  the size is set to 0                                                     */
/*  in a downlink sequence                                                   */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  *SectorInfo         OUT     Next sector information                      */
/*  FS_EXITCODE         OUT     Return code                                  */
/* @@                                                                        */
/*****************************************************************************/

FS_EXITCODE FS_opResetFile(FS_FILEID FileId)
{
FS_EXITCODE status = FS_SUCCESS;

  /* Reset a file by setting its size equal to zero     */
  /* and its append address to the address of the first */
  /* sector of the file                                 */
  FS_Files[FileId].Size = 0;
  FS_Files[FileId].WriteAddr = 0; 
  
  return (status);
}
