/****************************************************************************
/*  F i l e   D a t a                                                        
/*                                                                           
/*  Module       :  SRAMManager                                                
/*  C.I. No.     :                                                           
/*  $Revision: 1.3 $
/*  $Date: 2004/02/11 16:26:52 $
/*  Belonging to :                                                           
/*               :                                                           
/*  $RCSfile: SD_SRAMDriver_p.c,v $
/*  Program Type :                                                           
/*  Sub-modules  :                                                           
/*                                                                           
/****************************************************************************
/*  S W   D e v e l o p m e n t   E n v i r o n m e n t                      
/*                                                                           
/*  Host system  :                                                           
/*  SW Compiler  :                                                           
/*  $Author: faber $
/*               :                                                           
/****************************************************************************
/*  U p d a t i n g                                                          
/*                                                                           
/*  $Log: SD_SRAMDriver_p.c,v $
/*  Revision 1.3  2004/02/11 16:26:52  faber
/*  CM_TIME2REG introduced
/*
/*  Revision 1.2  2003/10/27 19:01:43  sebastiani
/*  SD_piSetEventTimeout_microseconds introduced, still not used
/*
/*  Revision 1.1.1.1  2003/08/04 09:40:21  sebastiani
/*  Imported sources laben rel. 19.06.2003 integrated with pam2
/*
/*  Revision 1.7  2003/04/29 13:30:01  aurora
/*  introduced the new register TRIG3 for manage page overrun every 64Kbyte
/*
/*  Revision 1.6  2003/03/12 10:56:22  aurora
/*  added the provider for the counter of bytes left on CMD I/F queue
/*
/*  Revision 1.5  2002/10/17 10:07:14  zulia
/*  added SD_PIFReset
/*
/*  Revision 1.4  2002/07/31 14:30:10  zulia
/*  Added driver for indipendent control of CMD I/F from DAQ I/F
/*
/*  Revision 1.3  2002/05/09 08:16:35  zulia
/*  *  acceptance release
/*
/*                                                                           
/*****************************************************************************/


/*============================= Include File ================================*/

#include <src/SRAMManager/SRAMDriver/SD_SRAMDriver_p.h>
#include <src/SRAMManager/SRAMDriver/SD_SRAMDriver_op.h>

/*============================== global define ==============================*/

/*============================== local types ================================*/

/*****************************************************************************/

/*====== S R A M D r i v e r      P R O V I D E D    I N T E R F A C E ======*/

/*****************************************************************************/
/* @Function: SD_piSRAMDriverInit                                            */
/* @Purpose :                                                                */
/*  Provided Interface to init the SRAM driver                               */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSRAMDriverInit(void)
{
SD_EXITCODE status;

  status = SD_opSRAMDriverInit();
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piSetDataWritePage                                          */
/* @Purpose :                                                                */
/*  Provided Interface to select the SRAM page to hold new incoming data.    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* Page                 IN      SRAM page to be selected in input            */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetDAQWritePage(UINT32 Page)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetDAQWritePage(Page);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piIsDAQWriteOk                                              */
/* @Purpose :                                                                */
/*  Provided Interface to retrive the status uf the last DAQ I/F write into  */
/*  SRAM.                                                                    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_TRANSFERSTAT      OUT     status of the last DAQ I/F SRAM write        */
/* @@                                                                        */
/*****************************************************************************/

SD_TRANSFERSTAT SD_piIsDAQWriteOk(void)
{
SD_TRANSFERSTAT status;

  status = SD_opIsDAQWriteOk();
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piGetDAQWriteResult                                         */
/* @Purpose :                                                                */
/*  Provided Interface to retrive the end address of the data written by the */
/*  DAQ I/F in the selected SRAM page.                                       */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_DAQ_RESULT        OUT     structure to hold the end address, page and  */
/*                              status of the last DAQ I/F SRAM write        */
/* @@                                                                        */
/*****************************************************************************/

SD_DAQ_WRESULT SD_piGetDAQWriteResult(void)
{
SD_DAQ_WRESULT Result;

  Result = SD_opGetDAQWriteResult();
  return (Result);
}

/*****************************************************************************/
/* @Function: SD_piStartTransferToCMD                                        */
/* @Purpose :                                                                */
/*  Provided Interface to start the transfer of data stored in the pervoiusly*/
/*  selected SRAM to the CMD I/F.                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piStartTransferToCMD(void)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opStartTransferToCMD();
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piWriteSRAM                                                 */
/* @Purpose :                                                                */
/*  Provided Interface to write a data byte in the pervoiusly selected       */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* data                 IN      8bit data to store into perviously selected  */
/*                              SRAM address                                 */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piWriteSRAM(UINT32 data)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opWriteSRAM(data);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piReadSRAM                                                  */
/* @Purpose :                                                                */
/*  Provided Interface to read a data byte from the pervoiusly selected      */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* BYTE                 OUT     8bit data read from perviously selected      */
/*                              SRAM address                                 */
/* @@                                                                        */
/*****************************************************************************/

UINT32 SD_piReadSRAM(void)
{
UINT32  data;

  data = SD_opReadSRAM();
  return (data);
}

/*****************************************************************************/
/* @Function: SD_piSetEventTimeout                                           */
/* @Purpose :                                                                */
/*  Provided Interface to set the event timeout generation counter. The LSB  */
/*  has a resolution of 65us.                                                */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* tics                 IN      counter value                                */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetEventTimeout_microseconds(UINT32 microseconds)
{
  /* ETO1 LSB register is in 65us - ETO2 LSB register is in 16 ms, so the formula is:
   *
   * microseconds = 16 * 10^3 * ETO2 + 65 * ETO1
   *
   * ETO2 = microseconds / 16*10^3
   * ETO1 = (microseconds - ETO2 * 10*10^3) / 65
   */
  SD_EXITCODE status = SD_SUCCESS;
  BYTE eto2 = microseconds / (16*1000);
  BYTE eto1 = (microseconds - eto2 * 16*1000) / 65;
  SD_opSetEventTimeout( (eto2 << 8) | eto1);
  return (status);
}

SD_EXITCODE SD_piSetEventTimeout(UINT32 tics)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetEventTimeout(tics);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piSetDataTimeout                                            */
/* @Purpose :                                                                */
/*  Provided Interface to set the data timeout generation counter. The LSB   */
/*  has a resolution of 508ns.                                               */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* tics                 IN      counter value                                */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetDataTimeout(UINT32 tics)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetDataTimeout(tics);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piTrigDataEvent                                             */
/* @Purpose :                                                                */
/*  Provided Interface that trigs the DAQ I/F to recieve incoming            */
/*  data packet, enabling the DTO, ETO timeouts and PAGE OVERRUN detection   */
/*  at each 16 Kbytes pages.                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piTrigDataEvent(void)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opTrigDataEvent();
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piTrigData64Event                                           */
/* @Purpose :                                                                */
/*  Provided Interface that trigs the DAQ I/F to recieve incoming            */
/*  data packet, enabling the DTO, ETO timeouts and PAGE OVERRUN detection   */
/*  at each 64 Kbytes pages.                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piTrigData64Event(void)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opTrigData64Event();
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piTrigCalibrationEvent                                      */
/* @Purpose :                                                                */
/*  Provided Interface that trigs the DAQ I/F to recieve incoming data       */
/*  enabling the DTO and ETO timeouts (for calibration mode)                 */
/*  and PAGE OVERRUN detection at 128 Kbyte                                  */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piTrigCalibrationEvent(void)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opTrigCalibrationEvent();
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piSetDataHeaderLength                                       */
/* @Purpose :                                                                */
/*  Provided Interface to set the length of the header of incoming data      */
/*  packets.                                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* length               IN      header length                                */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetDataHeaderLength(UINT32 length)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetDataHeaderLength(length);
  return (status);
}


/*****************************************************************************/
/* @Function: SD_piSetCalibrationHeaderLength                                */
/* @Purpose :                                                                */
/*  Provided Interface to set the length of the header of incoming           */
/*  calibration packets.                                                     */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* length               IN      header length                                */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetCalibrationHeaderLength(UINT32 length)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetCalibrationHeaderLength(length);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piIsCMDReadOk                                               */
/* @Purpose :                                                                */
/*  Provided Interface to retrive the status of the last SRAM to CMD write   */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_TRANSFERSTAT      OUT     Last SRAM Page send to CMD result            */
/* @@                                                                        */
/*****************************************************************************/

SD_TRANSFERSTAT SD_piIsCMDReadOk(void)
{
SD_TRANSFERSTAT status;

  status = SD_opIsCMDReadOk();
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piSetSRAMRWAddress                                          */
/* @Purpose :                                                                */
/*  Provided Interface to set the address for I/O operation between SRAM and */
/*  processor.                                                               */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* address              IN      read/write address                           */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetSRAMRWAddress(UINT32 address)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetSRAMRWAddress(address);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piGetSRAMRWAddress                                          */
/* @Purpose :                                                                */
/*  Provided Interface to get the address for I/O operation between SRAM and */
/*  processor.                                                               */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* address              IN      read/write address                           */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

UINT32 SD_piGetSRAMRWAddress(void)
{
UINT32 Address;

  Address = SD_opGetSRAMRWAddress();
  return (Address);
}

/*****************************************************************************/
/* @Function: SD_piSetWPBUSReadPage                                          */
/* @Purpose :                                                                */
/*  Provided Interface to select the SRAM page be tranfered from MMSU or CMD */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* Page                 IN      SRAM page to be transfered                   */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetWPBUSReadPage(UINT32 Page)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetWPBUSReadPage(Page);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piSetWPBUSReadParams                                        */
/* @Purpose :                                                                */
/*  Provided Interface to set the stop address for the SRAM transfer into    */
/*  both tne MMSU od CMD.                                                    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* ReadParams           IN      structure to hold the stop page end address  */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetWPBUSReadParams(SD_DAQ_WRESULT ReadParams)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetWPBUSReadParams(ReadParams);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piSetCMDBUSReadPage                                         */
/* @Purpose :                                                                */
/*  Provided Interface to select the SRAM page be tranfered from CMD         */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* Page                 IN      SRAM page to be transfered                   */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetCMDBUSReadPage(UINT32 Page)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetCMDBUSReadPage(Page);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piSetCMDBUSReadParams                                       */
/* @Purpose :                                                                */
/*  Provided Interface to set the stop address for the SRAM transfer into    */
/*  the CMD.                                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* ReadParams           IN      structure to hold the stop page end address  */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piSetCMDBUSReadParams(SD_DAQ_WRESULT ReadParams)
{
SD_EXITCODE status = SD_SUCCESS;

  SD_opSetCMDBUSReadParams(ReadParams);
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piMaskInt                                                   */
/* @Purpose :                                                                */
/*  Provided Interface to mask the Interrupts of the SRAM module             */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* IntLine              IN      Interrupts to mask                           */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piMaskInt(SD_INTLINE IntLine)
{
SD_EXITCODE status = SD_SUCCESS;
#ifndef SIMULATOR
  SD_opMaskInt(IntLine);
#endif
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piUnMaskInt                                                 */
/* @Purpose :                                                                */
/*  Provided Interface to set the Interrupt Mask of the SRAM module          */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* IntLine              IN      Interrupts to unmask                         */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piUnMaskInt(SD_INTLINE IntLine)
{
SD_EXITCODE status = SD_SUCCESS;
#ifndef SIMULATOR
  SD_opUnMaskInt(IntLine);
#endif
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piIsIntMasked                                               */
/* @Purpose :                                                                */
/*  Provided Interface to check if an Interrupt of the SRAM module is masked */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* IntLine              IN      Interrupts to unmask                         */
/* @@                                                                        */
/*****************************************************************************/

BOOL SD_piIsIntMasked(SD_INTLINE IntLine)
{
  return SD_piIsIntMasked(IntLine);
}

/*****************************************************************************/
/* @Function: SD_piClearInt                                                  */
/* @Purpose :                                                                */
/*  Provided Interface to clear the Interrupt pending register of the        */
/*  SRAM module                                                              */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  IntLine             IN      Interrupt to be cleared                      */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_piClearInt(SD_INTLINE IntLine)
{
SD_EXITCODE status = SD_SUCCESS;
#ifndef SIMULATOR
  SD_opClearInt(IntLine);
#endif
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piIsIntPending                                              */
/* @Purpose :                                                                */
/*  Provided Interface to check if an Interrupt of the SRAM module is pending*/
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  IntLine             IN      Interrupt to be checked                      */
/*  BOOL                OUT     True if the Interrupt is pending             */
/* @@                                                                        */
/*****************************************************************************/

BOOL SD_piIsIntPending(SD_INTLINE IntLine)
{
BOOL IsPending;

  IsPending = SD_opIsIntPending(IntLine);
  return (IsPending);
}


/*****************************************************************************/
/* @Function: SD_piPIFReset                                                  */
/* @Purpose :                                                                */
/*  Provided Interface to reset the PIF I/F                                  */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  status_code          OUT     Return code                                 */
/* @@                                                                        */
/*****************************************************************************/

status_code SD_piPIFReset(void)
{
status_code status = SUCCESSFUL;
#ifndef SIMULATOR
  SD_opPIFReset();
#endif
  return (status);
}

/*****************************************************************************/
/* @Function: SD_piGetCMDCounterLeft                                         */
/* @Purpose :                                                                */
/*  Provided Interface to get the number of bytes left in CMD I/F queue      */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* UINT32                OUT     number of byte left in CMD queue            */
/* @@                                                                        */
/*****************************************************************************/

UINT32 SD_piGetCMDCounterLeft(void)
{
  return (SD_opGetCMDCounterLeft());
}
