/****************************************************************************
/*  F i l e   D a t a                                                        
/*                                                                           
/*  Module       :  SRAMManager                                                
/*  C.I. No.     :                                                           
/*  $Revision: 1.1.1.1 $
/*  $Date: 2006/04/25 09:00:21 $
/*  Belonging to :                                                           
/*               :                                                           
/*  $RCSfile: SD_SRAMDriver_op.c,v $
/*  Program Type :                                                           
/*  Sub-modules  :                                                           
/*                                                                           
/****************************************************************************
/*  S W   D e v e l o p m e n t   E n v i r o n m e n t                      
/*                                                                           
/*  Host system  :                                                           
/*  SW Compiler  :                                                           
/*  $Author: kusanagi $
/*               :                                                           
/****************************************************************************
/*  U p d a t i n g                                                          
/*                                                                           
/*  $Log: SD_SRAMDriver_op.c,v $
/*  Revision 1.1.1.1  2006/04/25 09:00:21  kusanagi
/*  These program extract in an XML format the info contained into the ROOT files generated by YODA from the PAMELA data. To visualize the XML files in a more human readable format a collection of XSL files are given in the Data subfolder.
/*
/*  Revision 1.2  2003/08/07 08:46:55  wizard
/*  SIMULATOR bugfix ifdef/ifndef
/*  SM_PktSize update bug fix in SM_op_PageHasData in order to fix the laben bug of SM_pi_GetPacket
/*  introduced the notification to a task from the PageHasData
/*
/*  Revision 1.1.1.1  2003/08/04 09:40:21  sebastiani
/*  Imported sources laben rel. 19.06.2003 integrated with pam2
/*
/*  Revision 1.11  2003/05/20 12:27:37  aurora
/*  removed wait loops for EM between two or more reads on PCMCIA
/*
/*  Revision 1.10  2003/05/05 09:00:06  aurora
/*  correct the header length programming in DAQSetup
/*
/*  Revision 1.9  2003/04/29 13:30:01  aurora
/*  introduced the new register TRIG3 for manage page overrun every 64Kbyte
/*
/*  Revision 1.8  2003/03/12 10:56:22  aurora
/*  added the provider for the counter of bytes left on CMD I/F queue
/*
/*  Revision 1.7  2003/01/30 10:15:35  aurora
/*  introduced ifdef for EM and FM
/*
/*  Revision 1.6  2002/10/17 10:07:14  zulia
/*  added SD_PIFReset
/*
/*  Revision 1.5  2002/07/31 14:30:10  zulia
/*  Added driver for indipendent control of CMD I/F from DAQ I/F
/*
/*  Revision 1.4  2002/05/09 08:16:35  zulia
/*  *  acceptance release
/*
/*                                                                           
/*****************************************************************************/

/*============================= Include File ================================*/

#include <src/SRAMManager/SRAMDriver/SD_SRAMDriver_op.h>

/*============================== local define ===============================*/

/*============================== local types ================================*/

/*****************************************************************************/

/*====== S R A M D r i v e r      P R O V I D E D    I N T E R F A C E ======*/

/*****************************************************************************/
/* @Function: SD_opSRAMDriverInit                                            */
/* @Purpose :                                                                */
/*  Provided Interface to init the SRAM driver                               */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

SD_EXITCODE SD_opSRAMDriverInit(void)
{
SD_EXITCODE status = SD_SUCCESS;

  return (status);
}

/*****************************************************************************/
/* @Function: SD_opSetDAQWritePage                                           */
/* @Purpose :                                                                */
/*  Provided Interface to select the SRAM page to hold new incoming data.    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* Page                 IN      SRAM page to be selected in input            */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetDAQWritePage(UINT32 Page)
{
  BYTE  *reg;
#ifndef SIMULATOR
  reg = (BYTE *) ALV_DPAGE;
  *reg = (BYTE)(Page & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opIsDAQWriteOk                                              */
/* @Purpose :                                                                */
/*  Provided Interface to retrive the status uf the last DAQ I/F write into  */
/*  SRAM.                                                                    */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_TRANSFERSTAT      OUT     status of the last DAQ I/F SRAM write        */
/* @@                                                                        */
/*****************************************************************************/

SD_TRANSFERSTAT SD_opIsDAQWriteOk(void)
{
BYTE  *reg;
BYTE trok;
#ifndef SIMULATOR
  reg = (BYTE *) ALV_DQEA3;
  trok = *reg;
  trok = (trok & 0x80)? SD_TRANSFEROK : SD_TRANSFERFAIL;
  return (trok);
#else
  return SD_TRANSFEROK;
#endif
}

/*****************************************************************************/
/* @Function: SD_opGetDAQWriteResult                                         */
/* @Purpose :                                                                */
/*  Provided Interface to retrive the end address of the data written by the */
/*  DAQ I/F in the selected SRAM page.                                       */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* SD_DAQ_RESULT        OUT     structure to hold the end address, page and  */
/*                              status of the last DAQ I/F SRAM write        */
/* @@                                                                        */
/*****************************************************************************/

SD_DAQ_WRESULT SD_opGetDAQWriteResult(void)
{
  SD_DAQ_WRESULT result = 0x00000000;
#ifndef SIMULATOR
  BYTE    data;
  UINT32  data32;
  BYTE    *reg;
  
  reg = (BYTE *) ALV_DQEA3;
  data = *reg;
  data = data & 0x0000000f;
  data32 = ((UINT32)data << 14);
  result = data32;

  reg = (BYTE *) ALV_DQEA2;
  data = *reg;
  data = data & 0x0000003f;
  data32 = ((UINT32)data << 8);
  result |= data32;

  reg = (BYTE *) ALV_DQEA1;
  data = *reg;
  data = data & 0x000000ff;
  data32 = ((UINT32)data);
  result |= data32;
#endif
  return (result);
}

/*****************************************************************************/
/* @Function: SD_opStartTransferToCMD                                        */
/* @Purpose :                                                                */
/*  Provided Interface to start the transfer of data stored in the pervoiusly*/
/*  selected SRAM to the CMD I/F.                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* @@                                                                        */
/*****************************************************************************/

void SD_opStartTransferToCMD(void)
{
BYTE  *reg;

  reg = (BYTE *) ALV_CMD;
  *reg = DUMMYWRITE;
}

/*****************************************************************************/
/* @Function: SD_opWriteSRAM                                                 */
/* @Purpose :                                                                */
/*  Provided Interface to write a data byte in the pervoiusly selected       */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* data                 IN      8bit data to store into perviously selected  */
/*                              SRAM address                                 */
/* @@                                                                        */
/*****************************************************************************/

void SD_opWriteSRAM(UINT32 data)
{
BYTE  *reg;
#ifndef SIMULATOR
  reg = (BYTE *) ALV_HBSRAM;
  *reg = (BYTE)(data & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opReadSRAM                                                  */
/* @Purpose :                                                                */
/*  Provided Interface to read a data byte from the pervoiusly selected      */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* BYTE                 OUT     8bit data read from perviously selected      */
/*                              SRAM address                                 */
/* @@                                                                        */
/*****************************************************************************/

UINT32 SD_opReadSRAM(void)
{
  BYTE  *reg;
  UINT32 data=0;
#ifndef SIMULATOR
  reg = (BYTE *) ALV_HBSRAM;
  data = (*reg);
  data &= 0x000000FF;
#endif
  return (data);
}

/*****************************************************************************/
/* @Function: SD_opSetEventTimeout                                           */
/* @Purpose :                                                                */
/*  Provided Interface to set the event timeout generation counter. The LSB  */
/*  has a resolution of 65us.                                                */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* tics                 IN      counter value                                */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetEventTimeout(UINT32 tics)
{
#ifndef SIMULATOR
  BYTE  *reg;

  reg = (BYTE *) ALV_ETO1;
  *reg = (BYTE)(tics & 0x000000FF);
  reg = (BYTE *) ALV_ETO2;
  *reg = (BYTE)((tics>>8) & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opSetDataTimeout                                            */
/* @Purpose :                                                                */
/*  Provided Interface to set the data timeout generation counter. The LSB   */
/*  has a resolution of 508ns.                                               */
/*  SRAM address.                                                            */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* tics                 IN      counter value                                */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetDataTimeout(UINT32 tics)
{
#ifndef SIMULATOR
  BYTE  *reg;

  reg = (BYTE *) ALV_DTO1;
  *reg = (BYTE)(tics & 0x00FF);
  reg = (BYTE *) ALV_DTO2;
  *reg = (BYTE)((tics>>8) & 0x00FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opTrigDataEvent                                             */
/* @Purpose :                                                                */
/*  Provided Interface that trigs the DAQ I/F to recieve incoming            */
/*  data packet, enabling the DTO, ETO timeouts and PAGE OVERRUN detection   */
/*  at each 16 Kbytes pages.                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* @@                                                                        */
/*****************************************************************************/

void SD_opTrigDataEvent(void)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_TRIG;
  *reg = DUMMYWRITE;
#endif
}

/*****************************************************************************/
/* @Function: SD_opTrigData64Event                                           */
/* @Purpose :                                                                */
/*  Provided Interface that trigs the DAQ I/F to recieve incoming            */
/*  data packet, enabling the DTO, ETO timeouts and PAGE OVERRUN detection   */
/*  at each 64 Kbytes pages.                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* @@                                                                        */
/*****************************************************************************/

void SD_opTrigData64Event(void)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_TRIG3;
  *reg = DUMMYWRITE;
#endif
}


/*****************************************************************************/
/* @Function: SD_opTrigCalibrationEvent                                      */
/* @Purpose :                                                                */
/*  Provided Interface that trigs the DAQ I/F to recieve incoming data       */
/*  enabling the DTO and ETO timeouts (for calibration mode)                 */
/*  and PAGE OVERRUN detection at 128 Kbyte                                  */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* @@                                                                        */
/*****************************************************************************/

void SD_opTrigCalibrationEvent(void)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_TRIG2;
  *reg = DUMMYWRITE;
#endif
}

/*****************************************************************************/
/* @Function: SD_opSetDataHeaderLength                                       */
/* @Purpose :                                                                */
/*  Provided Interface to set the length of the header of incoming data      */
/*  packets.                                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* length               IN      header length                                */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetDataHeaderLength(UINT32 length)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_DHD;
  *reg = (BYTE)(length & 0x000000FF);
#endif
}


/*****************************************************************************/
/* @Function: SD_opSetCalibrationHeaderLength                                */
/* @Purpose :                                                                */
/*  Provided Interface to set the length of the header of incoming           */
/*  calibration packets.                                                     */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* length               IN      header length                                */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetCalibrationHeaderLength(UINT32 length)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_CHD;
  *reg = (BYTE)(length & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opIsCMDReadOk                                               */
/* @Purpose :                                                                */
/*  Provided Interface to retrive the status of the last SRAM to CMD write   */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* TRANSFERSTAT         OUT     Last SRAM Page store into MMSU result        */
/* @@                                                                        */
/*****************************************************************************/

SD_TRANSFERSTAT SD_opIsCMDReadOk(void)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_CMDEA3;
  return (*reg & 0x80) ? SD_TRANSFEROK : SD_TRANSFERFAIL;
#else
  return SD_TRANSFEROK;
#endif
}

/*****************************************************************************/
/* @Function: SD_opSetSRAMRWAddress                                          */
/* @Purpose :                                                                */
/*  Provided Interface to set the address for I/O operation between SRAM and */
/*  processor.                                                               */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* address              IN      read/write address                           */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetSRAMRWAddress(UINT32 address)
{
#ifndef SIMULATOR
  BYTE  *reg;

  reg = (BYTE *) ALV_HBP1;
  *reg = (BYTE)(address & 0x000000FF);
  reg = (BYTE *) ALV_HBP2;
  *reg = (BYTE)((address>>8) & 0x000000FF);
  reg = (BYTE *) ALV_HBP3;
  *reg = (BYTE)((address>>14) & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opGetSRAMRWAddress                                          */
/* @Purpose :                                                                */
/*  Provided Interface to get the address for I/O operation between SRAM and */
/*  processor.                                                               */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* address              IN      read/write address                           */
/* SD_EXITCODE          OUT     Exit code                                    */
/* @@                                                                        */
/*****************************************************************************/

UINT32 SD_opGetSRAMRWAddress(void)
{
  
  UINT32 address = 0x00000000;
#ifndef SIMULATOR
  BYTE  *reg;
  
  reg = (BYTE *) ALV_HBP3;
  address = ( (((UINT32)*reg)<<14) & 0x0003C000 );
  reg = (BYTE *) ALV_HBP2;
  address |= ( (((UINT32)*reg)<<8) & 0x00003F00 );
  reg = (BYTE *) ALV_HBP1;
  address |= ( ((UINT32)*reg) & 0x000000FF );
#endif
  return (address);
}

/*****************************************************************************/
/* @Function: SD_opSetWPBUSReadPage                                          */
/* @Purpose :                                                                */
/*  Provided Interface to select the SRAM page be tranfered from MMSU        */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* Page                 IN      SRAM page to be transfered                   */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetWPBUSReadPage(UINT32 Page)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_WPSA;
  *reg = (BYTE)(Page & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opSetWPBUSReadParams                                        */
/* @Purpose :                                                                */
/*  Provided Interface to set the stop address for the SRAM transfer into    */
/*  the MMSU                                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* ReadParams           IN      structure to hold the stop page end address  */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetWPBUSReadParams(SD_DAQ_WRESULT ReadParams)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_WPEA1;
  *reg = (BYTE)(ReadParams & 0x000000FF);
  reg = (BYTE *) ALV_WPEA2;
  *reg = (BYTE)((ReadParams>>8) & 0x000000FF);
  reg = (BYTE *) ALV_WPEA3;
  *reg = (BYTE)((ReadParams>>14) & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opSetCMDBUSReadPage                                         */
/* @Purpose :                                                                */
/*  Provided Interface to select the SRAM page be tranfered from CMD         */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* Page                 IN      SRAM page to be transfered                   */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetCMDBUSReadPage(UINT32 Page)
{
#ifndef SIMULATOR
  BYTE  *reg;
  reg = (BYTE *) ALV_CMDSA;
  *reg = (BYTE)(Page & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opSetCMDBUSReadParams                                       */
/* @Purpose :                                                                */
/*  Provided Interface to set the stop address for the SRAM transfer into    */
/*  the CMD.                                                                 */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* ReadParams           IN      structure to hold the stop page end address  */
/* @@                                                                        */
/*****************************************************************************/

void SD_opSetCMDBUSReadParams(SD_DAQ_WRESULT ReadParams)
{
#ifndef SIMULATOR
  BYTE  *reg;

  reg = (BYTE *) ALV_CMDEA1;
  *reg = (BYTE)(ReadParams & 0x000000FF);
  reg = (BYTE *) ALV_CMDEA2;
  *reg = (BYTE)((ReadParams>>8) & 0x000000FF);
  reg = (BYTE *) ALV_CMDEA3;
  *reg = (BYTE)((ReadParams>>14) & 0x000000FF);
#endif
}

/*****************************************************************************/
/* @Function: SD_opMaskInt                                                   */
/* @Purpose :                                                                */
/*  Provided Interface to mask the Interrupts of the SRAM module             */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* IntLine              IN      Interrupts to mask                           */
/* @@                                                                        */
/*****************************************************************************/

void SD_opMaskInt(SD_INTLINE IntLine)
{
#ifndef SIMULATOR
  BYTE mask;
  BYTE *reg;

  reg = (BYTE *) ALV_MSK;
  mask = *reg;
  mask |= IntLine;
  *reg = mask;
#endif
}

/*****************************************************************************/
/* @Function: SD_opUnMaskInt                                                 */
/* @Purpose :                                                                */
/*  Provided Interface to set the Interrupt Mask of the SRAM module          */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* IntLine              IN      Interrupts to unmask                         */
/* @@                                                                        */
/*****************************************************************************/

void SD_opUnMaskInt(SD_INTLINE IntLine)
{
#ifndef SIMULATOR
  BYTE mask;
  BYTE *reg;
  
  reg = (BYTE *) ALV_MSK;
  mask = *reg;
  mask &= (~IntLine);
  *reg = mask;
#endif
}

/*****************************************************************************/
/* @Function: SD_opIsIntMasked                                               */
/* @Purpose :                                                                */
/*  Provided Interface to check if an Interrupt of the SRAM module is masked */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* IntLine              IN      Interrupts to unmask                         */
/* @@                                                                        */
/*****************************************************************************/

BOOL SD_opIsIntMasked(SD_INTLINE IntLine)
{

  BOOL status = FALSE;
#ifndef SIMULATOR
  BYTE mask;
  BYTE *reg;
  
  reg = (BYTE *) ALV_MSK;
  mask = *reg;
  mask &= IntLine;
  if(mask) status = TRUE;
#endif
  return (status);
}

/*****************************************************************************/
/* @Function: SD_opClearInt                                                  */
/* @Purpose :                                                                */
/*  Provided Interface to clear the Interrupt pending register of the        */
/*  SRAM module                                                              */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  IntLine             IN      Interrupt to be cleared                      */
/* @@                                                                        */
/*****************************************************************************/

void SD_opClearInt(SD_INTLINE IntLine)
{
#ifndef SIMULATOR
  BYTE *reg;
  
  reg = (BYTE*) ALV_INT;
  *reg = (BYTE)(~IntLine);
#endif
}

/*****************************************************************************/
/* @Function: SD_opIsIntPending                                              */
/* @Purpose :                                                                */
/*  Provided Interface to check if an Interrupt of the SRAM module is pending*/
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/*  IntLine             IN      Interrupt to be checked                      */
/*  BOOL                OUT     True if the Interrupt is pending             */
/* @@                                                                        */
/*****************************************************************************/

BOOL SD_opIsIntPending(SD_INTLINE IntLine)
{
BOOL isPending = FALSE;
#ifndef SIMULATOR
 BYTE pend;
 BYTE *reg;
 
 reg = (BYTE *) ALV_INT;
 pend = *reg;
 if((pend & IntLine) != 0) isPending = TRUE;
#endif
 return (isPending);
}


/*****************************************************************************/
/* @Function: SD_opPIFReset                                                  */
/* @Purpose :                                                                */
/*  Provided Interface to reset the PIF I/F                                  */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* @@                                                                        */
/*****************************************************************************/

void SD_opPIFReset(void)
{
#ifndef SIMULATOR
  BYTE  *reg;
  
  reg = (BYTE *)ALV_RESET;
  *reg = 0xAA;
#endif
}

/*****************************************************************************/
/* @Function: SD_opGetCMDCounterLeft                                         */
/* @Purpose :                                                                */
/*  Provided Interface to get the number of bytes left in CMD I/F queue      */
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* UINT32                OUT     number of byte left in CMD queue            */
/* @@                                                                        */
/*****************************************************************************/

UINT32 SD_opGetCMDCounterLeft(void)
{
  UINT32 byteLeft = 0x00000000;
#ifndef SIMULATOR
  BYTE  *reg;
  
  SD_opLatchCMDCounter();
  
  /* read the counter                                 */
  reg = (BYTE *) ALV_QCMD3;
  byteLeft = ( (((UINT32)*reg) << 14) & 0x0001C000 );
  reg = (BYTE *) ALV_QCMD2;
  byteLeft |= ( (((UINT32)*reg) << 8) & 0x00003F00 );
  reg = (BYTE *) ALV_QCMD1;
  byteLeft |= (  ((UINT32)*reg)       & 0x000000FF );
#endif
  return (byteLeft);
}

/*****************************************************************************/
/* @Function: SD_opLatchCMDCounter                                           */
/* @Purpose :                                                                */
/*  Provided Interface to latch the the number of bytes left in CMD I/F queue*/
/*                                                                           */
/* @@                                                                        */
/* @Parameter Name      @Mode   @Description                                 */
/* @@                                                                        */
/*****************************************************************************/

void SD_opLatchCMDCounter(void)
{
#ifndef SIMULATOR
  BYTE  *reg;
  
  /* this write latch the counter in to the registers */
  reg = (BYTE *) ALV_QCMD1;
  *reg = 0xAA;
#endif
}
