/**
* LogToXML
* author  Nagni
* version 2.0 - 13 February 2006
*
* Description: Generate an XML file starting from a PAMELA unpacked file.
*
* Parameters:
* 	base - the path where to find the PAMELA unpacked root file.
*	outDirectory - the path where to put the output file.
*	xslPath - the path where to find an XSL format for the output.
*
* version 1.0 - 03 March 2005
* First implementation
*
* version 2.0 - 13 February 2006
* Modified for new Yoda unpacking structure (one single file)
*/

#include <log/LogRecord.h>
#include <log/LogEvent.h>
#include <PscuHeader.h>
#include <EventHeader.h>
#include <fstream>

#include <TFile.h>
#include <TObjString.h>
#include <TString.h>
#include <TTree.h>

void LogToXML(TString base, TString outDirectory = "", TString xslPath = ""){
  Int_t    	tmpSize;
  ofstream 	outputFile;
  stringstream 	oss;
  
  pamela::LogEvent  	*logev    = 0;
  pamela::LogRecord 	*logrec   = 0;
  pamela::EventHeader 	*eh  	  = 0;
  pamela::PscuHeader  	*ph  	  = 0;
  TFile   		*rootFile = new TFile(base);
  

  if (rootFile->IsZombie()) printf("The %s file does not exist", base.Data());
  TString fileName = ((TObjString*)base.Tokenize('\/')->Last())->GetString();
  TString filePath = base;
  filePath.ReplaceAll(fileName, "");
  fileName.ReplaceAll(".root", "");

  oss.str("");
  if (outDirectory == "") {
  	oss << fileName.Data() << "Log.xml";
  } else {
  	oss << outDirectory.Data() << fileName.Data() << "Log.xml";
  }
  const char* xmlFilePath = oss.str().c_str();
  
  outputFile.open(xmlFilePath, ios::trunc);
  if (!outputFile.is_open()){
  	printf("Cannot open the file %s for the output", xmlFilePath);
	exit(0);
  }

    TTree *tr = (TTree*)rootFile->Get("Log");
    Long64_t nevents    = tr->GetEntries();
    tr->SetBranchAddress("Log", &logev);
    tr->SetBranchAddress("Header", &eh);
    outputFile << "<?xml version='1.0' encoding='ISO-8859-1'?>\n";
    outputFile << "<!-- Prologo XML -->\n";
    outputFile << "<?xml-stylesheet type='text/xsl' href='" << xslPath.Data() << "'?>\n";
    outputFile << "<ROOT_SOURCE>\n";
    outputFile << "<TABLE_HEAD>\n";
    outputFile << "<COL id='prima'>Record OBT</COL>\n";
    outputFile << "<COL id='seconda'>Mask Type</COL>\n";
    outputFile << "<COL id='terza'>File Name</COL>\n";
    outputFile << "<COL id='quinta'>Info1</COL>\n";
    outputFile << "</TABLE_HEAD>\n";
    
    
    //<!-- Istruzione che indica il documento XSLT da associare -->    
    for (int i = 0; i < nevents; i++){
      tr->GetEntry(i);
      tmpSize = logev->Records->GetEntries();
      ph      = eh->GetPscuHeader();
      outputFile << "<LOG_EVENT>\n";
      outputFile << "\t<PACKET_OBT>" << dec << ph->GetOrbitalTime() << "</PACKET_OBT>\n";
      outputFile << "\t<PACKET_NUM>" << ph->GetCounter() << "</PACKET_NUM>\n";
      outputFile << "\t<LOG_RECORDS>\n";
      for (Int_t j = 0; j < tmpSize; j++){
	logrec = (pamela::LogRecord*)logev->Records->At(j);
	outputFile << "\t\t<LOG_RECORD>\n";
	outputFile << "\t\t\t<RECORD_OBT>" << dec << logrec->RECORD_OBT      << "</RECORD_OBT>\n";
	outputFile << "\t\t\t<MASKTYPE>"   << (short)logrec->MASKTYPE   << "</MASKTYPE>\n";
	outputFile << "\t\t\t<FILE_ID>"    << (int)logrec->FILE_ID    << "</FILE_ID>\n";
	outputFile << "\t\t\t<LINE_NO>"    << logrec->LINE_NO         << "</LINE_NO>\n";
	outputFile << "\t\t\t<INFO1>"      << logrec->INFO1           << "</INFO1>\n";
	outputFile << "\t\t</LOG_RECORD>\n";
      }
      outputFile << "\t</LOG_RECORDS>\n";
      outputFile << "</LOG_EVENT>\n";	
    }
    outputFile << "</ROOT_SOURCE>\n";
    outputFile.close();
}

int main(int argc, char* argv[]){
    TString outDir     = "";
    TString xslPath    = "";
    
 if (argc < 2){
    printf("You have to insert at least the file to analyze \n");
    printf("Try '--help' for more information. \n");
    exit(1);
  }  

  if (!strcmp(argv[1], "--help")){
        printf( "Usage: LogToXML FILE [OPTION] \n");
        printf( "\t --help                 Print this help and exit \n");
        printf( "\t -outDir[path]          Path where to put the output [default ~/tmp] \n");
        printf( "\t -xslPath[path]         Set the path to a XSL file for formatting [default  '']\n");
        exit(1);
  }

  for (int i = 2; i < argc; i++){
    if (!strcmp(argv[i], "-outDir")){
        if (++i >= argc){
            printf( "-outDir needs arguments. \n");
            printf( "Try '--help' for more information. \n");
            exit(1);
        } else {
            outDir = argv[i];
            continue;
        }
    }

    if (!strcmp(argv[i], "-xslPath")) 
    	if (++i >= argc){
            printf( "-xslPath needs arguments. \n");
            printf( "Try '--help' for more information. \n");
            exit(1);
        } else {
        	xslPath = argv[i];
        	continue;
        }
    }
	LogToXML(argv[1], outDir, xslPath);
}


   
