/**
 * \file CaloPreSampler.h
 * \author Emiliano Mocchiutti (2007/07/18)
 */
#ifndef calopresampler_h
#define calopresampler_h

#include <PamLevel2.h>

#include <TTree.h>
#include <TFriendElement.h>
#include <TChain.h>
#include <TFile.h>
#include <TList.h>
#include <TKey.h>
#include <TSystemFile.h>
#include <TSystemDirectory.h>
#include <TSQLServer.h>

#include <iostream>

using namespace std;

/**
 *
 * Class to store and calculate variables useful for presampler analysis
 */
class CaloPreSampler : public TObject {

 private:
    //
    PamLevel2 *L2;
    Bool_t debug;
    //
    // needed to avoid reprocessing the same event over and over to obtain the variables;
    //
    UInt_t OBT;
    UInt_t PKT;
    UInt_t atime;
    Bool_t ssel;
    //
    Bool_t usepl18x;
    //
    Bool_t simulation; ///< True when using simulated data, false by default;
    //
    CaloLevel0 *event; ///< CaloLevel0 event needed to process data from level0/1 to level2
    CaloStrip *cstrip; ///< CaloStrip object needed to easily recover parameter files
    CaloLevel2 *pcalo; ///< CaloLevel2 object which contains variables calculated with a reduced calorimeter.
    //   CaloLevel1 *c1; ///< CaloLevel1 object which contains variables calculated with a reduced calorimeter.
    //
    Int_t N; ///< Number of W planes to be used as presampler. NOTICE: none the silicon detectors attached to these planes will be used;
    Int_t NC; ///< Number of W planes to be used as calorimeter.
    Bool_t sel; ///< Selection mode: planes from 1 to 22-N are used, plane 18 - N is masked if "emulate18" variable is true;
    Bool_t cont; ///< Contamination mode: planes from N to 22 are used;
    Bool_t emulate18; ///< If true emulates the non-working plane 18;
    Bool_t nox; ///< If true do not use X-view
    Bool_t noy; ///< If true do not use Y-view
    Bool_t forcecalo;
    Int_t mask[2][22];
    //
    Float_t rigdefault; ///< Rigidity to be used if no track is available or in standalone mode
    Bool_t withtrk; ///< Use or not the tracker
    Int_t forcefitmode;
    //

 public:
    //
    //
    // CaloLevel1* GetCaloLevel1(){ Process(); return c1;}; ///< Retrieve CaloLevel1 pointer which contains variables calculated with a reduced calorimeter.
    CaloLevel2* GetCaloLevel2(){ Process(); return pcalo;}; ///< Retrieve CaloLevel2 pointer which contains variables calculated with a reduced calorimeter.
    CaloLevel2* GetLevel2Pointer(){ printf(" OBSOLETE, use GetCaloLevel2Pointer()!\n"); return pcalo;}; ///< Retrieve CaloLevel2 pointer which contains variables calculated with a reduced calorimeter.
    CaloLevel2* GetCaloLevel2Pointer(){ return pcalo;}; ///< Retrieve CaloLevel2 pointer which contains variables calculated with a reduced calorimeter.
    void SetCaloLevel2Pointer(CaloLevel2 *); ///< Set CaloLevel2 pointer which contains variables calculated with a reduced calorimeter.
    //
    CaloTrkVar* GetCaloTrack(Int_t t);
    //
    void SetNoWpreSampler(Int_t n); ///< Set the number of W planes to be used as presampler. NOTICE: none the silicon detectors attached to these planes will be used. Default: N = 4;
    void SetNoWcalo(Int_t n); ///< Set the number of W planes to be used as calorimeter. Default: NC = 22-N (N presampler);
    void SplitInto(Int_t NoWpreSampler, Int_t NoWcalo);
    void Selection(){sel = true; cont = false;}; ///< Set selection mode: planes from 1 to 22-N are used, plane 18 - N is masked if "emulate18" variable is true (DEFAULT);
    void Contamination(){sel = false; cont = true;}; ///< Set contamination mode: planes from N to 22 are used.
    void EmulatePlane18(Bool_t emu){emulate18 = emu;} ///< Emulates the non-working plane 18; Default = true;
    void Simulation(Bool_t sim){simulation = sim;}; ///< Set simulated data flag. Default FALSE;
    //
    CaloPreSampler();
    CaloPreSampler(PamLevel2 *L2);
    ~CaloPreSampler(){ Delete(); };
    //
    Int_t GetNoWpreSampler(){return N;}; ///< Get the number of W planes used as presampler. 
    Int_t GetNoWcalo(){return NC;}; ///< Get the number of W planes used as calorimeter.
    void SetDebug(Bool_t d){ debug=d; };
    void UseTracker(Bool_t ch){ withtrk = ch; };
    void SetDefaultRig(Float_t rig){ rigdefault = rig; };
    void SetForceFitMode(Int_t mode){ forcefitmode = mode; };
    void MaskX(Bool_t mskx){ nox = mskx; };
    void MaskY(Bool_t msky){ noy = msky; };
    void Mask(Int_t view, Int_t plane){ mask[view][plane] = 1;};
    void ForceCaloFit(){forcecalo=true;};
    void ForceNextProc(){atime=0;};
    //
    void UsePlane18X(Bool_t use){usepl18x = use;};
    //
    void Clear();
    void Clear(Option_t *option){Clear();};
    void Delete();
    void Delete(Option_t *option){Delete();};
    //
    void Process(); ///< Process data 
    void Print();
    void Print(Option_t *option){Print();};
    //
    CaloTrkVar* AddCaloTrkVar(float *al,int trktag);//ELENA

    ClassDef(CaloPreSampler,4);
};

#endif

