

/**
 * \file CaloNuclei.h
 * \author Emiliano Mocchiutti
 */
#ifndef calonuclei_h
#define calonuclei_h

#include <PamLevel2.h>

#include <TTree.h>
#include <TFriendElement.h>
#include <TChain.h>
#include <TFile.h>
#include <TList.h>
#include <TKey.h>
#include <TSystemFile.h>
#include <TSystemDirectory.h>
#include <TSQLServer.h>

#include <iostream>

using namespace std;

/**
 *
 * Class to store and calculate variables useful for nuclei analysis
 */
class CaloNuclei : public TObject {

 private:
    //
    PamLevel2 *L2;
    Bool_t debug;
    Bool_t usetrack;
    //
    // needed to avoid reprocessing the same event over and over to obtain the variables
    //
    UInt_t OBT;
    UInt_t PKT;
    UInt_t atime;
    Int_t tr;
    Int_t sntr;
    //
    Int_t interplane; ///< Number of available dE/dx measurements before interaction or exit from the calo (interaction plane) 
    Int_t N; ///< Number of dE/dx measurements to be used to calculate qpremeanN, default N = 5
    Int_t R; ///< Number of strip to be used around the trajectory to calculate qpremeanN, default R = 3
    Float_t preq; ///< Energy release (MIP) up to the interaction plane (included)
    Float_t postq; ///< Energy release (MIP) from the interaction plane (excluded) up to the last plane
    Float_t stdedx1; ///< Energy release (MIP) on the first Silicon detector (Y EVEN) around the strip with maximum energy release (no track information).
    Float_t dedx1; ///< Energy release (MIP) along the track on the first Silicon detector (Y EVEN).
    Float_t dedx3; ///< Energy release (MIP) along the track on the first three Silicon detectors (Y EVEN, X EVEN, Y ODD).
    Float_t qpremean; ///< Truncated mean (MIP) along the track up to the interaction plane preq using three points
    Float_t qpremeanN; ///< Truncated mean (MIP) along the track up to the interaction plane preq using N points
    Float_t ethr; ///< Threshold (MIP) needed to find the interaction plane
    Bool_t multhit; ///< True if the interaction plane has been determined by multiple hit counting
    Bool_t gap; ///< True if determining the interaction plane a big (>5 planes) gap has been found between a point and another along the track
    //

 public:
    //
    //
    Int_t Get_interplane(){ Process(); return interplane;}; ///< Number of available dE/dx measurements before interaction or exit from the calo (interaction plane) 
    Int_t Get_N(){ return N;}; ///< Number of available dE/dx measurements before interaction or exit from the calo (interaction plane) 
    Int_t Get_R(){ return R;}; ///< Number of strip to be used around the trajectory to calculate qpremeanN, default R = 3 
    Float_t Get_preq(){ Process(); return preq;}; ///< Energy release (MIP) up to the interaction plane (included)
    Float_t Get_postq(){ Process(); return postq;}; ///< Energy release (MIP) from the interaction plane (excluded) up to the last plane
    Float_t Get_StdEdx1(){ Process(); return stdedx1;}; ///< Energy release (MIP) on the first Silicon detector (Y EVEN) around the strip with maximum energy release (no track information, 3 strips in total).
    Float_t Get_dEdx1(){ Process(); return dedx1;}; ///< Energy release (MIP) along the track on the first Silicon detector (Y EVEN).
    Float_t Get_dEdx3(){ Process(); return dedx3;}; ///< Energy release (MIP) along the track on the first three Silicon detectors (Y EVEN, X EVEN, Y ODD).
    Float_t Get_qpremean(){ Process(); return qpremean;}; ///< Truncated mean (MIP) along the track up to the interaction plane preq using three points
    Float_t Get_qpremeanN(){ Process(); return qpremeanN;}; ///< Truncated mean (MIP) along the track up to the interaction plane preq using N points
    Float_t Get_ethr(){ Process(); return ethr;}; ///< Threshold (MIP) needed to find the interaction plane
    Bool_t IsMulthit(){ Process(); return multhit;}; ///< True if the interaction plane has been determined by multiple hit counting
    //
    void Set_N(Int_t n){ N=n;};
    void Set_R(Int_t r){ R=r;};
    //
    CaloNuclei();
    CaloNuclei(PamLevel2 *L2);
    ~CaloNuclei(){ Delete(); };
    //
    void SetDebug(Bool_t d){ debug=d; };
    void UseTrack(Bool_t d){ usetrack=d; };
    //
    void Clear();
    void Clear(Option_t *option){Clear();};
    void Delete();
    //
    void Process(); ///< Process data for track number 0
    void Process(Int_t ntr); ///< Process data for track number ntr
    void Print();
    //
    ClassDef(CaloNuclei,2);
};

#endif

