/**
 * \file CaloEnergy.h
 * \authors Emiliano Mocchiutti & Giovanna Jerse
 */
#ifndef caloenergy_h
#define caloenergy_h

#include <PamLevel2.h>

#include <TTree.h>
#include <TFriendElement.h>
#include <TChain.h>
#include <TFile.h>
#include <TList.h>
#include <TKey.h>
#include <TSystemFile.h>
#include <TSystemDirectory.h>
#include <TSQLServer.h>

#include <CaloPreSampler.h>
#include <CaloProfile.h>

#include <iostream>

using namespace std;

/**
 *
 */
class CaloEnergy : public TObject {

 private:
    //
    PamLevel2 *L2; ///< PamLevel2 object
    Bool_t debug; ///< debug flag
    Bool_t usepl18x;
    //
    // needed to avoid reprocessing the same event over and over to obtain the variables
    //
    UInt_t OBT; ///< CPU OBT
    UInt_t PKT; ///< CPU packet number
    UInt_t atime; ///< event absolute time
    TString sntr; ///< string containing the list of section the user want to process
    UInt_t AOBT; ///< CPU OBT
    UInt_t APKT; ///< CPU packet number
    UInt_t aatime;///< event absolute time
    TString asntr;///< string containing the list of section the user want to process
    //
    // margins, acceptance and containment
    //
    Float_t fM; ///< margin in the strip direction
    Float_t fM1; ///< margin along the strip reading direction
    Int_t fPl; ///< number of dE/dx measurements over the maximum that are used to find the energy
    Float_t fCount; ///< Number of sections inside the acceptance (only the section given by the user are checked)
    Int_t fRad; ///< Radius [strip] of the cylinder used to integrate the energy along the track, if negative radius is inf (the whole plane is used). Default: -1
    Int_t fNumSec; ///< Number of sections given by the user
    Bool_t fXosel; ///< true if event is contained in section XO
    Bool_t fXesel; ///< true if event is contained in section XE
    Bool_t fYosel; ///< true if event is contained in section YO   
    Bool_t fYesel; ///< true if event is contained in section YE
    Bool_t fSel; ///< true if event is contained in at least one of the given section (independet mode) or in all the given section (coherent mode)
    Bool_t fPartsel; ///< true if the event is contained only up to the last plane used for energy determination (can be used in conjunction with fXXmin)
    Int_t fXeout; ///< last plane [0,11] for which the trajectory is contained in section XE
    Int_t fYeout; ///< last plane [0,11] for which the trajectory is contained in section YE
    Int_t fXoout; ///< last plane [0,11] for which the trajectory is contained in section XO
    Int_t fYoout; ///< last plane [0,11] for which the trajectory is contained in section YO
    Int_t fXomin; ///< last plane [0,11] for which the trajectory MUST be contained in section XO. Default 1000 means all the planes, if less than 10 events can be only partially contained in a section (NB: THIS INTRODUCE AN ENERGY DEPENDENT SELECTION CONTAINMENT EFFICIENCY)
    Int_t fXemin; ///< last plane [0,11] for which the trajectory MUST be contained in section XE. Default 1000 means all the planes, if less than 10 events can be only partially contained in a section (NB: THIS INTRODUCE AN ENERGY DEPENDENT SELECTION CONTAINMENT EFFICIENCY)
    Int_t fYomin; ///< last plane [0,11] for which the trajectory MUST be contained in section YO. Default 1000 means all the planes, if less than 10 events can be only partially contained in a section (NB: THIS INTRODUCE AN ENERGY DEPENDENT SELECTION CONTAINMENT EFFICIENCY)
    Int_t fYemin; ///< last plane [0,11] for which the trajectory MUST be contained in section YE. Default 1000 means all the planes, if less than 10 events can be only partially contained in a section (NB: THIS INTRODUCE AN ENERGY DEPENDENT SELECTION CONTAINMENT EFFICIENCY)
    Bool_t indep; ///< flag to switch between INDEPENDENT or COHERENT mode, default false - COHERENT mode selected
    Float_t X0pl; ///< transversed X0 for each W plane taking into account inclination of the trajectory
    //
    // conversion factors
    //
    Float_t fConv_rxe; ///< MIP - energy conversion factor for section XE
    Float_t fConv_rxo; ///< MIP - energy conversion factor for section XO
    Float_t fConv_rye; ///< MIP - energy conversion factor for section YE
    Float_t fConv_ryo; ///< MIP - energy conversion factor for section YO
    // 
    // Longitudinal fit
    // 
    Bool_t fLong; ///< if true use the integral of the longitudinal profile to measure the energy (NOT IMPLEMENTED YET), default FALSE
    // 
    // Energies (MIP)
    //
    Float_t fXOen_maxplane; ///< total energy [MIP] used for energy determination as given by section XO
    Float_t fYOen_maxplane; ///< total energy [MIP] used for energy determination as given by section YO
    Float_t fXEen_maxplane; ///< total energy [MIP] used for energy determination as given by section XE
    Float_t fYEen_maxplane; ///< total energy [MIP] used for energy determination as given by section YE
    Float_t xomax_en; ///< energy at plane of maximum of section XO
    Float_t xemax_en; ///< energy at plane of maximum of section XE
    Float_t yomax_en; ///< energy at plane of maximum of section YO
    Float_t yemax_en; ///< energy at plane of maximum of section YE
    Float_t energyxe; ///< 11 planes detected energy [MIP] for section XE
    Float_t energyyo; ///< 11 planes detected energy [MIP] for section YO
    Float_t energyxo; ///< 11 planes detected energy [MIP] for section XO
    Float_t energyye; ///< 11 planes detected energy [MIP] for section YE
    Float_t en_xep[11]; ///< detected energy [MIP] for each plane of section XE
    Float_t en_yop[11]; ///< detected energy [MIP] for each plane of section YO
    Float_t en_xop[11]; ///< detected energy [MIP] for each plane of section XO
    Float_t en_yep[11]; ///< detected energy [MIP] for each plane of section YE
    Float_t encol[2][3]; ///< detected energy [MIP] for each column of views x and y
    Float_t entot[2]; ///< detected energy [MIP] for views x and y
    // 
    // Energies (GV)
    //
    Float_t fEnergyxe; ///< Energy as measured by section XE
    Float_t fEnergyxo; ///< Energy as measured by section XO
    Float_t fEnergyye; ///< Energy as measured by section YE
    Float_t fEnergyyo; ///< Energy as measured by section YO
    Float_t fEnergy; ///< Energy as measured by the average of the used section in "Independent mode" or energy as measured by the used section in "Coherent mode" 
    //
    // Plane of maximum
    //
    Int_t fMax_planexe; ///< plane of maximum energy release (independent mode) or last plane used for energy measurement (coherent mode) for section XE
    Int_t fMax_planexo; ///< plane of maximum energy release (independent mode) or last plane used for energy measurement (coherent mode) for section XO
    Int_t fMax_planeyo; ///< plane of maximum energy release (independent mode) or last plane used for energy measurement (coherent mode) for section YO
    Int_t fMax_planeye; ///< plane of maximum energy release (independent mode) or last plane used for energy measurement (coherent mode) for section YE
    Float_t fMax_plane; ///< average max plane [0,11] (independent mode) or last plane for energy measurement [0,43] (coherent mode)
    Float_t x0max; ///< plane of maximum given externally (only test purpose)
    Bool_t fAllpl; ///< use all 96 strips for each plane to determine the maximum OR only the energy along the track as defined with fRad
    //
    // Geometry
    //
    Float_t xe1; ///< position of strip  1 section XE
    Float_t xe2; ///< position of strip 32 section XE
    Float_t xe3; ///< position of strip 33 section XE
    Float_t xe4; ///< position of strip 64 section XE
    Float_t xe5; ///< position of strip 65 section XE
    Float_t xe6; ///< position of strip 96 section XE
    Float_t yo1; ///< position of strip  1 section YO
    Float_t yo2; ///< position of strip 32 section YO
    Float_t yo3; ///< position of strip 33 section YO
    Float_t yo4; ///< position of strip 64 section YO
    Float_t yo5; ///< position of strip 65 section YO
    Float_t yo6; ///< position of strip 96 section YO
    Float_t xo1; ///< position of strip  1 section XO
    Float_t xo2; ///< position of strip 32 section XO
    Float_t xo3; ///< position of strip 33 section XO
    Float_t xo4; ///< position of strip 64 section XO
    Float_t xo5; ///< position of strip 65 section XO
    Float_t xo6; ///< position of strip 96 section XO
    Float_t ye1; ///< position of strip  1 section YE
    Float_t ye2; ///< position of strip 32 section YE
    Float_t ye3; ///< position of strip 33 section YE
    Float_t ye4; ///< position of strip 64 section YE
    Float_t ye5; ///< position of strip 65 section YE
    Float_t ye6; ///< position of strip 96 section YE
    Float_t  track_coordx[22][2]; ///< XO and XE views, position (x and y) of the trajectory according to the fit
    Float_t  track_coordy[22][2]; ///< YO and YE views, position (x and y) of the trajectory according to the fit
    Float_t trk_z[22][2]; ///< Z position of calorimeter planes
    //
    // decode estrip
    //
    Float_t en; ///< energy [mip] for decodeestrip
    Int_t view; ///< view for decodeestrip
    Int_t plane; ///< plane for decodeestrip
    Int_t strip; ///< strip for decodeestrip
    Float_t enstrip[2][22][96]; ///< detected energy [MIP] for each strip of calorimeter
    //                                                   y ^
    //                                                    || 6 7 8
    // Columns                                            || 3 4 5
    //                                                    || 0 1 2
    Int_t fColumn; ///< Column number for the event [0,8] =============> x
    Int_t fColXE; ///< Column number for section XE
    Int_t fColXO; ///< Column number for section XO
    Int_t fColYE; ///< Column number for section YE
    Int_t fColYO; ///< Column number for section YO
    Bool_t multicol; ///< accept or not multicolumns events
    // 
    // other stuff
    //
    Bool_t fSimu; ///< true if we are using simulated data, default false
    CaloPreSampler *cp; ///< pointer to calopresampler object (object constructed only when invoking method UseCaloPreSampler() , default: use level2 data).
    CaloLong *clong; ///< pointer to calolong object (object constructed only when invoking method UseLongFit(), default use energy up to maximum).
    //
    // private methods
    //
    void DefineGeometry(); ///< called by constructors to fill geometrical variables (like xe1 etc).
    void Set(); ///< called by contructors to define default variables

 public:
    //
    // constructors and destructors
    //
    CaloEnergy(); ///< default constructor (does nothing)
    CaloEnergy(PamLevel2 *L2); ///< constructor
    CaloEnergy(PamLevel2 *L2, Bool_t simulation); ///< constructor
    ~CaloEnergy(){ Delete(); }; ///< default destructor
    //
    // Setters and behaviour methods
    //
    void SetDebug(Bool_t d){ debug=d; }; ///< set the debug flag (verbose print-out on STDOUT), default is false
    //
    void UseCaloPreSampler(); ///< use pre-sampler routine to refit the track (level2 default fitting could be wrong, in this case we force "shower fitting" in the DV library).
    void UseLevel2(); ///< use level2 default fitting
    //
    void UseLongFit();///< use or not the longitudinal fit to determine the energy 	
    void UseMeasuredEnergyUpToMax(){ fLong = false;}; ///< use the measured energy to determine the maximum (default)
    //
    void IndependentMode(){ indep = true; }; ///< Set the independent mode
    void CoherentMode(){ indep = false; }; ///< Set the coherent mode
    //
    void MultiColumns(){multicol = true;}; ///< accept multicolumns events
    void SingleColumn(){multicol = false;}; ///< accept events only if contained in a single column
    //
    void UseAllPlane2FindMax(){ fAllpl = true;};///< find the maximum (not long fit) integrating over all the 96 strips of the planes even if SetRadius has been used [default]
    void UseMeasuredEnergy2FindMax(){ fAllpl = false;};///< find the maximum (not long fit) using the energy measured and used to calculate the result 
    //
    void SetMargin(Float_t margin){fM = margin ; fM1 = margin - 0.122 - 0.096 + 0.096; if ( fM1 < 0. ) fM1 = 0.;}; ///< set the margin from the border of the silicon sensor (not from the first strip), set the same margin for both the directions
    void SetMarginStripDirection(Float_t margin){fM = margin ;}; ///< set the margin from the border of the silicon sensor (not from the first strip) in the strip direction
    void SetMarginStripReading(Float_t margin){fM1 = margin -0.122 - 0.096 + 0.096;};  ///< set the margin from the border of the silicon sensor (not from the first strip) in the strip reading direction
    //
    void SetRadius(Int_t strip){fRad = strip;}; ///< set the radius of the cylinder
    void SetMaxPlaneOffset(Int_t noplanes){fPl = noplanes;}; ///< set the number of dE/dx measurements to be used after the maximum
    //
    void SetX0max(Float_t xm){ x0max = xm;}; ///< set the plane of maximum from external source X0 (test purpose only)
    void SetRigX0max(Float_t rig){ x0max = -0.5+log(rig/0.0076);}; ///< set the plane of maximum from external source rigidity (GeV) (test purpose only)
    //
    void SetMinimumContainment(Int_t plane); ///< set the last plane [0,11] for which the trajectory MUST be contained in all the sections. Default 1000 means all the planes, if less than 10 events can be only partially contained in a section (NB: THIS INTRODUCE AN ENERGY DEPENDENT SELECTION CONTAINMENT EFFICIENCY)
    void SetMinimumContainment(TString section, Int_t plane); ///< set the last plane [0,11] for which the trajectory MUST be contained in section "section". Default 1000 means all the planes, if less than 10 events can be only partially contained in a section (NB: THIS INTRODUCE AN ENERGY DEPENDENT SELECTION CONTAINMENT EFFICIENCY)
    //
    void SetConversionFactor(Float_t conv_r); ///< Set the MIP-GV conversion factor for all the four sections.
    void SetConversionFactor(TString section, Float_t conv_r); ///< Set the MIP-GV conversion factor for section "section".
    //
    void ForceProcessing(){atime=0; PKT=0; APKT=0; aatime=0;}; ///< Force processing the event even if the same request is made twice without a getentry from pamlevel2
    //
    // Getters and checks methods
    //
    //
    Bool_t IsInsideAcceptance(TString section); ///< returns true if event is inside acceptance of the given sections (all if coherent mode, at least one in independent mode)
    Bool_t IsInsideReducedAcceptance(){return fPartsel;}; ///< returns true if the event is inside acceptance only up to the last used plane (see fXomin etc)
    //
    Bool_t IsInsideXE(){return(IsInsideAcceptance("XE"));};
    Bool_t InsideXEcheck(){return fXesel;};
    Bool_t IsInsideXO(){return(IsInsideAcceptance("XO"));};
    Bool_t InsideXOcheck(){return fXosel;};
    Bool_t IsInsideYE(){return(IsInsideAcceptance("YE"));};
    Bool_t InsideYEcheck(){return fYesel;};
    Bool_t IsInsideYO(){return(IsInsideAcceptance("YO"));};
    Bool_t InsideYOcheck(){return fYosel;};
    //
    Float_t GetEnergy(){ Process(); return fEnergy;}; ///< returns the energy [GV] determined for this event
    Float_t GetEnergy(TString section){ Process(section); return fEnergy;}; ///< returns the energy [GV] determined for this event
    //
    Float_t GetCount(){ return fCount;}; ///< returns the number of section inside acceptance for this event (equal to the number of given section in coherent mode)
    //
    Float_t GetEnergyAtMaxplane(TString section); ///< returns the energy [MIP] at the plane of maximum for section "section"
    Float_t GetMipEnergyAtMaxplane(TString section); ///< returns the energy [MIP] at the plane of maximum for section "section"
    //
    Float_t GetMaxEnergy(); ///< returns the total energy [MIP] before conversion
    Float_t GetMaxEnergy(TString section); ///< returns the total energy [MIP] before conversion for section "section"
    Float_t GetMipEnergy(); ///< returns the total energy [MIP] before conversion
    Float_t GetMipEnergy(TString section); ///< returns the total energy [MIP] before conversion for section "section"
    //
    Int_t GetMaxplane(TString section); ///< returns the plane of maximum (independent mode) or the last used plane (coherent mode) for section "section"
    Float_t GetMaxplane(){ return fMax_plane;}; ///< returns the average max plane [0,11] (independent mode) or last plane for energy measurement [0,43] (coherent mode)    
    //
    Int_t GetMinimumContainment(TString section); ///< get the last plane [0,11] for which the trajectory MUST be contained in section "section".
    //
    Float_t GetConversionFactor(TString section); ///< Get the MIP-GV conversion factor for section "section".
    //
    Float_t *Get_track_coordx(){ return *track_coordx;}; ///< X position of the track for all the planes and views
    Float_t *Get_track_coordy(){ return *track_coordy;}; ///< Y position of the track for all the planes and views
    //
    Float_t  Get_track_coordx(Int_t i, Int_t j){ return track_coordx[i][j];}; ///< X position of the track for plane i and view j
    Float_t  Get_track_coordy(Int_t i, Int_t j){ return track_coordy[i][j];}; ///< X position of the track for plane i and view j
    //
    Float_t *GetEncol(){ return *encol;}; ///< integrated energy over columns (encol[2][3]) [MIP]
    Float_t GetEncol(Int_t i, Int_t j){ return encol[i][j];}; ///< integrated energy over view i and column j [MIP]
    Float_t GetEncol(Int_t i); ///< integrated energy over view i given fColumn [MIP]
    Float_t *GetEntot(){ return entot;}; ///< integrated energy over views (entot[2])  [MIP]
    Float_t GetEntot(Int_t i){ return entot[i];}; ///< integrated energy over all view i [MIP]
    //
    Int_t GetColumn(){return fColumn;}; ///< number of column which contains the track
    Int_t GetColumn(TString section); ///< number of column which contains the track for section "section"
    //
    Float_t Get_X0pl(){return X0pl;}; ///< transversed X0 for each W plane taking into account inclination of the trajectory
    Float_t GetX0max(){ return x0max;}; ///< get the given X0 (test purpose only)
    //
    // Get pointers
    //
    CaloLong* GetCaloLong(){return clong;}; ///< Get calolong object.
    CaloPreSampler* GetCaloPreSampler(){return cp;}; ///< Get pre-sampler object.
    CaloEnergy* GetCaloEnergyPointer(){return this;}; ///< Get CaloEnergy pointer
    //
    // Other methods
    //
    
    void UsePlane18X(Bool_t use){usepl18x = use;};
    //
    void Clear(); ///< clear varibles
    void Clear(Option_t *option){Clear();}; ///< compatibility with TObject
    void Delete(); ///< delete object
    void Delete(Option_t *option){Delete();}; ///< compatibility with TObject
    //
    void Process(); ///< Process the event
    void Process(TString section);  ///< Process the event for section "section"
    void Print(); ///< Print variables on STDOUT
    void Print(Option_t *option){Print();}; ///< compatibility with TObject
    //
    ClassDef(CaloEnergy,4);
};

#endif

