//
//	Example to loop over events, reading more than one file.
//
//	Input parameters are:
//  - dir:      name of the directory where Level2 files are stored
//  - list:     name of a text file with the list of file names to be processed.
//    If list="" all ROOT files inside the directory are processed (ROOT files that does not
//    match Pamela Level2 structure are automatically discarded)
//  - treelist: string containing the list of trees to be loaded (es: "-ORB-ND" or "-ALL+TRK+CAL+TOF").
//    Possible options are:
//    +ALL +TRK +CAL +TRG +TOF +ND +AC +S4 +ORB
//    -ALL -TRK -CAL -TRG -TOF -ND -AC -S4 -ORB
//	- nmax:     maximum number of events to be processed
//
//  All selected trees in all the files are chained and made friends.
//  The whole "blob" can be accessed as if it was a single tree.
//
void example3(TString dir,TString list="", TString treelist="+ALL", Int_t nmax=500000000){

	//   Create a timer object to benchmark this loop
	TStopwatch timer;
	timer.Start();
	TString wd=gSystem->WorkingDirectory();
	//
	PamLevel2*  event  = new PamLevel2();       // << create pamela event
	
	// =========================================================
	// << First a TList of root files is created
	// << (either from a specific list or from the whole input directory)
 	TList *l  = event->GetListOfLevel2Files(dir,list);
	// << Hence a TChain is created from the list. 
	// << All the trees required by the user are made friend.
	TChain *T = event->GetPamTree(l,treelist);
	// << In the following the TChain can be used similarly to a TTree.
	//
	// NB! It is still possible to read a single file and get a TTree, by means of the methods:
	// TTree* LoadPamTrees(TFile *f);
	// TTree* LoadPamTrees(TFile *f,TString treelist);
	// =========================================================
	
	Int_t nevent = T->GetEntries();	
		
	Int_t ntrk=0;
	if(nevent < nmax)nmax=nevent;
	cout << endl<<" Start loop over events   "<<endl;
    for (Int_t i=0; i<nmax;i++){
	//
		T->GetEntry(i);
		
		if(event->GetNTracks() > 1)cout << " Ev "<<i << " multiple tracks: "<< event->GetNTracks() << endl;
		
		if(event->TrkLevel2::ntrk() >0){
			ntrk++;
		};
		
		for(Int_t it=0; it<event->GetNTracks(); it++){
			
			PamTrack *track = event->GetTrack(it);
			PamTrack *image = 0;
			
			if(track->HasImage())image = event->GetTrackImage(it);
			
			delete track;
			if(image)delete image;
		
		};

    };                                       // end loop over the events
    cout << endl << endl << " Done "<< endl<<endl;
	cout << ntrk <<" tracks over "<<nmax<<" events ("<< 100*ntrk/nmax<<"%)"<<endl;

//  Stop timer and print results
	timer.Stop();
	Double_t rtime = timer.RealTime();
	Double_t ctime = timer.CpuTime();
	printf("RealTime=%f seconds, CpuTime=%f seconds\n",rtime,ctime);

	gSystem->ChangeDirectory(wd);
	
};
