//
//  Example to integrate all detector info by means of the PamLevel2 class
//
//  An object of this class has access to all the public members of all the detector classes.
//  Some methods allows to sort tracks and solve the tracker y-view ambiguity (reject track images)
//
//  --- Modified on January 2007 ---
//
//  In order to access the members of the detector classes you have now to call the 
//  methods
//
//  TrkLevel2*   PamLevel2::GetTrkLevel2()
//  CaloLevel2*  PamLevel2::GetCaloLevel2()
//  OrbitalInfo* PamLevel2::GetOrbitalInfo()
//  ecc...
//
//  For example:
//
//  PamLevel2*  pam_event  = new PamLevel2();       
//  cout << pam_event->GetCaloLevel2()->qtot << endl;
//
//  The same is for the track. The class PamTrack combines the track-related info from
//  tracker, calorimeter and TOF. To access the members of these detector track-related members
//  you have to call the methods:
//
//  TrkTrack*   PamTrack::GetTrkTrack()
//  CaloTrkVar* PamTrack::GetCaloTrack()
//  ToFTrkVar*  PamTrack::GetToFTrack()
//
//  For example:
//
//  PamTrack *track = pam_event->GetTrack(0); //<< retrieve first track, solving the track-image ambiguity
//  if(track)cout<< track->GetTrkTrack()->GetRigidity()<<endl;
//
void example1(TString file){
    //
    // histos
    //
    TH1F* qtot    =  new TH1F("qtot","Total energy in the calorimeter",1000,0.,2000.);
    TH1F* npaddle =  new TH1F("npaddle","Total number of hit ToF paddles",16,-0.5,15.5);
    TH1F* ntrack  =  new TH1F("ntr","Number of fitted tracks",6,-0.5,5.5);
    //
    TH1F* rig   = new TH1F("rig","Particle rigidity",100,0.,100.);
    TH1F* resxs = new TH1F("resxs","Spatial residual (X) on the 1^ calo plane (sorted tracks)",100,-10.,10.);
    TH1F* resys = new TH1F("resys","Spatial residual (Y) on the 1^ calo plane (sorted tracks)",100,-10.,10.);
    TH1F* resxi = new TH1F("resxi","Spatial residual (X) on the 1^ calo plane (image tracks)",100,-10.,10.);
    TH1F* resyi = new TH1F("resyi","Spatial residual (Y) on the 1^ calo plane (image tracks)",100,-10.,10.); 
    //
    PamLevel2*  pam_event  = new PamLevel2();       // << create pamela event
    //
    TFile f(file);
    //
    TTree *T = pam_event->GetPamTree(&f);          // << get Pamela trees from file f

    Int_t nevent = T->GetEntries();
    //
    cout << endl<<" Start loop over events   ";
    for (Int_t i=0; i<nevent;i++){
	//
	pam_event->Clear();

	T->GetEntry(i);
	//================================================================================
	// some general quantities
	//================================================================================
	// tracker
	ntrack->Fill( pam_event->GetTrkLevel2()->GetNTracks() );									//<<
	// calorimeter
	qtot->Fill( pam_event->GetCaloLevel2()->qtot );												//<<
	// ToF
	Int_t npa=0;
	for(Int_t ipa=0; ipa<6; ipa++)npa = npa + pam_event->GetToFLevel2()->GetNHitPaddles(ipa);	//<<
	npaddle->Fill(npa);
	//================================================================================
	// track related variables
	//================================================================================
	for(Int_t it=0; it<pam_event->GetTrkLevel2()->GetNTracks(); it++){     // << loop over the "physical" tracks (no track images)
	    //
	    // << get the it-th physical pamela track 
	    // << PamTrack combines the tracker, calorimeter and ToF track-related variables
	    // << (in case of image, choose the best track by means of calorimeter and ToF data)
	    //
	    PamTrack *track = pam_event->GetTrack(it);								//<<
	    //
	    // << if the track fit is good, fill some histos
	    //
	    if( track->GetTrkTrack()->chi2 > 0 && track->GetTrkTrack()->chi2 < 100){
		// 
		// << spatial residuals on the first calorimeter plane for the sorted track 
		//
		Float_t rxs = track->GetCaloTrack()->tbar[0][0] - pam_event->GetCaloLevel2()->cbar[0][0];
		Float_t rys = track->GetCaloTrack()->tbar[0][1] - pam_event->GetCaloLevel2()->cbar[0][1];
		resxs->Fill(rxs);
		resys->Fill(rys);
		//
		rig->Fill( track->GetTrkTrack()->GetRigidity() );
		//
		// << spatial residuals on the first calorimeter plane for the image track 
		//
		if(track->GetTrkTrack()->HasImage()){                                // << if the sorted track has an image...
		    //
		    PamTrack *image = pam_event->GetTrackImage(it);   // << ...get the image track
		    //
		    Float_t rxi = image->GetCaloTrack()->tbar[0][0] - pam_event->GetCaloLevel2()->cbar[0][0];
		    Float_t ryi = image->GetCaloTrack()->tbar[0][1] - pam_event->GetCaloLevel2()->cbar[0][1];
		    resxi->Fill(rxi);
		    resyi->Fill(ryi);
		};
	    };
	};  // end loop over tracks
	//================================================================================

    };                                       // end loop over the events
    cout << endl << endl << " Done "<< endl<<endl;
    //
    // close file
    //
    f.Close();	
    
};
