

/**
 * \file PamLevel2.h
 * \author Elena Vannuccini
 */
#ifndef pamlevel2_h
#define pamlevel2_h

#include <TRefArray.h>

#include <TrkStruct.h>
#include <TrkParams.h>
#include <TrkLevel0.h>
#include <TrkLevel1.h>
#include <TrkLevel2.h>
#include <TrkHough.h>
#include <CaloLevel0.h>
#include <CaloLevel1.h>
#include <CaloLevel2.h>
#include <ToFLevel2.h>
#include <TrigLevel2.h>
#include <S4Level2.h>
#include <NDLevel2.h>
#include <AcLevel2.h>
#include <OrbitalInfo.h>
#include <RunInfo.h>
#include <GLTables.h>


#include <TTree.h>
#include <TFriendElement.h>
#include <TChain.h>
#include <TFile.h>
#include <TList.h>
#include <TKey.h>
#include <TSystemFile.h>
#include <TSystemDirectory.h>
#include <TSQLServer.h>

#include <iostream>

#define NCLONES 10

using namespace std;

/**
 * \brief Class to describe particle tracks in the apparatus, including tracker and track-related variables from other detectors (calorimeter and TOF).
 *
 * This class inherites from TrkTrack, CaloTrkVar ecc... 
 * 
 */
//class PamTrack : public TrkTrack, public CaloTrkVar, public ToFTrkVar {
class PamTrack : public TObject {

 private:

    TrkTrack*   trk_track;
    CaloTrkVar* calo_track;
    ToFTrkVar*  tof_track;

    bool candeleteobj;

//    TClonesArray *Cluster;        ///< clusters (if stored)

 public:

    PamTrack();
    PamTrack(TrkTrack* , CaloTrkVar* , ToFTrkVar* );
    PamTrack(const PamTrack&);
    ~PamTrack(){ Delete(); };

    void Clear();
    void Clear(Option_t *option){Clear();};
    void Delete();

    TrkTrack*   GetTrkTrack(){return trk_track;};
    CaloTrkVar* GetCaloTrack(){return calo_track;};
    ToFTrkVar*  GetToFTrack(){return tof_track;};

    ClassDef(PamTrack,1);
};

/**
 * \brief Class to describe PAMELA LEVEL2 data.
 *
 * This class inherites from TrkLevel2, CaloLevel2 ecc... and it overrides the TrkLevel2:GetTrack(int it) and TrkLevel2:GetTrackImage(int it) methods. 
 * This allows to solve the tracker ambiguity (that is, to choose between track images) by using also information from other detectors.
 */
		
//class PamLevel2 : public CaloLevel2, public CaloLevel1, public ToFLevel2, public TrkLevel2, public TrkLevel1, public TrkHough, public TrigLevel2, public S4Level2, public NDLevel2, public AcLevel2, public OrbitalInfo {
class PamLevel2 : public TObject {

 private:

    // ------------------------------
    // variables to retrieve run info
    // ------------------------------
//    Int_t     irun;          //current run entry, relative to the TChain
    Long64_t  irun;          //current run entry, relative to the TChain
//    ULong64_t runfirstentry; //first entry of run, relative to the TChain
//    ULong64_t runlastentry;  //last  entry of run, relative to the TChain
//    Int_t     irunentry;     //position of current entry, relative to the run
    Long64_t  runfirstentry; //first entry of run, relative to the TChain
    Long64_t  runlastentry;  //last  entry of run, relative to the TChain
    Long64_t  irunentry;     //position of current entry, relative to the run
    TChain*   run_tree;
    TTree*    run_tree_clone;
    TChain*   sel_tree;
    TTree*    sel_tree_clone;
    
    // ---------------------------------
    // variables to retrieve level0 info
    // ---------------------------------
    TFile*      l0_file;
    TTree*      l0_tree;
    Int_t       iroot;      
    Int_t       prevshift; // to remember if any shift is needed to synchronize l0 and l2 data instead of calculating any time
    // --------------------------------------------
    // variables to store Level0 info (if required)
    // --------------------------------------------
    EventHeader* h0_obj;
    TrkLevel0*   trk0_obj;
    CaloLevel0*  calo0_obj;

    // ---------------------------
    // variables to connect the DB
    // ---------------------------
    TSQLServer* dbc;
    TString host;
    TString user;
    TString psw;

    // ---------------------------------------------
    // variables to retrieve and clone level1-2 info
    // ---------------------------------------------
    TChain* pam_tree;
    TTree*  pam_tree_clone[NCLONES];

    // --------------------------------------------
    // variables to store Level0 info (if required)
    // --------------------------------------------
    TrkLevel1*   trk1_obj;
    TrkLevel2*   trk2_obj;
    TrkHough*    trkh_obj;
    CaloLevel1*  calo1_obj;
    CaloLevel2*  calo2_obj;
    ToFLevel2*   tof_obj;
    TrigLevel2*  trig_obj;
    S4Level2*    s4_obj;
    NDLevel2*    nd_obj;
    AcLevel2*    ac_obj;
    OrbitalInfo* orb_obj;
    GL_RUN*      run_obj;
    SoftInfo*    soft_obj; // Emiliano

    // -----------------------------------
    // variables to sort the track
    // -----------------------------------
    TClonesArray *tsorted;
    TClonesArray *timage;
    PamTrack    *track; //store the last required 
    TString howtosort;
    Float_t sortthr;

    // -----
    // flags
    // -----
    Bool_t CAL2; //level2 branch
    Bool_t CAL1; //level1 branch
    Bool_t CAL0; //yoda

    Bool_t TRK2; //level2 branch
    Bool_t TRK1; //level1 branch
    Bool_t TRKh; //hough branch
    Bool_t TRK0; //yoda

    Bool_t TOF;
    Bool_t TOF0;

    Bool_t TRG;
    Bool_t S4;
    Bool_t ND;
    Bool_t AC;
    Bool_t ORB;

    Bool_t RUN;

    Bool_t ISGP; //GP simulation


    Int_t  SELLI; //trees of selected events (there is a selection list)

    void Initialize();


 public:
	 
    PamLevel2(TString ddir,TString list,TString detlist);
    PamLevel2(TString ddir,TString list);
    PamLevel2();
    ~PamLevel2(){Delete();};
	
    void Clear();
    void Reset();
    void Delete();

	 
//  =============================================
//  methods needed to read/write Level0-1-2 trees
//  =============================================
    void    SetWhichTrees(TString);
    void    GetWhichTrees(TFile*);

    TTree*  GetPamTree(TFile*,TString);
    TTree*  GetPamTree(TFile* f){return GetPamTree(f,"");};

    TList*  GetListOfLevel2Files(TString, TString);
    Bool_t  CheckLevel2File(TString);
    TChain* GetPamTree(TList*,TString);
    TChain* GetPamTree(TString ddir,TString list,TString detlist){ return GetPamTree(GetListOfLevel2Files(ddir,list),detlist); };
    TChain* GetPamTree(TString ddir,TString list){ return GetPamTree(GetListOfLevel2Files(ddir,list),""); };

    TTree*  GetRunTree(TFile*);
    TChain* GetRunTree(TList*);
    TChain* GetRunTree(TString ddir,TString list){ return GetRunTree(GetListOfLevel2Files(ddir,list)); };
    Bool_t  UpdateRunInfo(TTree *,Long64_t);
    Bool_t  UpdateRunInfo(TChain *,Long64_t);
    Bool_t  UpdateRunInfo(Long64_t iev);//{ return UpdateRunInfo(run_tree,iev); };

    void    CreateCloneTrees0(TChain *fChain, TFile* file);
    void    CreateCloneTrees(TFile* file);
    void    CreateCloneTrees(){CreateCloneTrees(NULL);};
    void    FillCloneTrees();
    TTree  *GetCloneTree(TString);
    void    WriteCloneTrees();

    void    SetBranchAddress(TTree *);
    void    SetBranchAddress(TChain *);

//    Int_t   GetEntry(Int_t);
    Int_t   GetEntry(Long64_t);
    Int_t   GetYodaEntry(); 

    Long64_t GetEntries(){return pam_tree->GetEntries();};
    TChain* GetPamTree(){return pam_tree;};
    TChain* GetRunTree(){return run_tree;};
    TTree*  GetYodaTree();

    TFile*  GetYodaFile(){return l0_file;}; // can be useful to know the name of level0 file we are using or similar things

//  =============
//  DB connection
//  =============
    void SetDBHost(TString str){host=str;};
    void SetDBUser(TString str){user=str;};
    void SetDBPsw(TString str){psw=str;};
    TSQLServer* GetSQLServer(){return dbc;};
    Bool_t SetDBConnection();

//  ========================================================
//  methods needed to retrieve subdetector general variables
//  ========================================================
    void*       GetPointerTo(const char*);
    EventHeader  *GetEventHeader()  { return h0_obj; };

/*     TrkLevel0    *GetTrkLevel0()   { return trk0_obj; }; */
/*     CaloLevel0   *GetCaloLevel0()  { return calo0_obj; }; */
    TrkLevel0    *GetTrkLevel0();
    CaloLevel0   *GetCaloLevel0();


    TrkLevel1    *GetTrkLevel1()  { return trk1_obj; };
    TrkLevel2    *GetTrkLevel2()  { return trk2_obj; };
    TrkHough     *GetTrkHough()   { return trkh_obj; };
    CaloLevel1   *GetCaloLevel1() { return calo1_obj;};
    CaloLevel2   *GetCaloLevel2() { return calo2_obj;};
    ToFLevel2    *GetToFLevel2()  { return tof_obj;  };
    TrigLevel2   *GetTrigLevel2() { return trig_obj; };
    S4Level2     *GetS4Level2()   { return s4_obj;   };
    NDLevel2     *GetNDLevel2()   { return nd_obj;   };
    AcLevel2     *GetAcLevel2()   { return ac_obj;   };
    OrbitalInfo  *GetOrbitalInfo(){ return orb_obj;  };
    GL_RUN       *GetRunInfo()    { return run_obj;  };
    SoftInfo     *GetSoftInfo()   { return soft_obj; }; // Emiliano

//  ==============================================
//  methods to retrieve subdetectors stored tracks
//  ==============================================
    TrkTrack*   GetTrkStoredTrack(int seqno){return trk2_obj->TrkLevel2::GetStoredTrack(seqno); };
    CaloTrkVar* GetCaloStoredTrack(int);
    ToFTrkVar*  GetToFStoredTrack(int);
    //
    PamTrack*   GetPamTrackAlong(TrkTrack*);
//	PamTrack*   GetPamTrackAlong(CaloTrkVar*);      //se serve
//	PamTrack*   GetPamTrackAlong(ToFTrkVar*);      //se serve
	
//  =================================
//  methods to retrieve pamela tracks
//  =================================
    PamTrack*     GetStoredTrack(Int_t);
    //
    void          SortTracks(TString);          //assign value to variable sorted_tracks
    void          SortTracks();          //assign value to variable sorted_tracks
//    TRefArray*    GetTracks();                  //return sorted_tracks
    TClonesArray* GetTracks();                  //return sorted_tracks
    PamTrack*     GetTrack(int);
    PamTrack*     GetTrackImage(int);
    void          SetSortingMethod(TString how){ howtosort = how; return;};
    TString       GetSortingMethod(){return howtosort;};
    void          SetNucleiSortingThreshold(Float_t nucleithr){ sortthr = nucleithr; return;};
    Float_t       GetNucleiSortingThreshold(){ return sortthr;};
	
//  =================================
//  other methods
//  =================================
//    Int_t ntrk(){ return trk2_obj->TrkLevel2::ntrk(); };

    // -----
    // flags methods
    // -----
    Bool_t IsCAL2(){return CAL2;}; //level2 branch
    Bool_t IsCAL1(){return CAL1;}; //level1 branch
    Bool_t IsCAL0(){return CAL0;}; //yoda

    Bool_t IsTRK2(){return TRK2;}; //level2 branch
    Bool_t IsTRK1(){return TRK1;}; //level1 branch
    Bool_t IsTRKh(){return TRKh;}; //hough branch
    Bool_t IsTRK0(){return TRK0;}; //yoda

    Bool_t IsTOF(){return TOF;};
    Bool_t IsTOF0(){return TOF0;};

    Bool_t IsTRG(){return TRG;};
    Bool_t IsS4(){return S4;};
    Bool_t IsND(){return ND;};
    Bool_t IsAC(){return AC;};
    Bool_t IsORB(){return ORB;};

    Bool_t IsRUN(){return RUN;};

    Bool_t IsGood();

    Int_t  IsSELLI(){return SELLI;}; //trees of selected events (there is a selection list)

    Bool_t IsGP(){return ISGP;};

    void   SetGP(Bool_t tt){ISGP=tt;};


    ClassDef(PamLevel2,1);
};

//===============================================================================================
//
//
//
//
//===============================================================================================


#endif

