#include <PamLevel2.h>
//--------------------------------------
//
//
//--------------------------------------
/**
 * Default constructor 
 */
PamTrack::PamTrack(){
    trk_track  = this->TrkTrack::GetTrkTrack();
    calo_track = this->CaloTrkVar::GetCaloTrkVar();
    tof_track  = this->ToFTrkVar::GetToFTrkVar();
};
//--------------------------------------
//
//
//--------------------------------------
/**
 * Constructor 
 */
PamTrack::PamTrack(TrkTrack* t, CaloTrkVar* c, ToFTrkVar* o){ 
    trk_track  = this->TrkTrack::GetTrkTrack();
    calo_track = this->CaloTrkVar::GetCaloTrkVar();
    tof_track  = this->ToFTrkVar::GetToFTrkVar();
    if(t) *trk_track  = *t;
    if(c) *calo_track = *c;
    if(o) *tof_track  = *o;	
};

//--------------------------------------
//
//
//--------------------------------------
/**
 * Constructor 
 */
PamLevel2::PamLevel2(){
	
    trk2_obj  = TrkLevel2::GetTrkLevel2();
    trk1_obj  = TrkLevel1::GetTrkLevel1();
    trkh_obj  = TrkHough::GetTrkHough();
    calo1_obj = CaloLevel1::GetCaloLevel1();
    calo2_obj = CaloLevel2::GetCaloLevel2();
    tof_obj   = ToFLevel2::GetToFLevel2();
    trig_obj  = TrigLevel2::GetTrigLevel2();
    s4_obj    = S4Level2::GetS4Level2();
    nd_obj    = NDLevel2::GetNDLevel2();
    ac_obj    = AcLevel2::GetAcLevel2();
    orb_obj   = OrbitalInfo::GetOrbitalInfo();

    run_obj   = new GL_RUN();
    
//     trk1_clone  = 0;	
//     trk2_clone  = 0;
//     trkh_clone  = 0;
//     calo1_clone = 0;
//     calo2_clone = 0;
//     tof_clone   = 0;
//     trig_clone  = 0;
//     s4_clone    = 0;
//     nd_clone    = 0;
//     ac_clone    = 0;
//     orb_clone   = 0;

//     T_clone  = 0;
//     C_clone  = 0;
//     O_clone  = 0;
//     R_clone  = 0;
//     S_clone  = 0;
//     N_clone  = 0;
//     A_clone  = 0;
//     B_clone  = 0;

    for(Int_t i=0; i<8; i++ )tree_clone[i]=NULL;

    sorted_tracks = new TRefArray();
    
    CAL1 = true;
    CAL2 = true;
    TRK2 = true;
    TRK1 = false;
    TRKh = false;
    TRG = true;
    TOF = true;
    S4  = true;
    ND  = true;
    AC  = true;
    ORB = true;
    
    RUN = true;

};
/**
 * Destructor 
 */
PamLevel2::~PamLevel2(){		
    delete run_obj;
    Clear();
};
/**
 * Clear the event
 */
void PamLevel2::Clear(){
	
    TrkLevel1::Clear();
	
    TrkLevel2::Clear();
    CaloLevel2::Clear();
    ToFLevel2::Clear();
    TrigLevel2::Clear();
    S4Level2::Clear();
    NDLevel2::Clear();
    AcLevel2::Clear();
    OrbitalInfo::Clear();

//     if(trk1_clone)trk1_clone->Clear();
//     if(trk2_clone)trk2_clone->Clear();
//     if(trkh_clone)trkh_clone->Clear();
//     if(calo1_clone)calo1_clone->Clear();
//     if(calo2_clone)calo2_clone->Clear();
//     if(tof_clone)tof_clone->Clear();
//     if(trig_clone)trig_clone->Clear();
//     if(s4_clone)s4_clone->Clear();
//     if(nd_clone)nd_clone->Clear();
//     if(ac_clone)ac_clone->Clear();
//     if(orb_clone)orb_clone->Clear();
    
    sorted_tracks->Delete(); // clean the reference array
    
};


//--------------------------------------
//
//
//--------------------------------------
void *PamLevel2::GetPointerTo(const char* c ){

    TString objname = c;
    if(!objname.CompareTo("TrkLevel1"))return &trk1_obj;
    if(!objname.CompareTo("TrkLevel2"))return &trk2_obj;
    if(!objname.CompareTo("TrkHough"))return &trkh_obj;
    if(!objname.CompareTo("CaloLevel1"))return &calo1_obj;
    if(!objname.CompareTo("CaloLevel2"))return &calo2_obj;
    if(!objname.CompareTo("ToFLevel2"))return &tof_obj;
    if(!objname.CompareTo("TrigLevel2"))return &trig_obj;
    if(!objname.CompareTo("S4Level2"))return &s4_obj;
    if(!objname.CompareTo("NDLevel2"))return &nd_obj;
    if(!objname.CompareTo("AcLevel2"))return &ac_obj;
    if(!objname.CompareTo("OrbitalInfo"))return &orb_obj;
    if(!objname.CompareTo("RunInfo"))return &run_obj;
    return NULL;
};
//--------------------------------------
//
//
//--------------------------------------
/**
 * Retrieves the calorimeter track matching the seqno-th tracker stored track. 
 * (If seqno = -1 retrieves the self-trigger calorimeter track)
 */
 CaloTrkVar *PamLevel2::GetCaloStoredTrack(int seqno){
	 
     if( CaloLevel2::ntrk()==0 ){
	 cout << "PamLevel2::GetCaloStoredTrack(int) : requested tracker SeqNo "<< seqno <<" but no Calorimeter tracks are stored"<<endl;
	 return NULL;
     };
     
     CaloTrkVar *c = 0;
     Int_t it_calo=0;
     
     do{ 
	 c = CaloLevel2::GetCaloTrkVar(it_calo);
	 it_calo++; 
     } while( c && seqno != c->trkseqno && it_calo < CaloLevel2::ntrk());	
     
     if(!c || seqno != c->trkseqno){
	 c = 0;
	 if(seqno!=-1)cout << "PamLevel2::GetCaloStoredTrack(int) : requested tracker SeqNo "<< seqno <<" does not match Calorimeter stored tracks"<<endl;
     };
     return c;
     
 };
//--------------------------------------
 //
 //
//--------------------------------------
/**
  * Retrieves the ToF track matching the seqno-th tracker stored track. 
  * (If seqno = -1 retrieves the tracker-independent tof track)
 */
 ToFTrkVar *PamLevel2::GetToFStoredTrack(int seqno){
	 
     if( ToFLevel2::ntrk()==0 ){
	 cout << "PamLevel2::GetToFStoredTrack(int) : requested tracker SeqNo "<< seqno <<" but no ToF tracks are stored"<<endl;
	 return NULL;
     };
     
     ToFTrkVar *c = 0;
     Int_t it_tof=0;
     
     do{ 
	 c = ToFLevel2::GetToFTrkVar(it_tof);
	 it_tof++; 
     } while( c && seqno != c->trkseqno && it_tof < ToFLevel2::ntrk());	
     
     if(!c || seqno != c->trkseqno){
	 c = 0;
	 if(seqno!=-1)cout << "PamLevel2::GetToFStoredTrack(int) : requested tracker SeqNo "<< seqno <<" does not match ToF stored tracks"<<endl;
     };
     return c;
     
 };

//--------------------------------------
 //
 //
//--------------------------------------
/**
 * Give the pamela track associated to a tracker track, retrieving related calorimeter and tof track information. 
 */
 PamTrack* PamLevel2::GetPamTrackAlong(TrkTrack* t){
	 
     CaloTrkVar *c = 0;
     ToFTrkVar  *o = 0;
     
     if(CAL2) c = GetCaloStoredTrack(t->GetSeqNo());
     if(TOF) o = GetToFStoredTrack(t->GetSeqNo());
     
//    if(t && c && o)track = new PamTrack(t,c,o);
     PamTrack *track = new PamTrack(t,c,o);
     
     return track;
 
 };
//--------------------------------------
//
//
//--------------------------------------
/**
 * Retrieves the it-th stored track. 
 * It override TrkLevel2::GetTrack(int it). 
 * @param itrk Track number, ranging from 0 to GetNTracks().
 */

PamTrack* PamLevel2::GetStoredTrack(Int_t itrk){
    
    PamTrack *track = 0;
    
    if( itrk >=0 && itrk < TrkLevel2::ntrk() ){
	
	TrkTrack *t = TrkLevel2::GetStoredTrack(itrk);
	track = GetPamTrackAlong(t);
	
    }else{
	cout << "PamLevel2::GetStoredTrack(int) : tracker track SeqNo "<< itrk <<" does not exist (GetNTracks() = "<<TrkLevel2::GetNTracks()<<")"<<endl;
    };
    
    return track;
    
}
//--------------------------------------
//
//
//--------------------------------------
/**
 * Sort physical (tracker) tracks and stores them in the TRefArray (of TrkTrack objects) which pointer is  PamLevel2::sorted_tracks.
 * @param how String to set the sorting cryterium (es: "CAL" or "TRK+CAL+TOF" ecc...).
 * Sorting cryteria:
 * TRK: lower chi**2 
 * CAL: lower Y spatial residual on the first calorimeter plane
 * TOF: bigger numebr of hit PMTs along the track, on S12 S21 S32 (where paddles are along the Y axis).
 * The default sorting cryterium is "TOF+CAL".
 * 
 * The total number of physical tracks is always given by GetNTracks() and the it-th physical track can be retrieved by means of the methods GetTrack(int it) and GetTrack(int it, TString how).
 */
void PamLevel2::SortTracks(TString how){

//    cout << "call SortTracs() "<<endl;
  //Save current Object count
    Int_t ObjectNumber = TProcessID::GetObjectCount();

//    cout << "ObjectNumber  "<<ObjectNumber <<endl;
    
    sorted_tracks->Delete(); //temporaneo???
	
    // loop over the tracks sorted by the tracker
    Bool_t use_TRK = how.Contains("TRK", TString::kIgnoreCase);
    Bool_t use_CAL = how.Contains("CAL", TString::kIgnoreCase);
    Bool_t use_TOF = how.Contains("TOF", TString::kIgnoreCase);
    
    if( !CAL2 &&  use_CAL) use_CAL = false;
    if( !TOF &&  use_TOF) use_TOF = false;
    
    if( !TRK2 ){
//	cout << "SortTracks() : without tracker does not work!!! (not yet)" << endl;
	return;
    };

    //   cout << "use_CAL "<<use_CAL<<" use_TOF "<<use_TOF<<" use_TRK "<<use_TRK <<endl;
    
    for(Int_t i=0; i < TrkLevel2::GetNTracks(); i++){
	
	TrkTrack *ts = 0;
	
	// get tracker tracks
	TrkTrack   *tp = TrkLevel2::GetTrack(i);                    //tracker
	CaloTrkVar *cp = 0;
	ToFTrkVar  *op = 0;

//	cout << "trk track n. "<<i << " "<<hex<< tp <<dec<< endl;
	// if track has an image, check image selection
	if(tp->HasImage()){
	    
	    TrkTrack   *ti = TrkLevel2::GetTrackImage(i);              //tracker (image)
	    CaloTrkVar *ci = 0;
	    ToFTrkVar  *oi = 0;
	    
//	    cout << "its image "<<i << " "<<hex<< ti <<dec<< endl;

	    //assign starting scores
	    Int_t tp_score = 0;  //main track sorted by the tracker
	    Int_t ti_score = 0;  //image track
	    
	    // ------------------------
	    // calorimeter check
	    // ------------------------
	    // check the Y spatial residual on the first calorimeter plane
	    // (cut on calorimeter variables from Emiliano)
	    if(
		use_CAL            &&
		npcfit[1] > 15     &&   //no. of fit planes on Y view
		varcfit[1] < 1000. &&  //fit variance on Y view
		true){

		cp = GetCaloStoredTrack(tp->GetSeqNo());
		ci = GetCaloStoredTrack(ti->GetSeqNo());
		
		Float_t resy_p = cp->tbar[0][1] - cbar[0][1]; if(resy_p < 0)resy_p= - resy_p;
		Float_t resy_i = ci->tbar[0][1] - cbar[0][1]; if(resy_i < 0)resy_i= - resy_i;
		
		if(resy_p <= resy_i) tp_score++;
		else                 ti_score++;

//		cout << "CALO "<<tp_score<<ti_score<<endl;

	    };
	    // ------------------------
	    // TOF check
	    // ------------------------	    
	    // check the number of hit pmts along the track 
	    // on S12 S21 and S32, where paddles are parallel to Y axis
	    if( use_TOF ){
		
		Int_t nphit_p =0;
		Int_t nphit_i =0;
		
		op = GetToFStoredTrack(tp->GetSeqNo());
		oi = GetToFStoredTrack(ti->GetSeqNo());
		
/*				cout << "track: npmtadc "<< op->npmtadc << endl;
				cout << "track: npmttdc "<< op->npmttdc << endl;
				cout << "image: npmtadc "<< oi->npmtadc << endl;
				cout << "image: npmttdc "<< oi->npmttdc << endl;*/
		
		for (Int_t ih=0; ih < op->npmtadc; ih++){
		    Int_t pl = GetPlaneIndex( (op->pmtadc).At(ih) );
		    if(pl == 1 || pl == 2 || pl == 5)nphit_p++;
		};
		
		for (Int_t ih=0; ih < oi->npmtadc; ih++){
		    Int_t pl = GetPlaneIndex( (oi->pmtadc).At(ih) );
		    if(pl == 1 || pl == 2 || pl == 5)nphit_i++;	
		};
		
		if(
		    use_TOF            &&
		    (nphit_p+nphit_i) !=0 &&	
		    true){
		    
		    if( nphit_p >= nphit_i) tp_score++;
		    else ti_score++;
		};
//		cout << "TOF "<<tp_score<<ti_score<<endl;
	    };
	    if(tp_score == ti_score) use_TRK = true;
	    // ------------------------
	    // tracker check
	    // ------------------------
	    // chi**2 difference is not always large enough to distinguish among 
	    // the real track and its image.
	    // Tracker check will be applied always when calorimeter and tof information is ambiguous.
	    if(use_TRK){
		if(      tp->chi2 > 0 && tp->chi2 < ti->chi2 ) tp_score++ ;
		else if( ti->chi2 > 0 && ti->chi2 < tp->chi2 ) ti_score++ ;
//		cout << "TRK "<<tp_score<<ti_score<<endl;
	    };
	    
	    // ------------------------
	    // the winner is....
	    // ------------------------	    
	    if      (tp_score > ti_score) ts = tp;//the track sorted by the tracker!!
	    else if (tp_score < ti_score) ts = ti;//its image!!
	    else {
		ts = tp;
//				cout << "Warning - track image ambiguity not solved" << endl;
//				cout << ts->GetNtot() << " "<< ts->chi2 << " " << npcfit[1] << " "<< nphit_p << endl;
	    };
	    
	}else{
	    ts = tp;
	};
	
//	cout <<" SortTracks() "<<i<<" -- "<<ts<<endl;
	sorted_tracks->Add(ts);//save the track in the sorted array
//	cout << "SortTracks:: sorted_tracks->Add(it) "<<i<<" "<<ts<<endl;
	
    };
    //Restore Object count 
    //To save space in the table keeping track of all referenced objects
    //we assume that our events do not address each other. We reset the 
    //object count to what it was at the beginning of the event.
    TProcessID::SetObjectCount(ObjectNumber);
    
};
//--------------------------------------
//
//
//--------------------------------------
/**
 * This method overrides TrkLevel2::GetTracks(), where sorting is done by decreasing number of fit points and increasing chi^2.
 * PamLevel2::GetTracks() keeps the same track order given by TrkLevel2::GetTracks(), but checks image selection by using calorimeter and ToF tracking information.
 */
TRefArray *PamLevel2::GetTracks(){

//  *-*-*-*-*-*-*-*-*-*-*-*-*
    SortTracks("+CAL+TOF");
//  *-*-*-*-*-*-*-*-*-*-*-*-*

    return sorted_tracks;
 };
//--------------------------------------
 //
 //
//--------------------------------------
/**
 * Retrieves the it-th Pamela "physical" track. 
 * It override TrkLevel2::GetTrack(int it). 
 * @param it Track number, ranging from 0 to GetNTracks().
 */
PamTrack *PamLevel2::GetTrack(int it){
	
//  *-*-*-*-*-*-*-*-*-*-*-*-*
    SortTracks("+CAL+TOF");
//  *-*-*-*-*-*-*-*-*-*-*-*-*
    
    PamTrack *track = 0;
    
    if( it >=0 && it < TrkLevel2::GetNTracks() && it<sorted_tracks->GetEntriesFast() ){	
	TrkTrack   *t = (TrkTrack*)sorted_tracks->At(it);
	track = GetPamTrackAlong(t);
    }else{
	cout << "PamLevel2::GetTrack(int) : tracker track SeqNo "<< it <<" does not exist (GetNTracks() = "<<TrkLevel2::GetNTracks()<<")"<<endl;
    };
    
    return track;
    
};

//--------------------------------------
//
//
//--------------------------------------
/**
 * Retrieves (if present) the image of the it-th Pamela "physical" track, sorted by the method PamLevel2::SortTracks(). 
 * @param it Track number, ranging from 0 to GetNTracks().
 */
PamTrack *PamLevel2::GetTrackImage(int it){

//  *-*-*-*-*-*-*-*-*-*-*-*-*
    SortTracks("+CAL+TOF");
//  *-*-*-*-*-*-*-*-*-*-*-*-*
    
    PamTrack *image = 0;
    
    if( it >=0 && it < TrkLevel2::GetNTracks() ){
	TrkTrack *temp = (TrkTrack*)sorted_tracks->At(it);
	if( temp->HasImage() ){
	    TrkTrack   *t = TrkLevel2::GetStoredTrack(temp->GetImageSeqNo());
	    image = GetPamTrackAlong(t);
	}else{
	    cout <<"PamLevel2::GetTrackImage(int) : Track SeqNo "<<it<<" does not have image"<<endl;
	};
    }else{
	cout << "PamLevel2::GetTrackImage(int) : Tracker track SeqNo "<< it <<" does not exist (GetNTracks() = "<<TrkLevel2::GetNTracks()<<")"<<endl;
    };
    
    return image;
}

//--------------------------------------
//
//
//--------------------------------------
/**
 * Get the Pamela detector trees in a single file and make them friends.
 * @param f TFile pointer
 * @param detlist String to select trees to be included
 * @return Pointer to a TTree
 */
TTree *PamLevel2::GetPamTree(TFile *f, TString detlist ){

//     if( !detlist.IsNull() ) SetWhichTrees(detlist);
//     else                    GetWhichTrees(f);
    SetWhichTrees(detlist);

    TTree *Trout =0;

    TString fname = f->GetName();
    if(!CheckLevel2File(fname))return NULL;

    UInt_t *found=0;    

    cout<< "GetPamTree(TFile*,TString): detector list --> ";
    if(TRK1)cout<<"TRK1 ";
    if(TRK2)cout<<"TRK2 ";
    if(TRKh)cout<<"TRKH ";
    if(CAL1)cout<<"CAL1 ";
    if(CAL2)cout<<"CAL2 ";
    if(TOF)cout<<"TOF ";
    if(TRG)cout<<"TRG ";
    if(AC)cout<<"AC ";
    if(ND)cout<<"ND ";
    if(S4)cout<<"S4 ";
    if(ORB)cout<<"ORB ";
    cout << endl;

    f->cd();

// Tracker
    TTree *T = (TTree*)f->Get("Tracker");
    if(T && (TRK2||TRK1||TRKh)) {
	if(TRK2)T->SetBranchAddress("TrkLevel2", GetPointerTo("TrkLevel2"));
	else    T->SetBranchStatus("TrkLevel2",0,found);
	if(TRK2)cout << "Tracker      : set branch address TrkLevel2"<<endl;
	if(TRK1)T->SetBranchAddress("TrkLevel1", GetPointerTo("TrkLevel1"));
	else    T->SetBranchStatus("TrkLevel1",0,found);
	if(TRK1)cout << "Tracker      : set branch address TrkLevel1"<<endl;
 	if(TRKh)T->SetBranchAddress("TrkHough", GetPointerTo("TrkHough"));
	else    T->SetBranchStatus("TrkHough",0,found);
 	if(TRKh)cout << "Tracker      : set branch address TrkHough"<<endl;
        if(!Trout)Trout=T;
        else Trout->AddFriend(T);
    }else{
        cout << "Tracker      : missing tree"<<endl;
    };
    // Calorimeter
    TTree *C = (TTree*)f->Get("Calorimeter");
    if(C && (CAL2||CAL1)) {
        if(CAL2)C->SetBranchAddress("CaloLevel2", GetPointerTo("CaloLevel2"));
	else    C->SetBranchStatus("CaloLevel2",0,found);
        if(CAL2)cout << "Calorimeter  : set branch address CaloLevel2"<<endl;
        if(CAL1)C->SetBranchAddress("CaloLevel1", GetPointerTo("CaloLevel1"));
	else    C->SetBranchStatus("CaloLevel1",0,found);
        if(CAL1)cout << "Calorimeter  : set branch address CaloLevel1"<<endl;
        if(!Trout)Trout=C;
        else Trout->AddFriend(C);
    }else{
        cout << "Calorimeter  : missing tree"<<endl;
    };

    // ToF    
    TTree *O = (TTree*)f->Get("ToF");
    if(O && TOF) {
        O->SetBranchAddress("ToFLevel2", GetPointerTo("ToFLevel2"));
        cout << "ToF          : set branch address ToFLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend(O);
    }else{
        cout << "ToF         : missing tree"<<endl;
    };
    // Trigger
    TTree *R = (TTree*)f->Get("Trigger");
    if(R && TRG) {
        R->SetBranchAddress("TrigLevel2", GetPointerTo("TrigLevel2"));
        cout << "Trigger      : set branch address TrigLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend(R);
    }else{
        cout << "Trigger      : missing tree"<<endl;
    };
    // S4
    TTree *S = (TTree*)f->Get("S4");
    if(S && S4) {
        S->SetBranchAddress("S4Level2", GetPointerTo("S4Level2"));
        cout << "S4           : set branch address S4Level2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend(S);
    }else{
        cout << "S4           : missing tree"<<endl;
    };
    // Neutron Detector
    TTree *N = (TTree*)f->Get("NeutronD");
    if(N && ND) {
        N->SetBranchAddress("NDLevel2", GetPointerTo("NDLevel2"));
        cout << "NeutronD     : set branch address NDLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend(N);
    }else{
        cout << "NeutronD     : missing tree"<<endl;
    };
    // Anticounters
    TTree *A = (TTree*)f->Get("Anticounter");
    if(A && AC) {
        A->SetBranchAddress("AcLevel2", GetPointerTo("AcLevel2"));
        cout << "Anticounter  : set branch address AcLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend(A);
    }else{
        cout << "Anticounter  : missing tree"<<endl;
    };
    // Orbital Info
    TTree *B = (TTree*)f->Get("OrbitalInfo");
    if(B && ORB) {
	B->SetBranchAddress("OrbitalInfo", GetPointerTo("OrbitalInfo"));
	cout << "OrbitalInfo  : set branch address OrbitalInfo"<<endl;
	if(!Trout)Trout=O;
	else Trout->AddFriend(B);
    }else{
	cout << "OrbitalInfo  : missing tree"<<endl;
    };
    
    cout<<endl<<" Number of entries: "<<Trout->GetEntries()<<endl<<endl;
    
    return Trout;
    
}
//--------------------------------------
//
//
//--------------------------------------
/**
 * Get list of Level2 files.
 * @param ddir Level2 data directory.
 * @param flisttxt Name of txt file containing file list.
 * @return Pointer to a TList of TSystemFiles
 * If no input file list is given , all the Level2 files inside the directory are processed.
 */
TList*  PamLevel2::GetListOfLevel2Files(TString ddir, TString flisttxt = ""){
	
    TString wdir = gSystem->WorkingDirectory();	
    
    if(ddir=="")ddir = wdir;
//	TSystemDirectory *datadir = new TSystemDirectory(gSystem->BaseName(ddir),ddir);
    cout << "Level2 data directory : "<<  ddir << endl;
    
    TList *contents  = new TList; // create output list
    contents->SetOwner();
    
    char *fullpath;
    
    // if no input file list is given:	
    if ( flisttxt != "" ){
	
	if( !gSystem->IsFileInIncludePath(flisttxt,&fullpath) ){ 
	    cout <<"File "<<flisttxt<<" not found"<<endl;
	    return 0;
	}	
//		flisttxt = gSystem->ConcatFileName(gSystem->DirName(flisttxt),gSystem->BaseName(flisttxt));
	flisttxt = fullpath;
	
	if( !gSystem->ChangeDirectory(ddir) ){
	    cout << "Cannot change directory : "<<ddir<<endl;
	    return 0; 
	}
	
	cout <<"Input file list : " << flisttxt <<endl;
	ifstream in; 
	in.open(flisttxt, ios::in); //open input file list
	int line=0;
	while (1) {
	    TString file;
	    in >> file;
	    if (!in.good()) break;
	    line++;
//	    cout <<"(1) " << file << endl;
	    if(file.IsNull()){
		cout << "-- list interrupted at line "<<line <<endl;
		break;
	    }
	    if(file.Contains("#"))file = file(0,file.First("#"));
//	    cout <<"(2) " << file << endl;
	    if( gSystem->IsFileInIncludePath(file,&fullpath) ){

		contents->Add(new TSystemFile(fullpath,gSystem->DirName(fullpath)));// add file to the list

	    }else{
//		if(file.Data()!="")cout << "File: "<<file<<" ---> missing "<< endl;
	    };
	};
	in.close();
	
    }else{
	
	cout << "No input file list given."<<endl;
	cout << "Check for existing root files."<<endl;
//		cout << "Warking directory: "<< gSystem->WorkingDirectory()<< endl;
	
	TSystemDirectory *datadir = new TSystemDirectory(gSystem->BaseName(ddir),ddir);
	TList *temp = datadir->GetListOfFiles();
//		temp->Print();
//		cout << "*************************************" << endl;
	
	TIter next(temp);
	TSystemFile *questo = 0;
	
	
	while ( (questo = (TSystemFile*) next()) ) {
	    TString name =  questo-> GetName();
	    if( name.EndsWith(".root") ){
		char *fullpath;
		gSystem->IsFileInIncludePath(name,&fullpath);
		contents->Add(new TSystemFile(fullpath,gSystem->DirName(fullpath)));
	    };
	}
	delete temp;
	delete datadir;
	
    };
    gSystem->ChangeDirectory(wdir); // back to the working directory
//	cout << endl << "Selected files:" << endl;
//	contents->Print();
    cout << contents->GetEntries()<<" files selected\n";
//	cout << endl;
//	cout << "Working directory: "<< gSystem->WorkingDirectory()<< endl;
    return contents;
};
//--------------------------------------
//
//
//--------------------------------------
/**
 * Get the Pamela detector chains from a list of files and make them friends.
 * @param fl Pointer to a TList of TSystemFiles 
 * @param detlist String to select trees to be included
 * @return Pointer to a TChain
 */
TChain *PamLevel2::GetPamTree(TList *fl, TString detlist ){
    
    TChain *Trout =0;

//    if( !detlist.IsNull() )SetWhichTrees(detlist);
    SetWhichTrees(detlist);
    
    cout<< "GetPamTree(TList*,TString): detector list --> ";
    if(TRK1)cout<<"TRK1 ";
    if(TRK2)cout<<"TRK2 ";
    if(TRKh)cout<<"TRKH ";
    if(CAL1)cout<<"CAL1 ";
    if(CAL2)cout<<"CAL2 ";
    if(TOF)cout<<"TOF ";
    if(TRG)cout<<"TRG ";
    if(AC)cout<<"AC ";
    if(ND)cout<<"ND ";
    if(S4)cout<<"S4 ";
    if(ORB)cout<<"ORB ";
    cout << endl;


    
    TChain *T = 0;	
    TChain *C = 0;
    TChain *O = 0;
    TChain *R = 0;
    TChain *S = 0;
    TChain *N = 0;
    TChain *A = 0;
    TChain *B = 0;
    
    if(TRK2||TRK1||TRKh) T = new TChain("Tracker");	
    if(CAL2||CAL1)       C = new TChain("Calorimeter");
    if(TOF)              O = new TChain("ToF");
    if(TRG)              R = new TChain("Trigger");
    if(S4)               S = new TChain("S4");
    if(ND)               N = new TChain("NeutronD");
    if(AC)               A = new TChain("Anticounter");
    if(ORB)              B = new TChain("OrbitalInfo");
    
    // loop over files and create chains	
    TIter next(fl);
    TSystemFile *questo = 0;
    while ( (questo = (TSystemFile*) next()) ) {
	TString name =  questo->GetName();
//	cout << "File: "<< name << endl;
	if( CheckLevel2File(name) ){
	    if(TRK2||TRK1||TRKh) T->Add(name);
	    if(CAL1||CAL2)       C->Add(name);
	    if(TOF)              O->Add(name);
	    if(TRG)              R->Add(name);
	    if(S4)               S->Add(name);
	    if(ND)               N->Add(name);
	    if(AC)               A->Add(name);
	    if(ORB)              B->Add(name);
	};
    }
    
    cout << "done chain \n";

    UInt_t *found=0;
// Tracker
    if(T && (TRK2||TRK1||TRKh)) {
	if(TRK2)T->SetBranchAddress("TrkLevel2", GetPointerTo("TrkLevel2"));
	else    T->SetBranchStatus("TrkLevel2",0,found);
	if(TRK2)cout << "Tracker      : set branch address TrkLevel2"<<endl;
	if(TRK1)T->SetBranchAddress("TrkLevel1", GetPointerTo("TrkLevel1"));
	else    T->SetBranchStatus("TrkLevel1",0,found);
	if(TRK1)cout << "Tracker      : set branch address TrkLevel1"<<endl;
 	if(TRKh)T->SetBranchAddress("TrkHough", GetPointerTo("TrkHough"));
	else    T->SetBranchStatus("TrkHough",0,found);
 	if(TRKh)cout << "Tracker      : set branch address TrkHough"<<endl;
        if(!Trout)Trout=T;
        else Trout->AddFriend("Tracker");
    }else{
        cout << "Tracker      : missing tree"<<endl;
    };
    // Calorimeter
    if(C && (CAL2||CAL1)) {
        if(CAL2)C->SetBranchAddress("CaloLevel2", GetPointerTo("CaloLevel2"));
	else    C->SetBranchStatus("CaloLevel2",0,found);
        if(CAL2)cout << "Calorimeter  : set branch address CaloLevel2"<<endl;
        if(CAL1)C->SetBranchAddress("CaloLevel1", GetPointerTo("CaloLevel1"));
	else    C->SetBranchStatus("CaloLevel1",0,found);
        if(CAL1)cout << "Calorimeter  : set branch address CaloLevel1"<<endl;
        if(!Trout)Trout=C;
        else Trout->AddFriend("Calorimeter");
    }else{
        cout << "Calorimeter  : missing tree"<<endl;
    };
    // ToF    
    if(O && TOF) {
        O->SetBranchAddress("ToFLevel2", GetPointerTo("ToFLevel2"));
        cout << "ToF          : set branch address ToFLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend("ToF");
    }else{
        cout << "ToF         : missing tree"<<endl;
    };
    // Trigger
    if(R && TRG) {
        R->SetBranchAddress("TrigLevel2", GetPointerTo("TrigLevel2"));
        cout << "Trigger      : set branch address TrigLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend("Trigger");
    }else{
        cout << "Trigger      : missing tree"<<endl;
    };
    // S4
    if(S && S4) {
        S->SetBranchAddress("S4Level2", GetPointerTo("S4Level2"));
        cout << "S4           : set branch address S4Level2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend("S4");
    }else{
        cout << "S4           : missing tree"<<endl;
    };
    // Neutron Detector
    if(N && ND) {
        N->SetBranchAddress("NDLevel2", GetPointerTo("NDLevel2"));
        cout << "NeutronD     : set branch address NDLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend("NeutronD");
    }else{
        cout << "NeutronD     : missing tree"<<endl;
    };
    // Anticounters
    if(A && AC) {
        A->SetBranchAddress("AcLevel2", GetPointerTo("AcLevel2"));
        cout << "Anticounter  : set branch address AcLevel2"<<endl;
        if(!Trout)Trout=O;
        else Trout->AddFriend("Anticounter");
    }else{
        cout << "Anticounter  : missing tree"<<endl;
    };
    // Orbital Info
    if(B && ORB) {
	B->SetBranchAddress("OrbitalInfo", GetPointerTo("OrbitalInfo"));
	cout << "OrbitalInfo  : set branch address OrbitalInfo"<<endl;
	if(!Trout)Trout=O;
	else Trout->AddFriend("OrbitalInfo");
    }else{
	cout << "OrbitalInfo  : missing tree"<<endl;
    };
    
//    cout<<endl<<" Number of entries: "<<Trout->GetEntries()<<endl<<endl;

    
//     // Tracker
//     if(TRK2||TRK1||TRKh){
// 	if(!Tout)Tout=T;
// 	else Tout->AddFriend("Tracker");
//     }
//     // Calorimeter
//     if(CAL2||CAL1) {		
// 	if(!Tout)Tout=C;
// 	else Tout->AddFriend("Calorimeter");
//     };
//     // ToF    
//     if(TOF) {
// 	if(!Tout)Tout=O;
// 	else Tout->AddFriend("ToF");
//     };
//     // Trigger
//     if(TRG) {
// 	if(!Tout)Tout=R;
// 	else Tout->AddFriend("Trigger");
//     };
//     // S4
//     if(S4) {
// 	if(!Tout)Tout=S;
// 	else Tout->AddFriend("S4");
//     };
//     // Neutron Detector
//     if(ND) {
// 	if(!Tout)Tout=N;
// 	else Tout->AddFriend("NeutronD");
//     };
//     // Anticounters
//     if(AC) {
// 	if(!Tout)Tout=A;
// 	else Tout->AddFriend("Anticounter");
//     };
//     // OrbitalInfo
//     if(ORB) {
// 	if(!Tout)Tout=B;
// 	else Tout->AddFriend("OrbitalInfo");
//     };
    
//    cout<<endl<<" Number of entries: "<<Tout->GetEntries()<<endl<<endl;
    
//    if( Tout->GetEntries() )PamLevel2::SetBranchAddress( Tout );
    
    return Trout;
}



//--------------------------------------
//
//
//--------------------------------------
/**
 * Set branch addresses for Pamela friend trees
 */
void PamLevel2::SetBranchAddress(TTree *t){

    TRK2    = TRK2 & t->GetBranchStatus("TrkLevel2");
    TRK1    = TRK1 & t->GetBranchStatus("TrkLevel1");
    TRKh    = TRKh & t->GetBranchStatus("TrkHough");
    CAL2    = CAL2 & t->GetBranchStatus("CaloLevel2");
    CAL1    = CAL1 & t->GetBranchStatus("CaloLevel1");
    TOF    = TOF & t->GetBranchStatus("ToFLevel2");
    TRG    = TRG & t->GetBranchStatus("TrigLevel2");
    S4     = S4  & t->GetBranchStatus("S4Level2");
    ND     = ND  & t->GetBranchStatus("NDLevel2");
    AC     = AC  & t->GetBranchStatus("AcLevel2");
    ORB    = ORB & t->GetBranchStatus("OrbitalInfo");


    // Tracker
    if(TRK1) {
	t->SetBranchAddress("TrkLevel1", GetPointerTo("TrkLevel2"));
	cout << "Tracker      : set branch address TrkLevel1"<<endl;
    };
    if(TRK2) {
	t->SetBranchAddress("TrkLevel2", GetPointerTo("TrkLevel1"));
	cout << "Tracker      : set branch address TrkLevel2"<<endl;
    };
    if(TRKh) {
	t->SetBranchAddress("TrkHough", GetPointerTo("TrkHough"));
	cout << "Tracker      : set branch address TrkHough"<<endl;
    };
    
    // Calorimeter
    if(CAL1) {
	t->SetBranchAddress("CaloLevel1", GetPointerTo("CaloLevel1"));
	cout << "Calorimeter  : set branch address CaloLevel1"<<endl;
    };
    if(CAL2) {
	t->SetBranchAddress("CaloLevel2", GetPointerTo("CaloLevel2"));
	cout << "Calorimeter  : set branch address CaloLevel2"<<endl;
    };
    
    // ToF    
    if(TOF) {
	t->SetBranchAddress("ToFLevel2", GetPointerTo("ToFLevel2"));
	cout << "ToF          : set branch address ToFLevel2"<<endl;
    };
    // Trigger
    if(TRG) {
	t->SetBranchAddress("TrigLevel2", GetPointerTo("TrigLevel2"));
	cout << "Trigger      : set branch address TrigLevel2"<<endl;
    };
    // S4
    if(S4) {
	t->SetBranchAddress("S4Level2", GetPointerTo("S4Level2"));
	cout << "S4           : set branch address S4Level2"<<endl;
    };
    // Neutron Detector
    if(ND) {
	t->SetBranchAddress("NDLevel2", GetPointerTo("NDLevel2"));
	cout << "NeutronD     : set branch address NDLevel2"<<endl;
    };
    // Anticounters
    if(AC) {
	t->SetBranchAddress("AcLevel2", GetPointerTo("AcLevel2"));
	cout << "Anticounter  : set branch address AcLevel2"<<endl;
    };
    // OrbitalInfo
    if(ORB) {
	t->SetBranchAddress("OrbitalInfo", GetPointerTo("OrbitalInfo"));
	cout << "OrbitalInfo  : set branch address OrbitalInfo"<<endl;
    };
    
}
/**
 * Set branch addresses for Pamela friend trees
 */
void PamLevel2::SetBranchAddress(TChain *t){

    TRK2    = TRK2 & t->GetBranchStatus("TrkLevel2");
    TRK1    = TRK1 & t->GetBranchStatus("TrkLevel1");
    TRKh    = TRKh & t->GetBranchStatus("TrkHough");
    CAL1    = CAL1 & t->GetBranchStatus("CaloLevel1");
    CAL2    = CAL2 & t->GetBranchStatus("CaloLevel2");
    TOF    = TOF & t->GetBranchStatus("ToFLevel2");
    TRG    = TRG & t->GetBranchStatus("TrigLevel2");
    S4     = S4  & t->GetBranchStatus("S4Level2");
    ND     = ND  & t->GetBranchStatus("NDLevel2");
    AC     = AC  & t->GetBranchStatus("AcLevel2");
    ORB    = ORB & t->GetBranchStatus("OrbitalInfo");

    // Tracker
     if(TRK2) {
	t->SetBranchAddress("TrkLevel2", GetPointerTo("TrkLevel2"));
	cout << "Tracker      : set branch address TrkLevel2"<<endl;
    };
    if(TRK1) {
	t->SetBranchAddress("TrkLevel1", GetPointerTo("TrkLevel1"));
	cout << "Tracker      : set branch address TrkLevel1"<<endl;
    };
     if(TRKh) {
 	t->SetBranchAddress("TrkHough", GetPointerTo("TrkHough"));
 	cout << "Tracker      : set branch address TrkHough"<<endl;
     };
    
    // Calorimeter
    if(CAL2) {
	t->SetBranchAddress("CaloLevel2", GetPointerTo("CaloLevel2"));
	cout << "Calorimeter  : set branch address CaloLevel2"<<endl;
    };   
    if(CAL1) {
	t->SetBranchAddress("CaloLevel1", GetPointerTo("CaloLevel1"));
	cout << "Calorimeter  : set branch address CaloLevel1"<<endl;
    };
    
    // ToF    
    if(TOF) {
	t->SetBranchAddress("ToFLevel2", GetPointerTo("ToFLevel2"));
	cout << "ToF          : set branch address ToFLevel2"<<endl;
    };
    // Trigger
    if(TRG) {
	t->SetBranchAddress("TrigLevel2", GetPointerTo("TrigLevel2"));
	cout << "Trigger      : set branch address TrigLevel2"<<endl;
    };
    // S4
    if(S4) {
	t->SetBranchAddress("S4Level2", GetPointerTo("S4Level2"));
	cout << "S4           : set branch address S4Level2"<<endl;
    };
    // Neutron Detector
    if(ND) {
	t->SetBranchAddress("NDLevel2", GetPointerTo("NDLevel2"));
	cout << "NeutronD     : set branch address NDLevel2"<<endl;
    };
    // Anticounters
    if(AC) {
	t->SetBranchAddress("AcLevel2", GetPointerTo("AcLevel2"));
	cout << "Anticounter  : set branch address AcLevel2"<<endl;
    };
    // OrbitalInfo
    if(ORB) {
	t->SetBranchAddress("OrbitalInfo", GetPointerTo("OrbitalInfo"));
	cout << "OrbitalInfo  : set branch address OrbitalInfo"<<endl;
    };
    
}


//--------------------------------------
//
//
//--------------------------------------
/**
 * Get the Run tree chain from a list of files.
 * @param fl Pointer to a TList of TSystemFiles 
 * @return Pointer to a TChain
 */
TChain *PamLevel2::GetRunTree(TList *fl){
	
    TChain *R = new TChain("Run");
    
    // loop over files and create chains	
    TIter next(fl);
    TSystemFile *questo = 0;
    while ( (questo = (TSystemFile*) next()) ) {
	TString name =  questo->GetName();
//		cout << "File: "<< name << endl;
	if( CheckLevel2File(name) ){
	    R->Add(name);
	};
    }
    
    R->SetBranchAddress("RunInfo", GetPointerTo("RunInfo"));
    cout << "Run         : set branch address RunInfo"<<endl;
    
    return R;
    
}
//--------------------------------------
//
//
//--------------------------------------
/**
 * Get the Run tree  from a file.
 * @param f Pointer to a TFile 
 * @return Pointer to a TTree
 */
TTree *PamLevel2::GetRunTree(TFile *f){


    TTree *R = (TTree*)f->Get("Run");
    
    R->SetBranchAddress("RunInfo", GetPointerTo("RunInfo"));
    cout << "Run         : set branch address RunInfo"<<endl;
    
    return R;
    
}
//--------------------------------------
//
//
//--------------------------------------
/**
 * Set which trees shoul be analysed
 * @param detlist TString containing the sequence of trees required 
*/
void PamLevel2::SetWhichTrees(TString detlist){
	
    if(detlist.IsNull() || detlist.Contains("+ALL", TString::kIgnoreCase)){
	CAL1 = true;
	CAL2 = true;
	TRK2 = true;
	TRK1 = false;
	TRKh = false;
	TRG = true;
	TOF = true;
	S4  = true;
	ND  = true;
	AC  = true;
	ORB = true;
    }else if( detlist.Contains("-ALL", TString::kIgnoreCase) ){
	CAL1 = false;
	CAL2 = false;
	TRK2 = false;
	TRK1 = false;
	TRKh = false;
	TRG = false;
	TOF = false;
	S4  = false;
	ND  = false;
	AC  = false;
	ORB = false;
    };
    
//  -------------------------------------------------------------------------
    if( detlist.Contains("CAL1", TString::kIgnoreCase) ){
	if ( detlist.Contains("-CAL1", TString::kIgnoreCase) )CAL1=false;
	if ( detlist.Contains("+CAL1", TString::kIgnoreCase) )CAL1=true;
    };
	
    if( detlist.Contains("CAL2", TString::kIgnoreCase)){
	if ( detlist.Contains("-CAL2", TString::kIgnoreCase) )CAL2=false;
	if ( detlist.Contains("+CAL2", TString::kIgnoreCase) )CAL2=true;
    };
	
    if( detlist.Contains("+CAL", TString::kIgnoreCase) && !CAL1 && !CAL2 )CAL2=true;
    if( detlist.Contains("-CAL", TString::kIgnoreCase) && CAL1 && CAL2 ){
	CAL2=false;
	CAL1=false;
    }
//  -------------------------------------------------------------------------
    if( detlist.Contains("TRK1", TString::kIgnoreCase) ){
	if ( detlist.Contains("-TRK1", TString::kIgnoreCase) )TRK1=false;
	if ( detlist.Contains("+TRK1", TString::kIgnoreCase) )TRK1=true;
    };
	
    if( detlist.Contains("TRK2", TString::kIgnoreCase)){
	if ( detlist.Contains("-TRK2", TString::kIgnoreCase) )TRK2=false;
	if ( detlist.Contains("+TRK2", TString::kIgnoreCase) )TRK2=true;
    };

    if( detlist.Contains("TRKh", TString::kIgnoreCase)){
	if ( detlist.Contains("-TRKh", TString::kIgnoreCase) )TRKh=false;
	if ( detlist.Contains("+TRKh", TString::kIgnoreCase) )TRKh=true;
    };
	
    if( detlist.Contains("+TRK", TString::kIgnoreCase) && !TRK1 && !TRK2 && !TRKh )TRK2=true;
    if( detlist.Contains("-TRK", TString::kIgnoreCase) && TRK1 && TRK2 && TRKh){
	TRK2=false;
	TRK1=false;
	TRKh=false;
    }
//  -------------------------------------------------------------------------
    
    if( detlist.Contains("-TRG", TString::kIgnoreCase) )TRG = false;
    else if( detlist.Contains("+TRG", TString::kIgnoreCase) )TRG = true;
    
    if( detlist.Contains("-TOF", TString::kIgnoreCase) )TOF = false;
    else if( detlist.Contains("+TOF", TString::kIgnoreCase) )TOF = true;
    
    if( detlist.Contains("-S4",  TString::kIgnoreCase) )S4  = false;
    else if( detlist.Contains("+S4",  TString::kIgnoreCase) )S4  = true;
    
    if( detlist.Contains("-ND",  TString::kIgnoreCase) )ND  = false;
    else if( detlist.Contains("+ND",  TString::kIgnoreCase) )ND  = true;
    
    if( detlist.Contains("-AC",  TString::kIgnoreCase) )AC  = false;
    else if( detlist.Contains("+AC",  TString::kIgnoreCase) )AC  = true;
    
    if( detlist.Contains("-ORB", TString::kIgnoreCase) )ORB = false;
    else if( detlist.Contains("+ORB", TString::kIgnoreCase) )ORB = true;

//     cout<< "Set detector list --> ";
//     if(TRK1)cout<<"TRK1 ";
//     if(TRK2)cout<<"TRK2 ";
//     if(TRKh)cout<<"TRKH ";
//     if(CAL1)cout<<"CAL1 ";
//     if(CAL2)cout<<"CAL2 ";
//     if(TOF)cout<<"TOF ";
//     if(TRG)cout<<"TRG ";
//     if(AC)cout<<"AC ";
//     if(ND)cout<<"ND ";
//     if(S4)cout<<"S4 ";
//     if(ORB)cout<<"ORB ";
//     cout << endl;
    
};


/**
 * Set tree/branch detector flags from the content of a tree
 */
void  PamLevel2::GetWhichTrees(TFile* f){
	

    // -----------
    // reset flags
    // -----------
    CAL1   = false;     
    CAL2   = false;     
    TRK2   = false;     
    TRK1   = false;     
    TRKh   = false;     
    TRG    = false;     
    TOF    = false;     
    S4     = false;    
    ND     = false;    
    AC     = false;     
    ORB    = false;    
    
    RUN    = false;
        
    cout << "Checking file: "<<f->GetName()<<endl;
    if( !f || f->IsZombie() ){
	cout << "File: "<< f->GetName() <<" Non valid root file"<< endl; 
	return;
    }

    TList *lk = f->GetListOfKeys();
    TIter next(lk);
    TKey *key =0;

    Int_t nev = 0;

    while( (key = (TKey*)next()) ){
	
	if( !strcmp(key->GetName(),"Run"        ) )RUN = true;

	//=========================================================	
       	if( !strcmp(key->GetName(),"Trigger"    ) ){ 
	    TRG = true;
	    Int_t nevt = ((TTree*)f->Get("Trigger"))->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<" Trigger tree has "<<nevt<<" events instead of "<<nev<< endl; 
		TRG = false;
	    }else nev=nevt;
	}
	//=========================================================	
	if( !strcmp(key->GetName(),"ToF"        ) ){
	    TOF = true;
	    Int_t nevt = ((TTree*)f->Get("ToF"))->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<"     ToF tree has "<<nevt<<" events instead of "<<nev<< endl; 
		TOF = false;
	    }else nev=nevt;
	}
	//=========================================================  
	if( !strcmp(key->GetName(),"S4"         ) ){
	    S4 = true;
	    Int_t nevt = ((TTree*)f->Get("S4"))->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<"      S4 tree has "<<nevt<<" events instead of "<<nev<< endl; 
		S4 = false;
	    }else nev=nevt;
	}
	//=========================================================  

	if( !strcmp(key->GetName(),"NeutronD"   ) ){
	    ND = true;
	    Int_t nevt = ((TTree*)f->Get("NeutronD"))->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<"NeutronD tree has "<<nevt<<" events instead of "<<nev<< endl; 
		ND =false;
	    }else nev=nevt;
	}	
        //=========================================================  
	if( !strcmp(key->GetName(),"Anticounter") ){
	    AC = true;
	    Int_t nevt = ((TTree*)f->Get("Anticounter"))->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<" Anticounter tree has "<<nevt<<" events instead of "<<nev<< endl; 
		AC =false;
	    }else nev=nevt;
	}
	//=========================================================  
	if( !strcmp(key->GetName(),"OrbitalInfo") ){
	    ORB = true;
	    Int_t nevt = ((TTree*)f->Get("OrbitalInfo"))->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<" OrbitalInfo tree has "<<nevt<<" events instead of "<<nev<< endl; 
		ORB = false;
	    }else nev=nevt;
	}
	//=========================================================  
	if( !strcmp(key->GetName(),"Tracker"    ) ){
	    TTree *T = (TTree*)f->Get("Tracker");
	    for(Int_t i=0; i<T->GetListOfBranches()->GetEntries(); i++){
		TString name = T->GetListOfBranches()->At(i)->GetName();
		if( !name.CompareTo("TrkLevel1") )TRK1=true;
		if( !name.CompareTo("TrkLevel2") )TRK2=true;
		if( !name.CompareTo("TrkHough") )TRKh=true;
	    };	    
	    Int_t nevt = T->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<" Tracker tree has "<<nevt<<" events instead of "<<nev<< endl; 
		TRK1 = false;
		TRK2 = false;
		TRKh = false;
	    }else nev=nevt;
	    T->Delete();
	};
	//=========================================================  
	if( !strcmp(key->GetName(),"Calorimeter"    ) ){
	    TTree *T = (TTree*)f->Get("Calorimeter");
	    for(Int_t i=0; i<T->GetListOfBranches()->GetEntries(); i++){
		TString name = T->GetListOfBranches()->At(i)->GetName();
		if( !name.CompareTo("CaloLevel1") )CAL1=true;
		if( !name.CompareTo("CaloLevel2") )CAL2=true;
	    };	   
	    Int_t nevt = T->GetEntries();
	    if( nev && nevt!=nev){
		cout << "File: "<< f->GetName() <<"  Calorimeter tree has "<<nevt<<" events instead of "<<nev<< endl; 
		CAL1 = false;
		CAL2 = false;
	    }else nev=nevt;
	    T->Delete();
	};	

    };
    
    lk->Delete();

//     cout<< "Get detector list from input file --> ";
//     if(TRK1)cout<<"TRK1 ";
//     if(TRK2)cout<<"TRK2 ";
//     if(TRKh)cout<<"TRKH ";
//     if(CAL1)cout<<"CAL1 ";
//     if(CAL2)cout<<"CAL2 ";
//     if(TOF)cout<<"TOF ";
//     if(TRG)cout<<"TRG ";
//     if(AC)cout<<"AC ";
//     if(ND)cout<<"ND ";
//     if(S4)cout<<"S4 ";
//     if(ORB)cout<<"ORB ";
//     cout << endl;
       
    return ;
	
};


//--------------------------------------
//
//
//--------------------------------------
/**
 * Check if a file contains selected Pamela Level2 trees.
 * @param name File name
 * @return true if the file is ok.
 */
Bool_t  PamLevel2::CheckLevel2File(TString name){
	
    Bool_t CAL1__ok   = false;     
    Bool_t CAL2__ok   = false;     
    Bool_t TRK2__ok   = false;     
    Bool_t TRK1__ok   = false;     
    Bool_t TRKh__ok   = false;     
    Bool_t TRG__ok    = false;     
    Bool_t TOF__ok    = false;     
    Bool_t S4__ok     = false;    
    Bool_t ND__ok     = false;    
    Bool_t AC__ok     = false;     
    Bool_t ORB__ok    = false;    
    
    Bool_t RUN__ok    = false;
    
    

    cout << "Checking file: "<<name<<endl;
    TFile *f = new TFile(name.Data());
    if( !f || f->IsZombie() ){
	cout << "File: "<< f->GetName() <<" discarded ---- Non valid root file"<< endl; return false;
    }
//    cout << "Get list of keys: "<<f<<endl;
    TList *lk = f->GetListOfKeys();
//    lk->Print();
    TIter next(lk);
    TKey *key =0;

    Int_t nev = 0;

    while( (key = (TKey*)next()) ){
	
// 	cout << key->GetName() << ""<<key->GetClassName()<<endl;
//		cout << " Get tree: " << f->Get(key->GetName())<<endl;
// 	nev_previous = nev;
// 	cout << " n.entries  "<< nev <<endl;
// 	if( key->GetClassName()=="TTree" && nev_previous && nev != nev_previous ){
// 	    nev = ((TTree*)f->Get(key->GetName()))->GetEntries();
// 	    cout << "File: "<< f->GetName() <<" discarded ---- "<< key->GetName() << " tree: n.entries does not match "<<nev<<" "<<nev_previous<< endl;
// 	    return false;
// 	};

	if( !strcmp(key->GetName(),"Run"        ) )RUN__ok = true;	

	//=========================================================	
       	if( !strcmp(key->GetName(),"Trigger"    ) ){ 
	    TRG__ok = true;
	    if(TRG){
		Int_t nevt = ((TTree*)f->Get("Trigger"))->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- Trigger tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	}
	//=========================================================	
	if( !strcmp(key->GetName(),"ToF"        ) ){
	    TOF__ok = true;
	    if(TOF){
		Int_t nevt = ((TTree*)f->Get("ToF"))->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- ToF tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	}
	//=========================================================  
	if( !strcmp(key->GetName(),"S4"         ) ){
	    S4__ok = true;
	    if(S4){
		Int_t nevt = ((TTree*)f->Get("S4"))->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- S4 tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	}
	//=========================================================  

	if( !strcmp(key->GetName(),"NeutronD"   ) ){
	    ND__ok = true;
	    if(ND){
		Int_t nevt = ((TTree*)f->Get("NeutronD"))->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- NeutronD tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	}	
        //=========================================================  
	if( !strcmp(key->GetName(),"Anticounter") ){
	    AC__ok = true;
	    if(AC){
		Int_t nevt = ((TTree*)f->Get("Anticounter"))->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- Anticounter tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	}
	//=========================================================  
	if( !strcmp(key->GetName(),"OrbitalInfo") ){
	    ORB__ok = true;
	    if(ORB){
		Int_t nevt = ((TTree*)f->Get("OrbitalInfo"))->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- OrbitalInfo tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	}
	//=========================================================  
	if( !strcmp(key->GetName(),"Tracker"    ) ){
	    TTree *T = (TTree*)f->Get("Tracker");
	    if(TRK1||TRK2||TRKh){
		Int_t nevt = T->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- Tracker tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	    for(Int_t i=0; i<T->GetListOfBranches()->GetEntries(); i++){
		TString name = T->GetListOfBranches()->At(i)->GetName();
		if( !name.CompareTo("TrkLevel1") )TRK1__ok=true;
		if( !name.CompareTo("TrkLevel2") )TRK2__ok=true;
		if( !name.CompareTo("TrkHough") )TRKh__ok=true;
	    };	    
	    T->Delete();
	};
	//=========================================================  
	if( !strcmp(key->GetName(),"Calorimeter"    ) ){
	    TTree *T = (TTree*)f->Get("Calorimeter");
	    if(CAL1||CAL2){
		Int_t nevt = T->GetEntries();
		if( nev && nevt!=nev){
		    cout << "File: "<< f->GetName() <<" discarded ---- Calorimeter tree has "<<nevt<<" events instead of "<<nev<< endl; 
		    return false;
		}
		nev=nevt;
	    }
	    for(Int_t i=0; i<T->GetListOfBranches()->GetEntries(); i++){
		TString name = T->GetListOfBranches()->At(i)->GetName();
		if( !name.CompareTo("CaloLevel1") )CAL1__ok=true;
		if( !name.CompareTo("CaloLevel2") )CAL2__ok=true;
	    };	   
	    T->Delete();
	};	

    };
    
//     cout<< "CheckLevel2File(TString): detector list --> ";
//     if(TRK1__ok)cout<<"TRK1 ";
//     if(TRK2__ok)cout<<"TRK2 ";
//     if(TRKh__ok)cout<<"TRKH ";
//     if(CAL1__ok)cout<<"CAL1 ";
//     if(CAL2__ok)cout<<"CAL2 ";
//     if(TOF__ok)cout<<"TOF ";
//     if(TRG__ok)cout<<"TRG ";
//     if(AC__ok)cout<<"AC ";
//     if(ND__ok)cout<<"ND ";
//     if(S4__ok)cout<<"S4 ";
//     if(ORB__ok)cout<<"ORB ";
//     cout << endl;


    if(!RUN__ok) {
	cout << "File: "<< f->GetName() <<" *WARNING* ---- Missing RunInfo tree"<< endl; 
//	return false;
    };

    if(CAL1 && !CAL1__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing CaloLevel1 branch"<< endl; 
	return false;
    };
    if(CAL2 && !CAL2__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing CaloLevel2 branch"<< endl; 
	return false;
    };
    if(TRK2 && !TRK2__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing TrkLevel2 branch"<< endl; 
	return false;
    };
    if(TRK1 && !TRK1__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing TrkLevel1 branch"<< endl; 
	return false;
    };
    if(TRKh && !TRKh__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing TrkHough branch"<< endl;
	return false;
    };
    if(ORB && !ORB__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing ORB tree"<< endl; 
	return false;
    };
    if(AC && !AC__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing AC tree"<< endl; 
	return false;
    };
    if(S4 && !S4__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing S4 tree"<< endl; 
	return false;
    };
    if(TOF && !TOF__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing ToF tree"<< endl; 
	return false;
    };

    if(ND && !ND__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing ND tree"<< endl;
	return false;
    };
    if(TRG && !TRG__ok){
	cout << "File: "<< f->GetName() <<" discarded ---- Missing Trigger tree"<< endl; 
	return false;
    };
    

    lk->Delete();
    f->Close();

//     cout<< "CheckLevel2File(TString): detector list --> ";
//     if(TRK1)cout<<"TRK1 ";
//     if(TRK2)cout<<"TRK2 ";
//     if(TRKh)cout<<"TRKH ";
//     if(CAL1)cout<<"CAL1 ";
//     if(CAL2)cout<<"CAL2 ";
//     if(TOF)cout<<"TOF ";
//     if(TRG)cout<<"TRG ";
//     if(AC)cout<<"AC ";
//     if(ND)cout<<"ND ";
//     if(S4)cout<<"S4 ";
//     if(ORB)cout<<"ORB ";
//     cout << endl;
       
    return true;
	
};


/**
 * Create clone-trees
 */
void PamLevel2::CreateCloneTrees0( TChain *fChain, TFile *ofile ){

    cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;
    cout << "Create clones of PAMELA trees "<<endl;
    
    Int_t i=0;
    tree_clone[i] = fChain->GetTree()->CloneTree(0);
    TString name =  tree_clone[i]->GetName();
    name.Append("_clone");
//    tree_clone[i]->SetName(name.Data());
    cout << tree_clone[i]->GetName() <<endl;
    i++;

    TList *li = fChain->GetListOfFriends();
    TIter next(li);
    TFriendElement* T_friend=0;
    ofile->cd();
    while( (T_friend = (TFriendElement*)next()) ){
// 	cout<<T_friend->IsA()->GetName()<<" "<<T_friend->GetName()<<hex << T_friend->GetTree() << dec<<endl;
//	cout<<T_friend->GetTree()->GetName()<< endl;
	tree_clone[i] = T_friend->GetTree()->CloneTree(0);
	tree_clone[i]->SetAutoSave(1000000);
	name =  tree_clone[i]->GetName();
	name.Append("_clone");
//	tree_clone[i]->SetName(name.Data());
	cout << tree_clone[i]->GetName() << endl;
	i++;
    }

    cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;

}

/**
 * Create clone-trees
 */
void PamLevel2::CreateCloneTrees(TFile *ofile){

    ofile->cd();

    cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;
    cout << "Create new PAMELA trees "<<endl;
        
    Int_t i=0;

    if(TRK1||TRK2||TRKh){
	tree_clone[i] = new TTree("Tracker","PAMELA tracker level2 data ");	
	if(TRK1) {
	    tree_clone[i]->Branch("TrkLevel1","TrkLevel1", GetPointerTo("TrkLevel1"));
	    tree_clone[i]->BranchRef();
	    cout << "Tracker      : branch TrkLevel1"<<endl;
	};
	if(TRK2) {
	    tree_clone[i]->Branch("TrkLevel2", "TrkLevel2",GetPointerTo("TrkLevel2"));
	    cout << "Tracker      : branch TrkLevel2"<<endl;	    
	};
	if(TRKh) {
	    tree_clone[i]->Branch("TrkHough","TrkHough", GetPointerTo("TrkHough"));
	    cout << "Tracker      : branch TrkHough"<<endl;
	};
	i++;
    }

    // Calorimeter
    if(CAL1||CAL2){
	tree_clone[i] = new TTree("Calorimeter","PAMELA calorimeter level2 data ");	
	if(CAL1) {
	    tree_clone[i]->Branch("CaloLevel1", "CaloLevel1", GetPointerTo("CaloLevel1"));
	    cout << "Calorimeter  : branch CaloLevel1"<<endl;
	};
	if(CAL2) {
	    tree_clone[i]->Branch("CaloLevel2","CaloLevel2", GetPointerTo("CaloLevel2"));
	    cout << "Calorimeter  : branch CaloLevel2"<<endl;
	};
	i++;
    }    

    // ToF    
    if(TOF) {
	tree_clone[i] = new TTree("ToF","PAMELA ToF level2 data ");	
	tree_clone[i]->Branch("ToFLevel2","ToFLevel2", GetPointerTo("ToFLevel2"));
	cout << "ToF          : branch ToFLevel2"<<endl;
	i++;
    };
    // Trigger
    if(TRG) {
	tree_clone[i] = new TTree("Trigger","PAMELA trigger level2 data ");	
	tree_clone[i]->Branch("TrigLevel2","TrigLevel2", GetPointerTo("TrigLevel2"));
	cout << "Trigger      : branch TrigLevel2"<<endl;
	i++;
    };
    // S4
    if(S4) {
	tree_clone[i] = new TTree("S4","PAMELA S4 level2 data ");	
	tree_clone[i]->Branch("S4Level2","S4Level2", GetPointerTo("S4Level2"));
	cout << "S4           : branch S4Level2"<<endl;
	i++;
    };
    // Neutron Detector
    if(ND) {
	tree_clone[i] = new TTree("NeutronD","PAMELA neutron detector level2 data ");	
	tree_clone[i]->Branch("NDLevel2","NDLevel2", GetPointerTo("NDLevel2"));
	cout << "NeutronD     : branch NDLevel2"<<endl;
	i++;
    };
    // Anticounters
    if(AC) {
	tree_clone[i] = new TTree("Anticounter","PAMELA anticounter detector level2 data ");	
	tree_clone[i]->Branch("AcLevel2","AcLevel2", GetPointerTo("AcLevel2"));
	cout << "Anticounter  : branch AcLevel2"<<endl;
	i++;
    };
    // OrbitalInfo
    if(ORB) {
	tree_clone[i] = new TTree("OrbitalInfo","PAMELA oribital info  ");	
	tree_clone[i]->Branch("OrbitalInfo","OrbitalInfo", GetPointerTo("OrbitalInfo"));
	cout << "OrbitalInfo  : branch OrbitalInfo"<<endl;
	i++;
    };
    cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;

}

// /**
//  * Create clone-trees
//  */
// void PamLevel2::CreateCloneTrees(){

//     cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;
//     cout << "Create new PAMELA trees "<<endl;
        
//     if(TRK1||TRK2||TRKh){
// 	T_clone = new TTree("Tracker_clone","PAMELA tracker level2 data ");	
// 	if(TRK1) {
// 	    trk1_clone = new TrkLevel1();
// 	    T_clone->Branch("TrkLevel1","TrkLevel1", &trk1_clone);
// 	    T_clone->BranchRef();
// 	    cout << "Tracker      : branch TrkLevel1"<<endl;
// 	};
// 	if(TRK2) {
// 	    trk2_clone = new TrkLevel2();
// 	    T_clone->Branch("TrkLevel2", "TrkLevel2",&trk2_clone);
// 	    cout << "Tracker      : branch TrkLevel2"<<endl;	    
// 	};
// 	if(TRKh) {
// 	    trkh_clone = new TrkHough();
// 	    T_clone->Branch("TrkHough","TrkHough", &trkh_clone);
// 	    cout << "Tracker      : branch TrkHough"<<endl;
// 	};
// 	cout << hex << T_clone << dec << endl;
//     }

//     // Calorimeter
//     if(CAL1||CAL2){
// 	C_clone = new TTree("Calorimeter_clone","PAMELA calorimeter level2 data ");	
// 	if(CAL1) {
// 	    calo1_clone = new CaloLevel1();
// 	    C_clone->Branch("CaloLevel1", "CaloLevel1", &calo1_clone);
// 	    cout << "Calorimeter  : branch CaloLevel1"<<endl;
// 	};
// 	if(CAL2) {
// 	    calo2_clone = new CaloLevel2();
// 	    C_clone->Branch("CaloLevel2","CaloLevel2", &calo2_clone);
// 	    cout << "Calorimeter  : branch CaloLevel2"<<endl;
// 	};
//     }    

//     // ToF    
//     if(TOF) {
// 	O_clone = new TTree("ToF_clone","PAMELA ToF level2 data ");	
// 	tof_clone = new ToFLevel2();
// 	O_clone->Branch("ToFLevel2","ToFLevel2", &tof_clone);
// 	cout << "ToF          : branch ToFLevel2"<<endl;
//     };
//     // Trigger
//     if(TRG) {
// 	R_clone = new TTree("Trigger_clone","PAMELA trigger level2 data ");	
// 	trig_clone = new TrigLevel2();
// 	R_clone->Branch("TrigLevel2","TrigLevel2", &trig_clone);
// 	cout << "Trigger      : branch TrigLevel2"<<endl;
//     };
//     // S4
//     if(S4) {
// 	S_clone = new TTree("S4_clone","PAMELA S4 level2 data ");	
// 	s4_clone = new S4Level2();
// 	S_clone->Branch("S4Level2","S4Level2", &s4_clone);
// 	cout << "S4           : branch S4Level2"<<endl;
//     };
//     // Neutron Detector
//     if(ND) {
// 	N_clone = new TTree("NeutronD_clone","PAMELA neutron detector level2 data ");	
// 	nd_clone = new NDLevel2();
// 	N_clone->Branch("NDLevel2","NDLevel2", &nd_clone);
// 	cout << "NeutronD     : branch NDLevel2"<<endl;
//     };
//     // Anticounters
//     if(AC) {
// 	A_clone = new TTree("Anticounter_clone","PAMELA anticounter detector level2 data ");	
// 	ac_clone = new AcLevel2();
// 	A_clone->Branch("AcLevel2","AcLevel2", &ac_clone);
// 	cout << "Anticounter  : branch AcLevel2"<<endl;
//     };
//     // OrbitalInfo
//     if(ORB) {
// 	B_clone = new TTree("OrbitalInfo_clone","PAMELA oribital info  ");	
// 	orb_clone = new OrbitalInfo();
// 	B_clone->Branch("OrbitalInfo","OrbitalInfo", &orb_clone);
// 	cout << "OrbitalInfo  : branch OrbitalInfo"<<endl;
//     };
//     cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;

// }

/**
 * Fill tree (created with CreateCloneTrees)
 *
 */
//void PamLevel2::FillNewPamTree(TTree *T){
void PamLevel2::FillCloneTrees(){
        
// //  --------------------------------------
// //  copy the event
// //  --------------------------------------
//     if(trk1_clone)  *trk1_clone = *trk1_obj;
//     if(trk2_clone){
// 	trk2_clone->Clear();
// 	trk2_obj->Copy(*trk2_clone);
// //	*trk2_clone = *trk2_obj;
//     }
//     if(trkh_clone)  *trkh_clone = *trkh_obj;
//     if(calo1_clone){
// //	*calo1_clone = *calo1_obj;
// 	calo1_clone->Clear();
// 	calo1_obj->Copy(*calo1_clone);
//     }
//     if(calo2_clone){
// //	*calo2_clone = *calo2_obj;
// 	calo2_clone->Clear();
// 	calo2_obj->Copy(*calo2_clone);
//     }
//     if(tof_clone)   *tof_clone  = *tof_obj;
//     if(trig_clone)  *trig_clone = *trig_obj;
//     if(s4_clone)    *s4_clone   = *s4_obj;
//     if(nd_clone)    *nd_clone   = *nd_obj;
//     if(ac_clone)    *ac_clone   = *ac_obj;
//     if(orb_clone)   *orb_clone  = *orb_obj;

//     if(T_clone)T_clone->Fill();
//     if(C_clone)C_clone->Fill();
//     if(O_clone)O_clone->Fill();
//     if(R_clone)R_clone->Fill();
//     if(S_clone)S_clone->Fill();
//     if(N_clone)N_clone->Fill();
//     if(A_clone)A_clone->Fill();
//     if(O_clone)O_clone->Fill();

    for(Int_t i=0; i<8; i++){
	if(tree_clone[i])tree_clone[i]->Fill();
    }
}


TTree* PamLevel2::GetCloneTree(TString name){

    for(Int_t i=0; i<8; i++){	
	if(tree_clone[i]){
	    TString na = tree_clone[i]->GetName();
	    if(!name.CompareTo(na))return tree_clone[i];
	};
    }
    return NULL;

}
void PamLevel2::WriteCloneTrees(){
    cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;
    cout << "Write clones of PAMELA trees "<<endl;
    for(Int_t i=0; i<8; i++){	
	if(tree_clone[i]){
	    cout << tree_clone[i]->GetName()<<endl;
	    tree_clone[i]->Write();
	};
    }
    cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;

}

/**
 * Create a new (empty) Pamela trees 
 */
// TTree* PamLevel2::GetNewPamTree(){

//     if(tree_clone)return tree_clone;

//     TTree *Tout = 0;
    
//     cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;
//     cout << "Create new PAMELA trees "<<endl;
    
    
//     if(TRK1||TRK2||TRKh){
// 	TTree *T = new TTree("Tracker_clone","PAMELA tracker level2 data ");	
// 	if(TRK1) {
// 	    trk1_clone = new TrkLevel1();
// 	    T->Branch("TrkLevel1","TrkLevel1", &trk1_clone);
// 	    T->BranchRef();
// 	    cout << "Tracker      : branch TrkLevel1"<<endl;
// 	};
// 	if(TRK2) {
// 	    trk2_clone = new TrkLevel2();
// 	    T->Branch("TrkLevel2", "TrkLevel2",&trk2_clone);
// 	    cout << "Tracker      : branch TrkLevel2"<<endl;	    
// 	};
// 	if(TRKh) {
// 	    trkh_clone = new TrkHough();
// 	    T->Branch("TrkHough","TrkHough", &trkh_clone);
// 	    cout << "Tracker      : branch TrkHough"<<endl;
// 	};
// 	if(!Tout)Tout=T;
// 	else Tout->AddFriend("Tracker_clone");
//     }

//     // Calorimeter
//     if(CAL1||CAL2){
// 	TTree *C = new TTree("Calorimeter_clone","PAMELA calorimeter level2 data ");	
// 	if(CAL1) {
// 	    calo1_clone = new CaloLevel1();
// 	    C->Branch("CaloLevel1", "CaloLevel1", &calo1_clone);
// 	    cout << "Calorimeter  : branch CaloLevel1"<<endl;
// 	};
// 	if(CAL2) {
// 	    calo2_clone = new CaloLevel2();
// 	    C->Branch("CaloLevel2","CaloLevel2", &calo2_clone);
// 	    cout << "Calorimeter  : branch CaloLevel2"<<endl;
// 	};
// 	if(!Tout)Tout=C;
// 	else Tout->AddFriend("Calorimeter_clone");
//     }    

//     // ToF    
//     if(TOF) {
// 	TTree *O = new TTree("ToF_clone","PAMELA ToF level2 data ");	
// 	tof_clone = new ToFLevel2();
// 	O->Branch("ToFLevel2","ToFLevel2", &tof_clone);
// 	cout << "ToF          : branch ToFLevel2"<<endl;
// 	if(!Tout)Tout=O;
// 	else Tout->AddFriend("ToF_clone");
//     };
//     // Trigger
//     if(TRG) {
// 	TTree *R = new TTree("Trigger_clone","PAMELA trigger level2 data ");	
// 	trig_clone = new TrigLevel2();
// 	R->Branch("TrigLevel2","TrigLevel2", &trig_clone);
// 	cout << "Trigger      : branch TrigLevel2"<<endl;
// 	if(!Tout)Tout=R;
// 	else Tout->AddFriend("Trigger_clone");
//     };
//     // S4
//     if(S4) {
// 	TTree *S = new TTree("S4_clone","PAMELA S4 level2 data ");	
// 	s4_clone = new S4Level2();
// 	S->Branch("S4Level2","S4Level2", &s4_clone);
// 	cout << "S4           : branch S4Level2"<<endl;
// 	if(!Tout)Tout=S;
// 	else Tout->AddFriend("S4_clone");
//     };
//     // Neutron Detector
//     if(ND) {
// 	TTree *N = new TTree("NeutronD_clone","PAMELA neutron detector level2 data ");	
// 	nd_clone = new NDLevel2();
// 	N->Branch("NDLevel2","NDLevel2", &nd_clone);
// 	cout << "NeutronD     : branch NDLevel2"<<endl;
// 	if(!Tout)Tout=N;
// 	else Tout->AddFriend("NeutronD_clone");
//     };
//     // Anticounters
//     if(AC) {
// 	TTree *A = new TTree("Anticounter_clone","PAMELA anticounter detector level2 data ");	
// 	ac_clone = new AcLevel2();
// 	A->Branch("AcLevel2","AcLevel2", &ac_clone);
// 	cout << "Anticounter  : branch AcLevel2"<<endl;
// 	if(!Tout)Tout=A;
// 	else Tout->AddFriend("Anticounter_clone");
//     };
//     // OrbitalInfo
//     if(ORB) {
// 	TTree *B = new TTree("OrbitalInfo_clone","PAMELA oribital info  ");	
// 	orb_clone = new OrbitalInfo();
// 	B->Branch("OrbitalInfo","OrbitalInfo", &orb_clone);
// 	cout << "OrbitalInfo  : branch OrbitalInfo"<<endl;
// 	if(!Tout)Tout=B;
// 	else Tout->AddFriend("OrbitalInfo_clone");
//     };
//     cout << "+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+" <<endl;

//     tree_clone = Tout;
//     tree_clone->SetDirectory(0);

//     return Tout;
// }
// /**
//  * Fill a tree (created with GetNewPamTree)
//  *
//  */
// //void PamLevel2::FillNewPamTree(TTree *T){
// void PamLevel2::FillNewPamTree(){
        

//     if(trk1_clone)  *trk1_clone = *trk1_obj;
//     if(trk2_clone){
// 	trk2_clone->Clear();
// 	trk2_obj->Copy(*trk2_clone);
// //	*trk2_clone = *trk2_obj;
//     }
//     if(trkh_clone)  *trkh_clone = *trkh_obj;
//     if(calo1_clone){
// //	*calo1_clone = *calo1_obj;
// 	calo1_clone->Clear();
// 	calo1_obj->Copy(*calo1_clone);
//     }
//     if(calo2_clone){
// //	*calo2_clone = *calo2_obj;
// 	calo2_clone->Clear();
// 	calo2_obj->Copy(*calo2_clone);
//     }
//     if(tof_clone)   *tof_clone  = *tof_obj;
//     if(trig_clone)  *trig_clone = *trig_obj;
//     if(s4_clone)    *s4_clone   = *s4_obj;
//     if(nd_clone)    *nd_clone   = *nd_obj;
//     if(ac_clone)    *ac_clone   = *ac_obj;
//     if(orb_clone)   *orb_clone  = *orb_obj;

//     TTree *T = tree_clone;

//     T->Fill(); //fill main tree
// //    cout<<T->IsA()->GetName()<<" "<<T->GetName()<<endl;
//     TList *li = T->GetListOfFriends();
//     TIter next(li);
//     TFriendElement* T_friend=0;
//     while( (T_friend = (TFriendElement*)next()) ){
// // 	cout<<T_friend->IsA()->GetName()<<" "<<T_friend->GetName()<<hex << T_friend->GetTree() << dec<<endl;
//  	T_friend->GetTree()->Fill();//fill friends
//     }

// }
