#include "PamVMCTofDig.h"
#include "TDatabasePDG.h"

#include <TMath.h>
#include <TRandom.h>

using TMath::Power;
using TMath::Exp;
using TMath::Abs;

ClassImp(PamVMCTofDig)


void PamVMCTofDig::LoadCalib(){

 cout<<"Loading Tof Calibrations..."<<endl;

  Int_t time=3;
  Int_t type=202;

  fdberr = fsql->Query_GL_PARAM(time,type); 

  fquery.str("");
  fquery << fsql->GetPAR()->PATH.Data() << "/";
  fquery << fsql->GetPAR()->NAME.Data();

  ThrowCalFileUsage("TOF",fquery.str().c_str());
  
  fcfile.open(fquery.str().c_str());

  if(!fcfile) ThrowCalFileWarning("TOF"); else {

    Int_t temp;
  
    for(Int_t i=0; i<NP; i++){
      fcfile >> temp;
      fcfile >> fatte1[i];
      fcfile >> flambda1[i];
      fcfile >> fatte2[i];
      fcfile >> flambda2[i];
      fcfile >> temp;
    }

  }

  fcfile.close();
}




void PamVMCTofDig::DigitizeTOF(Int_t EventNo, Int_t PrimaryPDG){

  fDEBUG = kFALSE;

  // pC < 800 
  const Float_t ADC_pC0A =  -4.437616e+01;
  const Float_t ADC_pC1A =   1.573329e+00;
  const Float_t ADC_pC2A =   2.780518e-04;
  const Float_t ADC_pC3A =  -2.302160e-07;
  // pC > 800:
  const Float_t ADC_pC0B =    -2.245756e+02;
  const Float_t ADC_pC1B =     2.184156e+00;
  const Float_t ADC_pC2B =    -4.171825e-04;
  const Float_t ADC_pC3B =     3.789715e-08;
  
  const Float_t pCthres=40.;     // threshold in charge
  const Int_t ADClast=4095;      // no signal --> ADC ch=4095
  const Int_t ADCsat=3100;       // saturation value for the ADCs
  const Int_t TDClast=4095;


  for(Int_t i =0; i<NP; i++){
    fQevePmt_pC[i]=ftdc[i]=ftdc1[i]=0.;
    ftdcpmt[i]=1000.;
  }

  

  TClonesArray* hc = 0;
  const char* keyplane [6] = {"S11Y","S12X","S21X","S22Y","S31Y","S32X"};
  for(Int_t i=0; i<6; i++){
    hc = (TClonesArray*)fhitscolmap.GetValue(keyplane[i]);
    if (hc) DigitizeTofPlane(i,hc, PrimaryPDG);
    hc = 0;
  }

  if(fDEBUG){
    cout<<"Summarized values for ADC in PC:"<<endl;
    for(Int_t i=0; i<NP; i++){
      cout<<"PMT #"<<i<<" fQevePmt_pC="<<fQevePmt_pC[i]<<endl;
    }
  }
  //+++++ ADC +++++

  for(Int_t i=0; i<NP; i++){
    if (fQevePmt_pC[i] <= 800.)  
      fADCtof[i]= (Int_t)(ADC_pC0A + ADC_pC1A*fQevePmt_pC[i] 
			  + ADC_pC2A*Power(fQevePmt_pC[i],2) 
			  + ADC_pC3A*Power(fQevePmt_pC[i],3));
    if ((fQevePmt_pC[i] > 800.) && (fQevePmt_pC[i] <= 2485.))
      fADCtof[i]= (Int_t)(ADC_pC0B + ADC_pC1B*fQevePmt_pC[i] 
			  + ADC_pC2B*Power(fQevePmt_pC[i],2) 
			  + ADC_pC3B*Power(fQevePmt_pC[i],3));
    if (fQevePmt_pC[i] > 2485.) fADCtof[i]= (Int_t)(1758. + 0.54*fQevePmt_pC[i]);  
    //assuming a fictional 0.54 ch/pC above ADCsat
    if (fADCtof[i]>ADCsat) fADCtof[i]=ADCsat;
    if (fQevePmt_pC[i] < pCthres)  fADCtof[i]= ADClast;
    if (fADCtof[i] < 0) fADCtof[i]=ADClast;    
    if (fADCtof[i] > ADClast) fADCtof[i]=ADClast;
  }

  // ======  TDC coincidence  ======


  for(Int_t i=0; i<NP; i++) {
    if((ftdcpmt[i] - fc1_S[i]) > 1e-7) {
      ftdcpmt[i] = 0.;
      ftdc[i] = 0.;
    }
  }// cycle to introduce a window for tdc

  Double_t t_coinc = 0;
  Int_t ilast = 100;
  for (Int_t ii=0; ii<NP;ii++)
    if (ftdc[ii] > t_coinc) {
      t_coinc = ftdc[ii];
      ilast = ii;
    }

  for (Int_t ii=0; ii<NP;ii++){
    if (ftdc[ii] != 0) ftdc1[ii] = t_coinc - ftdcpmt[ii];  // test 2
    ftdc1[ii] = ftdc1[ii]/ftdcres[ii];                     // divide by TDC resolution 
    if (ftdc[ii] != 0) ftdc1[ii] = ftdc1[ii] + fc3_S[ii];  // add cable length c3
  } 
  
  if (fDEBUG)cout<<"====== TOF coincidence ======"<<endl;
  for(Int_t i=0; i<NP; i++){
    if(ftdc1[i] != 0.){
      fTDCint[i]=(Int_t)ftdc1[i];
      if (fTDCint[i]>4093) fTDCint[i]=TDClast;  // 18-oct WM
      if (fDEBUG)cout<<"PMT: "<<i<<" ADC: "<<fADCtof[i]<<" TDC: "
		     <<fTDCint[i]<<endl;
    } else
      fTDCint[i]= TDClast;
  }
  if (fDEBUG)cout<<"============================="<<endl;

  //------ use channelmap for ToF: 18-oct WM
  Int_t  channelmap[] =  {3,21,11,29,19,45,27,37,36,28,44,20,5,12,13,4,
			  6,47,14,39,22,31,30,23,38,15,46,7,0,33,16,24,
			  8,41,32,40,25,17,34,9,42,1,2,10,18,26,35,43};
  Int_t ADChelp[NP],TDChelp[NP];
  for(Int_t i=0; i<NP; i++){
    Int_t ii=channelmap[i];
    ADChelp[ii]= fADCtof[i];
    TDChelp[ii]= fTDCint[i];
  }
  for(Int_t i=0; i<NP; i++){
    fADCtof[i]= ADChelp[i];
    fTDCint[i]= TDChelp[i];
  }

  if (fDEBUG){
    cout<<"====== TOF coincidence after... ======"<<endl;
    for(Int_t i=0; i<NP; i++){
      cout<<"PMT: "<<i<<" ADC: "<<fADCtof[i]<<" TDC: "
	  <<fTDCint[i]<<endl;
    }
    cout<<"============================="<<endl;
  }
  // ======  write DataTof  =======

  UChar_t Ctrl3bit[8]={32,0,96,64,160,128,224,192};  // DC (msb in 8 bit word )
  UChar_t tofBin;
  UChar_t DataTof[276];
  for (Int_t j=0; j < 276; j++)DataTof[j]=0x00;
  UChar_t *pTof=DataTof;
  for (Int_t j=0; j < 12; j++){   // loop on TDC #12
    Int_t j12=j*23;               // for each TDC 23 bytes (8 bits)
    DataTof[j12+0]=0x00;   // TDC_ID
    DataTof[j12+1]=0x00;   // EV_COUNT
    DataTof[j12+2]=0x00;   // TDC_MASK (1)
    DataTof[j12+3]=0x00;   // TDC_MASK (2)
    for (Int_t k=0; k < 4; k++){   // for each TDC 4 channels (ADC+TDC)
      Int_t jk12=j12+4*k;         // ADC,TDC channel (0-47)
      tofBin =(UChar_t)(fADCtof[k+4*j]/256);   // ADC# (msb) 
      DataTof[jk12+4] = Bin2GrayTof(tofBin,DataTof[jk12+4]);
      /* control bits inserted here, after the bin to gray conv - DC*/ 
      DataTof[jk12+4] = Ctrl3bit[2*k] | DataTof[jk12+4];
      tofBin=(UChar_t)(fADCtof[k+4*j]%256);   // ADC# (lsb)
      DataTof[jk12+5] = Bin2GrayTof(tofBin,DataTof[jk12+5]);
      tofBin=(UChar_t)(fTDCint[k+4*j]/256);   // TDC# (msb)
      DataTof[jk12+6]=Bin2GrayTof(tofBin,DataTof[jk12+6]);
      /* control bits inserted here, after the bin to gray conv - DC*/ 
      DataTof[jk12+6] = Ctrl3bit[2*k+1] | DataTof[jk12+6];
      tofBin=(UChar_t)(fTDCint[k+4*j]%256);   // TDC# (lsb)
      //if(fDEBUG) cout<<" digit TDC: "<<jk12<<" TDC:"<<(Int_t)tofBin<<endl;
      DataTof[jk12+7]=Bin2GrayTof(tofBin,DataTof[jk12+7]);
    }
    DataTof[j12+20]=0x00;   // TEMP1
    DataTof[j12+21]=0x00;   // TEMP2
    DataTof[j12+22]= EvaluateCrc(pTof,22);   // CRC
    pTof+=23;
  }

  //===== write DataTrigger =======

  UChar_t DataTrigger[152];
  for (Int_t j=0; j < 152; j++)DataTrigger[j]=0x00;
  UChar_t *pTrg=DataTrigger;
  // Only the variables with a (*) are modified; the others are set to 0
  // info given in #bites data + #bites crc
  // TB_READ_PMT_PLANE      :  6 + 1
  // TB_READ_EVENT_COUNT    :  3 + 1 (*)
  // TB_READ_TRIGGER_RATE   : 12 + 1
  // TB_READ_D_L_TIME       :  4 + 1
  // TB_READ_S4_CAL_COUNT   :  4 + 1
  // TB_READ_PMT_COUNT1     : 48 + 1
  // TB_READ_PMT_COUNT2     : 48 + 1
  // TB_READ_PATTERN_BUSY   :  8 + 1
  // TB_READ_PATTERN_TRIGGER:  7 + 1 (*)
  // TB_READ_TRIGGER_CONF   :  2 + 1 (*)
  // TB_READ_EVENT_COUNT
  UInt_t cTrg  = EventNo;  //counter
  UInt_t cTrg2 = 0;  //counter with bits inverted, according to document 
                     //"formato dati provenienti dalla trigger board"
  for (Int_t i=0; i < 24; i++){  // Use the first 24 bits
    if (cTrg & (0x1 << i) )
      cTrg2 = cTrg2 | (0x1 << (24-i));
  }
  DataTrigger[7] = (UChar_t)(cTrg2 >> 16); // 8 MSbits (out  of 24)
  DataTrigger[8] = (UChar_t)(cTrg2 >> 8);  // 8 "middle" bits
  DataTrigger[9] = (UChar_t)(cTrg2);       // 8 LSbits
  pTrg=DataTrigger+7;
  DataTrigger[10]=EvaluateCrc(pTrg, 3);
  
  // TB_READ_PATTERN_TRIGGER: bytes 141-148:
  // PatternTrigMap[i] corresponds to bit i in TB_READ_PATTERN_TRIGGER:
  // mapping according to documents:
  // 1. "formato dati provenienti dalla trigger board"
  // 2. "The ToF quicklook software", Appendix A (Campana, Nagni)
  Int_t PatternTrigMap[]={29,42,43,1,16,7,17,28,33,41,46,2,15,8,18,27,
			  30,40,44,3,14,9,19,26,32,37,47,4,13,10,20,25,
			  34,31,38,45,5,12,21,24,36,35,39,48,6,11,22,23};
  for (Int_t i=0; i<NP; i++){
    if (ftdc1[channelmap[i]]!=0)
      DataTrigger[147-(Int_t)((PatternTrigMap[i]+1)/8)]=DataTrigger[147-(Int_t)((PatternTrigMap[i]+1)/8)] | (0x1 << (PatternTrigMap[i]%8));
  }
  pTrg=DataTrigger+141;
  DataTrigger[148]=EvaluateCrc(pTrg, 7);
  
  // TB_READ_TRIGGER_CONF   : set always acq.mode TOF4
  //
  // TOF1: S1-S2-S3 (&,|)
  // TOF4: S2-S3 (&,&)
  DataTrigger[149]=0x02;
  DataTrigger[150]=0x0;
  pTrg=DataTrigger+149;
  DataTrigger[151]=EvaluateCrc(pTrg, 2);


  //++++++ WRITE EVERYTHING TO DIGITIZER'S BUFFER +++//
  if(fDEBUG) cout<<"TOF Digitizing"<<endl;

  fDataC.clear(); //clearing Tof & Trig data buffer

  for(Int_t i= 0; i<152; i++) fDataC.push_back(DataTrigger[i]);

  for(Int_t i= 0; i<276; i++) fDataC.push_back(DataTof[i]);

}

Float_t PamVMCTofDig::TimeRes(Int_t PrimaryPDG){

  Float_t time_res[8] = {425.,210.,170.,130.,120.,120.,120.,120.}; 
  for(Int_t i=0;i<8;i++) time_res[i]/=1.4;//1.17;1.5;1.3*/
  Int_t Z = Int_t((TDatabasePDG::Instance()->GetParticle(PrimaryPDG))->Charge()/3.);
    
  Float_t dt1 = 1.e-12*time_res[0];  // single PMT resolution for Z=1  (WM, Nov'07)
  if ((Z > 1) && (Z < 9)) dt1=1.e-12*time_res[(Z-1)];
  if  (Z > 8) dt1=120.e-12;
  
  return dt1;
}

void PamVMCTofDig::DigitizeTofPlane(Int_t planeNo, TClonesArray* HitColl, Int_t PrimaryPDG){
  
  PamVMCDetectorHit * hit = 0;
  const Float_t veff0 = 100.*1.0e8;//(m/s) light velocity in scintillator
  const Float_t veff1 = 100.*1.5e8;//(m/s) light velocity in lightguide
  const Float_t FGeo[2] = {0.5, 0.5}; //geometrical factor
  const Float_t Pho_keV = 10.;// photons per keV in scintillator
  const Float_t effi = 0.21; //photocathofe efficiency
  const Float_t pmGain = 3.5e6; //PMT Gain: the same for all PMTs
  const Float_t echarge = 1.6e-19; // electron charge
  const Float_t thresh=20.; // to be defined better... (Wolfgang)

  const Float_t dimel[6] = {33.0, 40.8 ,18.0, 15.0, 15.0, 18.0}; //(cm) TOF paddles dimensions
  //  S11 8 paddles  33.0 x 5.1 cm
  //  S12 6 paddles  40.8 x 5.5 cm
  //  S21 2 paddles  18.0 x 7.5 cm
  //  S22 2 paddles  15.0 x 9.0 cm
  //  S31 3 paddles  15.0 x 6.0 cm
  //  S32 3 paddles  18.0 x 5.0 cm

  const Float_t s_l_g[6] = {8.0, 8.0, 20.9, 22.0, 9.8, 8.3 };  //(cm) length of the lightguide

  const Float_t ScaleFact[48]={0.39, 0.49, 0.38, 0.40, 0.65, 0.51, 0.43,
	0.49, 0.58, 0.38, 0.53, 0.57, 0.53, 0.45, 0.49, 0.16,
	0.15, 0.44, 0.28, 0.57, 0.26, 0.72, 0.37, 0.29, 0.30, 0.89,
	0.37, 0.08,  0.27, 0.23, 0.12, 0.22, 0.15, 0.16, 0.21,
	0.19, 0.41, 0.32, 0.39, 0.38, 0.28, 0.66, 0.28, 0.40, 0.39, 0.40, 0.37, 0.35 };

  Float_t t1, t2, tpos, Npho;
  Float_t path[2], knorm[2], Atten[2], QhitPad_pC[2], QhitPmt_pC[2];
  Int_t padNo, pmtleft, pmtright;
  //LOOP
  for(Int_t i =0; i<HitColl->GetEntriesFast(); i++){
    
    hit = (PamVMCDetectorHit*)HitColl->At(i);
    
    t1=t2 = hit->GetTOF();
    padNo = hit->GetPOS()-1;
    pmtleft=pmtright=0;
    if(planeNo==2){
      if(padNo==0)
	padNo=1;
      else
	padNo=0;
    }

    Paddle2Pmt(planeNo,padNo, &pmtleft, &pmtright);
    
    switch(planeNo){
    case 0:
    case 3:
    case 4:
      tpos = (hit->GetYIN()+hit->GetYOUT())/2.; //Y-planes
      break;
    case 1:
    case 2:
    case 5:
      tpos = (hit->GetXIN()+hit->GetXOUT())/2.; //X-planes
      break;
    default:
      cout<<"PamVMCTofDig::DigitizeTOFPlane wrong plane no "<<planeNo<<endl;
      tpos = -100.;
      break;
    }

    path[0]= tpos + dimel[planeNo]/2.;   // path to left PMT
    path[1]= dimel[planeNo]/2.- tpos;    // path to right PMT

    if (fDEBUG) {
      cout <<"+++++  TOF HIT VERBOSE INFORMATION: +++++"<<endl;
      cout <<"planeNo "<<planeNo<<" padNo "<< padNo <<" tpos  "<< tpos <<"\n";
      cout <<"pmtleft, pmtright "<<pmtleft<<","<<pmtright<<endl;
    }

    Npho = hit->GetEREL()*Pho_keV*1.0e6; //calculation of photons number

    for(Int_t j=0; j<2; j++){
      QhitPad_pC[j]= Npho*FGeo[j]*effi*pmGain*echarge*1.E12*ScaleFact[pmtleft+j];
      knorm[j]=fatte1[pmtleft+j]*Exp(flambda1[pmtleft+j]*dimel[planeNo]/2.*Power(-1,j+1)) + 
	fatte2[pmtleft+j]*Exp(flambda2[pmtleft+j]*dimel[planeNo]/2.*Power(-1,j+1));
      Atten[j]=fatte1[pmtleft+j]*Exp(tpos*flambda1[pmtleft+j]) + 
	fatte2[pmtleft+j]*Exp(tpos*flambda2[pmtleft+j]) ;
      QhitPmt_pC[j]= QhitPad_pC[j]*Atten[j]/knorm[j];
      if (fDEBUG) {
	cout<<"pmtleft:"<<pmtleft<<" j:"<<j<<endl;
	cout<<"atte1:"<<fatte1[pmtleft+j]<<" lambda1:"<<flambda1[pmtleft+j]
	    <<" atte2:"<<fatte2[pmtleft+j]<<" lambda2:"<<flambda2[pmtleft+j] 
	    <<endl;    
	cout<<j<<" tpos:"<<tpos<<" knorm:"<<knorm[j]<<" "<<Atten[j]<<" "
	    <<"QhitPmt_pC "<<QhitPmt_pC[j]<<endl;    
      }
    }
    
    if(fDEBUG)cout<<"Energy release (keV):"<<hit->GetEREL()*1.e6<<" Npho:"<<Npho<<
	       " QhitPmt_pC(left,right):"<<QhitPmt_pC[0]<<" "<<QhitPmt_pC[1]<<endl;

    fQevePmt_pC[pmtleft]  += QhitPmt_pC[0];  //cdding charge from separate hits
    fQevePmt_pC[pmtright] += QhitPmt_pC[1];

    // TDC
    // WM right and left <->    
    t1 += Abs(path[0]/veff0) + s_l_g[planeNo]/veff1;
    t2 += Abs(path[1]/veff0) + s_l_g[planeNo]/veff1;  // Signal reaches PMT
    t1 = frandom->Gaus(t1,TimeRes(PrimaryPDG)); //apply gaussian error  dt
    t2 = frandom->Gaus(t2,TimeRes(PrimaryPDG)); //apply gaussian error  dt
    t1 += fc1_S[pmtleft] ;  // Signal reaches Discriminator ,TDC starts  to run
    t2 += fc1_S[pmtright] ;

    // check if signal is above threshold
    // then check if tdcpmt is already filled by another hit...
    // only re-fill if time is smaller
    if (QhitPmt_pC[0] > thresh) {
      if (ftdcpmt[pmtleft] == 1000.) {  // fill for the first time
	ftdcpmt[pmtleft] = t1;
	ftdc[pmtleft] = t1 + fc2_S[pmtleft] ;  // Signal reaches Coincidence
      }
      if (ftdcpmt[pmtleft] < 1000.) // is already filled!
	if (t1 <  ftdcpmt[pmtleft]) {
	  ftdcpmt[pmtleft] = t1;
	  t1 += fc2_S[pmtleft] ;  // Signal reaches Coincidence
	  ftdc[pmtleft] = t1;
	}
    }
    if (QhitPmt_pC[1] > thresh) {
      if (ftdcpmt[pmtright] == 1000.) {  // fill for the first time
	ftdcpmt[pmtright] = t2;
	ftdc[pmtright] = t2 + fc2_S[pmtright] ;  // Signal reaches Coincidence
      }
      if (ftdcpmt[pmtright] < 1000.)  // is already filled!
	if (t2 <  ftdcpmt[pmtright]) {
	  ftdcpmt[pmtright] = t2;
	  t2 += fc2_S[pmtright] ;
	  ftdc[pmtright] = t2;
	}
    }
    if(fDEBUG)cout<<"Time(ns):"<<hit->GetTOF()*1.0E9
		  <<" t1:"<<t1*1.0E9
		  <<" t2:"<<t2*1.0E9<<endl
		  <<"+++++  END OF TOF HIT +++++"<<endl;
  }; 
  //END OF HIT COLLECTION LOOP 
}

void PamVMCTofDig::Paddle2Pmt(Int_t planeNo, Int_t padNo, Int_t *pl, Int_t *pr){
  //* @param plane    (0 - 5)
  //* @param paddle   (plane=0, paddle = 0,...5)
  //* @param padid    (0 - 23)  
  //
  Int_t padid=-1;
  Int_t pads[6]={8,6,2,2,3,3};
  //
  Int_t somma=0;
  for(Int_t j=0; j<planeNo; j++) somma+=pads[j];
  padid=padNo+somma;
  *pl = padid*2;
  *pr = *pl + 1; // WM

}


UChar_t PamVMCTofDig::Bin2GrayTof(UChar_t binaTOF,UChar_t grayTOF){
  union graytof_data {
    UChar_t word;
    struct bit_field {
      unsigned b0:1;
      unsigned b1:1;
      unsigned b2:1;
      unsigned b3:1;
      unsigned b4:1;
      unsigned b5:1;
      unsigned b6:1;
      unsigned b7:1;
    } bit;
  } bi,gr;
  //
  bi.word = binaTOF;
  gr.word = grayTOF;
  // 
  gr.bit.b0 = bi.bit.b1 ^ bi.bit.b0;
  gr.bit.b1 = bi.bit.b2 ^ bi.bit.b1;
  gr.bit.b2 = bi.bit.b3 ^ bi.bit.b2;
  gr.bit.b3 = bi.bit.b3;
  //
  /* bin to gray conversion 4 bit per time*/
  //
  gr.bit.b4 = bi.bit.b5 ^ bi.bit.b4;
  gr.bit.b5 = bi.bit.b6 ^ bi.bit.b5;
  gr.bit.b6 = bi.bit.b7 ^ bi.bit.b6;
  gr.bit.b7 = bi.bit.b7;
  //
  return(gr.word); 
}

void PamVMCTofDig::Crc8Tof(UChar_t *oldCRC, UChar_t *crcTof){
  union crctof_data {
    UChar_t word;
    struct bit_field {
      unsigned b0:1;
      unsigned b1:1;
      unsigned b2:1;
      unsigned b3:1;
      unsigned b4:1;
      unsigned b5:1;
      unsigned b6:1;
      unsigned b7:1;
    } bit;
  } c,d,r;

  c.word = *oldCRC;
  //d.word = *newCRC;
  d.word = *crcTof;
  r.word = 0;

  r.bit.b0 = c.bit.b7 ^ c.bit.b6 ^ c.bit.b0 ^ 
             d.bit.b0 ^ d.bit.b6 ^ d.bit.b7;

  r.bit.b1 = c.bit.b6 ^ c.bit.b1 ^ c.bit.b0 ^ 
             d.bit.b0 ^ d.bit.b1 ^ d.bit.b6;

  r.bit.b2 = c.bit.b6 ^ c.bit.b2 ^ c.bit.b1 ^ c.bit.b0 ^
             d.bit.b0 ^ d.bit.b1 ^ d.bit.b2 ^ d.bit.b6;

  r.bit.b3 = c.bit.b7 ^ c.bit.b3 ^ c.bit.b2 ^ c.bit.b1 ^ 
             d.bit.b1 ^ d.bit.b2 ^ d.bit.b3 ^ d.bit.b7;

  r.bit.b4 = c.bit.b4 ^ c.bit.b3 ^ c.bit.b2 ^
             d.bit.b2 ^ d.bit.b3 ^ d.bit.b4;

  r.bit.b5 = c.bit.b5 ^ c.bit.b4 ^ c.bit.b3 ^
             d.bit.b3 ^ d.bit.b4 ^ d.bit.b5;

  r.bit.b6 = c.bit.b6 ^ c.bit.b5 ^ c.bit.b4 ^ 
             d.bit.b4 ^ d.bit.b5 ^ d.bit.b6;

  r.bit.b7 = c.bit.b7 ^ c.bit.b6 ^ c.bit.b5 ^ 
             d.bit.b5 ^ d.bit.b6 ^ d.bit.b7 ;

  *crcTof=r.word;
  //return r.word; 
};




UChar_t PamVMCTofDig::EvaluateCrc(UChar_t *pTrg, Int_t nb) {
  Bool_t DEBUG=false;
  if (DEBUG)
    return(0x00);

  UChar_t crcTrg=0x00;
  UChar_t *pc=&crcTrg, *pc2;
  pc2=pTrg;
  for (Int_t jp=0; jp < nb; jp++)
    Crc8Tof(pc2++,pc);
  return(crcTrg);
}
