// C/C++ headers
//
#include <fstream>
#include <string.h>
#include <iostream>
#include <stdio.h>
#include <ctype.h>

//
// ROOT headers
//

#include <TTree.h>

#include <TSystem.h>
#include <TStopwatch.h>
#include <TSystemDirectory.h>
#include <TGraphAsymmErrors.h>
#include <TString.h>
#include <TFile.h>
#include <TClass.h>
#include <TGraph.h>
#include <TLine.h>
#include <TH1F.h>
#include <TMath.h>
#include <TRandom3.h>
#include <TF1.h>
//
// RunInfo header
//
#include <RunInfo.h>
#include <PamLevel2.h>
#include <TClonesArray.h>

#include "PamCut/PamCutEnv.h"
#include <VKmask.h>

//PamVMC headers
#include "PamVMCDigMgr.h"
#include "PamVMCTrkHit.h"
#include "PamVMCTrkF77GpsSpe.h"
#include "PamVMCPrimaryGenerator.h"
using namespace std;
//This program is standalone Digitizer for PamVMC


void Usage ();

TClonesArray* FindColl(TTree*t, TString brname, TString type = "PamVMCDetectorHit");
void RegisterPDG(); 
//Here user should define all nuclei and particles that are not in PDG tabe of ROOT
//otherwise application will crash on TOF digitization whrer charge information needed;

void UnCompressCalo(TClonesArray* cal_comp, TClonesArray & cal_u);

//This will make a list of arrays 12X3X8 for each mask 
//It run once on Initialization stage
Bool_t **** MakeBoolOfMasks( TList* msk_in_list );


void KillBadStrips(pGPSSPEHits * hits_in, pGPSSPEHits * hits_out, Bool_t****m_msk,  cGPSSPE* out, TList *msk_lst, Int_t cutoff_bin);


void ApplyMask( pGPSSPEHits * hits_in, cGPSSPE* out, Bool_t *** mask);
Bool_t IsStripXGood( Int_t istrip, Int_t npstrip, Int_t ntstrip, Bool_t *** mask);
Bool_t IsStripYGood( Int_t istrip, Int_t npstrip, Int_t ntstrip, Bool_t *** mask);


int main(int argc, char * argv[] ){


  if(argc!=5){
    Usage();
    return 0;
  }
  
  TString DataPATH  = gSystem->Getenv("PWD");
  TString inputfilename(argv[1]);
  if(!inputfilename.Contains(".root")){
    cerr<<"Error! Not root file on input!"<<endl;
    return 0;
  }
  
  TString outputfilename(argv[2]);

  TStopwatch timer;
  timer.Start();
  TFile* rfin =  new TFile(inputfilename, "UPDATE");
  if(!rfin){
    cerr<<"Error! Can't read root file with hits collections!"<<endl;
    return 0;
  }

  TString masksfilename(argv[3]);
  TFile* rfmasks = new TFile( masksfilename );
  if(!rfmasks){
    cerr<<"Error! Can't read root file with masks!"<<endl;
    return 0;
  }

  TList *msk_list = (TList*)rfmasks->Get("masks");
  Bool_t ****m_msk = MakeBoolOfMasks( msk_list );


  TString cutoff_b_s(argv[4]);
  Int_t cutoff_bin = cutoff_b_s.Atoi();

  TTree::SetMaxTreeSize(1000*Long64_t(2000000000));
     
  TTree * hits_tree = (TTree*)rfin->Get("hits");
  if(!hits_tree){
    cerr<<"Tree hits are not found in file "<<rfin->GetName()<<endl;
    TString treename = rfin->GetName();
    treename.Replace(treename.Index(".root",5),5,"");
    hits_tree = (TTree*)rfin->Get(treename);
    if(!hits_tree) return 0;
  }

  TObjArray * frndArr = new TObjArray();
  hits_tree->SetBranchAddress("RNDM",&frndArr);

  PamVMCRndMgr* rnd_mgr = PamVMCRndMgr::Instance_ext();
  cout<<"dmdr"<<endl;
  PamVMCDigMgr* d_mgr = PamVMCDigMgr::Instance_ext();
  cout<<"dmdr"<<endl;

  //AC CARD: C1D1, C2D1, CAT: TOP1, CAS: SID1
  PamVMCAcDig* ac = new PamVMCAcDig();
  ac->RegisterCollections("C1D1", FindColl(hits_tree,"C1D1"));
  ac->RegisterCollections("C2D1", FindColl(hits_tree,"C2D1"));
  ac->RegisterCollections("TOP1", FindColl(hits_tree,"TOP1"));
  ac->RegisterCollections("SID1", FindColl(hits_tree,"SID1"));
 
  //TOF: S11Y, S12X, S21X, S22Y, S31Y, S32X
  PamVMCTofDig* tof = new PamVMCTofDig();
  tof->RegisterCollections("S11Y", FindColl(hits_tree,"S11Y"));
  tof->RegisterCollections("S12X", FindColl(hits_tree,"S12X"));
  tof->RegisterCollections("S21X", FindColl(hits_tree,"S21X"));
  tof->RegisterCollections("S22Y", FindColl(hits_tree,"S22Y"));
  tof->RegisterCollections("S31Y", FindColl(hits_tree,"S31Y"));
  tof->RegisterCollections("S32X", FindColl(hits_tree,"S32X"));


  //Calorimeter CAST
#define NSTRIPS 4224
  PamVMCCaloDig* cal = new PamVMCCaloDig();
  TClonesArray *cal_u = new TClonesArray("PamVMCDetectorHit",NSTRIPS); //uncompressed array
  TClonesArray* cal_comp = FindColl(hits_tree,"CAST"); // compressed array from file
  cal->RegisterCollections("CAST", cal_u);  

  //Tracker GPSSPE
  PamVMCTrkDig* trk = new PamVMCTrkDig();
  TBranch *gps = hits_tree->GetBranch("GPSSPE");
  trk->RegisterCollections("TSPA", FindColl(hits_tree,"TSPA"));
  pGPSSPEHits * hits_in = new pGPSSPEHits();
  pGPSSPEHits * hits_out = new pGPSSPEHits();
  cGPSSPE* out = new cGPSSPE();
  if(gps){
    hits_tree->SetBranchAddress("GPSSPE",&hits_in);
    hits_tree->SetBranchStatus("GPSSPE",1);
    cout<<"TRKCOLL="<<hits_in<<endl;
    trk->RegisterTrkData(hits_out);
  }  


  //S4
  PamVMCS4Dig* s4 = new PamVMCS4Dig();
  s4->RegisterCollections("S4", FindColl(hits_tree,"S4"));  

  //ND: NDTI
  PamVMCNDDig* nd = new PamVMCNDDig();
  nd->RegisterCollections("NDTI", FindColl(hits_tree,"NDTI"));  


  //Primaries
  TClonesArray *prims = FindColl(hits_tree,"PRIM","PamVMCPrimary");  

  //Initialization
  d_mgr->SetDIG("Tof",tof);
  d_mgr->SetDIG("AC",ac);
  d_mgr->SetDIG("CAST",cal);
  d_mgr->SetDIG("TSPA",trk);
  d_mgr->SetDIG("S4",s4);
  d_mgr->SetDIG("NDTI",nd);

  //Random for masks extraction
  gRandom = new TRandom3(0);

  d_mgr->Initialize( new TRandom3(0)); //Load Calibrations and set dummy random

  PamVMCRawMgr* r_mgr = PamVMCRawMgr::Instance();
  r_mgr->CreateOutFile(outputfilename.Data());
  
  r_mgr->WriteToFile();

  Int_t nev = hits_tree->GetEntries();
  cout<<"TOTAL NUMBER OF EVENTS:"<<nev<<endl;
  RegisterPDG();
  //Event loop
  d_mgr->PrintCollections();
  for(Int_t i = 0; i<nev; i++){
    hits_tree->GetEntry(i);
    cout<<"--> Setting Random"<<endl;
    for(Int_t j = 0; j<frndArr->GetEntries(); j++){
      TRandom3 * rnd = (TRandom3*)frndArr->At(j);
      TString nm = rnd->GetName();
      if ( nm=="AC" ) ac->SetRandom(rnd);
      if ( nm=="Calo" ) cal->SetRandom(rnd);
      if ( nm=="Tracker" ) trk->SetRandom(rnd);
      if ( nm=="S4" ) s4->SetRandom(rnd);
      if ( nm=="Tof" ) tof->SetRandom(rnd);
      if ( nm=="ND" ) nd->SetRandom(rnd);  
    }

    cout<<"--> Digitizing evnt"<<endl;
    UnCompressCalo(cal_comp, *cal_u);
    KillBadStrips( hits_in, hits_out, m_msk, out, msk_list, cutoff_bin);
    d_mgr->Digitize(((PamVMCPrimary*)prims->At(0))->fID,((PamVMCPrimary*)prims->At(0))->fPDG);
    cout<<"--> Writing evnt"<<endl;
    r_mgr->WriteEvent();
    rnd_mgr->ClearColl();
  }

  r_mgr->FinishRun();
  d_mgr->PrintCollections();
  timer.Stop();
  gDirectory->Delete("tmp.root");
  rfin->Close();
  return 0;   
}

TClonesArray* FindColl(TTree*t, TString brname, TString type){
  if(t->GetBranch(brname)){
    TClonesArray* coll = 0;//new TClonesArray(type,4224);
    t->GetBranch(brname)->SetAddress(&coll);
    cout<<"coll="<<coll<<endl;
    return coll; 
  }
  return 0;
}

void RegisterPDG(){
  TDatabasePDG::Instance()->AddParticle("H-2","H-2",1.875613,kTRUE,0., 3. /* |e|/3 charge*/,"Ion",1000010020);
  TDatabasePDG::Instance()->AddParticle("H-3","H-3",2.808921,kTRUE,0., 3. /* |e|/3 charge*/,"Ion",1000010030);
  TDatabasePDG::Instance()->AddParticle("He-4","He-4",3.727379,kTRUE,0., 6. /* |e|/3 charge*/,"Ion",1000020040);
  TDatabasePDG::Instance()->AddParticle("He-3","He-3",2.808391,kTRUE,0., 6. /* |e|/3 charge*/,"Ion",1000020030);

  Double_t gev = 0.93146;// conversion a.m.u. to gev;

  //nuclei and other stuff
  TDatabasePDG::Instance()->AddParticle("Li-6","Li-6",6.015122*gev,kTRUE,0., 9. /* |e|/3 charge*/,"Ion",1000030060);
  TDatabasePDG::Instance()->AddParticle("Li-7","Li-7",7.016004*gev,kTRUE,0., 9. /* |e|/3 charge*/,"Ion",1000030070);
  TDatabasePDG::Instance()->AddParticle("Be-7","Be-7",7.01692983*gev,kTRUE,0., 12. /* |e|/3 charge*/,"Ion",1000040070);
  TDatabasePDG::Instance()->AddParticle("Be-9","Be-9",9.0121822*gev,kTRUE,0., 12. /* |e|/3 charge*/,"Ion",1000040090);
  TDatabasePDG::Instance()->AddParticle("Be-10","Be-10",10.013533818*gev,kTRUE,0., 12. /* |e|/3 charge*/,"Ion",1000040100);
  TDatabasePDG::Instance()->AddParticle("B-10","B-10",10.0129370*gev,kTRUE,0., 15. /* |e|/3 charge*/,"Ion",1000050100);
  TDatabasePDG::Instance()->AddParticle("B-11","B-11",11.0093054*gev,kTRUE,0., 15. /* |e|/3 charge*/,"Ion",1000050110);
  TDatabasePDG::Instance()->AddParticle("C-12","C-12",12.*gev,kTRUE,0., 18. /* |e|/3 charge*/,"Ion",1000060120);
  TDatabasePDG::Instance()->AddParticle("N-14","N-14",14.0030740048*gev,kTRUE,0., 21. /* |e|/3 charge*/,"Ion",1000070140);
  TDatabasePDG::Instance()->AddParticle("O-16","O-16",15.99491461956*gev,kTRUE,0., 23. /* |e|/3 charge*/,"Ion",1000080160);
}

void UnCompressCalo(TClonesArray* cal_comp, TClonesArray & cal_u){
  if(!cal_comp) return;
  PamVMCDetectorHit* hit = NULL;
  PamVMCDetectorHit* newhit = NULL;
  cal_u.Clear("C");
  for(Int_t j = 0; j<cal_comp->GetEntries(); j++){
      hit=((PamVMCDetectorHit*)cal_comp->At(j));
      newhit = (PamVMCDetectorHit*)cal_u.New(hit->GetPOS());
      *newhit = *hit;
  }
}

void KillBadStrips(pGPSSPEHits * hits_in, pGPSSPEHits * hits_out, Bool_t****m_msk,  cGPSSPE* out, TList *msk_lst, Int_t cutoff_bin){

  hits_out->Clean();

  //First I have to do is extract random mask
  TH2D* prof_2d = (TH2D*)msk_lst->At(msk_lst->GetEntries()-1);
  TH1D* prof_1d = (TH1D*)prof_2d->ProjectionY("prof_1d",cutoff_bin, cutoff_bin+1);
  Int_t mask_id = prof_1d->FindBin( prof_1d->GetRandom() );
 
  Bool_t*** msk = m_msk[ mask_id - 1];

  ApplyMask( hits_in, out,  msk );
  hits_out->SetData( out );

  cout<<"X-view: before:"<<hits_in->GetNXHit()<<", after:"<<hits_out->GetNXHit()
      <<"Y-view: before:"<<hits_in->GetNYHit()<<", after:"<<hits_out->GetNYHit()<<endl;
}


void ApplyMask( pGPSSPEHits * hits_in, cGPSSPE* out, Bool_t *** mask){
  
  out->nstrpx = out->nstrpy = 0;

  for(Int_t k = 0; k<hits_in->GetNXHit(); k++){
    GPSSPEData * in_data = hits_in->GetXHit( k );
    if (IsStripXGood( in_data->fistrip, in_data->fnpstrip, in_data->fntstrip, mask)){
      out->qstripx[out->nstrpx] = in_data->fqstrip;
      out->xstripx[out->nstrpx] = in_data->fstripc;
      out->npstripx[out->nstrpx] = in_data->fnpstrip;
      out->ntstripx[out->nstrpx] = in_data->fntstrip;
      out->istripx[out->nstrpx] = in_data->fistrip;
      out->nstrpx++;
    }
  }
 
  for(Int_t k = 0; k<hits_in->GetNYHit(); k++){
    GPSSPEData * in_data = hits_in->GetYHit( k );
    if (IsStripYGood( in_data->fistrip, in_data->fnpstrip, in_data->fntstrip, mask)){
      out->qstripy[out->nstrpy] = in_data->fqstrip;
      out->ystripy[out->nstrpy] = in_data->fstripc;
      out->npstripy[out->nstrpy] = in_data->fnpstrip;
      out->ntstripy[out->nstrpy] = in_data->fntstrip;
      out->istripy[out->nstrpy] = in_data->fistrip;
      out->nstrpy++;
    }
  }
}

Bool_t IsStripXGood( Int_t istrip, Int_t npstrip, Int_t ntstrip, Bool_t *** mask){
  
  Int_t iladd = -1;
  Int_t iv = (npstrip-1)*2 + 1;
  if( (ntstrip==1) || (ntstrip==4)) iladd = 0; //1 F 
  if( (ntstrip==2) || (ntstrip==5)) iladd = 1; //2 F
  if( (ntstrip==3) || (ntstrip==6)) iladd = 2; //3 F  
  Int_t j_strip = 2*( istrip - iladd*1024 );
  Int_t ivkl = Int_t( (j_strip/2. - 1)/128. );
  return mask[iv][iladd][ivkl]; //0..11, 0..2, 0..7

}

Bool_t IsStripYGood( Int_t istrip, Int_t npstrip, Int_t ntstrip, Bool_t *** mask){
  
  Int_t iladd = -1;
  Int_t iv = (npstrip-1)*2;
  if( (ntstrip==1) || (ntstrip==4)) iladd = 0; //1 F 
  if( (ntstrip==2) || (ntstrip==5)) iladd = 1; //2 F
  if( (ntstrip==3) || (ntstrip==6)) iladd = 2; //3 F  
  Int_t j_strip = istrip - iladd*1024;
  Int_t ivkl = Int_t( (j_strip - 1)/128. );
  return mask[iv][iladd][ivkl]; //0..11, 0..2, 0..7

}

Bool_t **** MakeBoolOfMasks( TList* msk_in_list ){

  Bool_t ****m_msk = new Bool_t***[msk_in_list->GetEntries() - 1];
  for(Int_t m = 0; m<msk_in_list->GetEntries() - 1; m++){
    Bool_t ***vk_matrix = new Bool_t**[12];
    for(Int_t i=0; i<12; ++i){
      vk_matrix[i] = new Bool_t*[3];
      for(Int_t j=0; j<3; ++j)
    	vk_matrix[i][j] = new Bool_t[8];
    }
    
    VKmask * msk = (VKmask*)msk_in_list->At(m);
    msk->GetMaskBool( vk_matrix );
    m_msk[m] = vk_matrix;
  
    //Print given mask
    cout<<"--->MASK "<<m<<" printout: "<<endl;
    for(Int_t i=0; i<12; ++i){
      for(Int_t j=0; j<3; ++j){
	for(Int_t k=0; k<8; ++k){
	  cout<<vk_matrix[i][j][k]<<" ";
	}
      }
      cout<< msk->GetMaskRow(i)<<endl;
    }
    
    cout<<"<--- END OF MASK "<<m<<endl;
  }

  return m_msk;
}


void Usage(){
  cout<<"NAME"<<endl;
  cout<<"       vmc_dig - a standalone digitizer for PamVMC application"<<endl;
  cout<<"SYNOPSIS"<<endl;
  cout<<"       vmc_dig  INPROOTFILE OUTRAWFILE MASKSPROFILE IBIN"<<endl; 
  cout<<"DISCRIPTION"<<endl;
  cout<<"       This tool convert hits collections of PamVMC in raw format "<<endl; 
  cout<<"       There are only 4 parameters: input root/output raw filenames,"<<endl;
  cout<<"       input root-file with list of VKmask objects + TH2D histogram, "<<endl;
  cout<<"       and bin number which should be taken from TH2D to do  masks profile. "<<endl;
  cout<<"       Please note that your input file should contain RNDM branch"<<endl;
  cout<<"       ( produced with <all_detectors/> option in <save_mode> section )"<<endl;
}
