/*
 * TrkChi2DeflTimeCut.h
 *
 *  Created on: 6-oct-2009
 *      Author: S. Ricciarini
 */

/*! @file TrkChi2DeflTimeCut.h The TrkChi2DeflTimeCut class definition file */

#ifndef TRKCHI2DEFLTIMECUT_H_
#define TRKCHI2DEFLTIMECUT_H_

#include "../../PamCutBase/PamCutBase.h"

#include <TH2F.h>
#include <TTimeStamp.h>

/*! @brief The tracker chi2 vs |deflection| cut.
 *
 * Events whose track has been fitted with a chi2 greater than Chi2(|eta|) are discarded.
 * Chi2(|eta|) is a function which gives the value of chi2 distribution, corresponding to a given quantile (depending on the calibration text file), as a function of |deflection| (|eta|); in current implementation it is parameterized as:
 *
 *   4 parameters p0...p3:
 *
 *   Chi2(|eta|) = p0 + ( p1 * |eta|^p2 * (1 + (p3 * |eta|)^2) )
 *   with eta expressed in GV^-1
 *
 *   5 parameters p0..p4
 *
 *   Chi2(|eta|) = p0 + ( p1 * |eta|^p2 * (1 + (p3 * |eta|)^p4) )
 *   with eta expressed in GV^-1
 *
 * The parameters are read from a calibration text file (prepared for a given quantile, e.g. 95%), whose format must be:
 * ...
 * first_day	last_day	p0	p1	p2	...
 * with first_day and last_day expressed in the format YYMMDD
 * ...
 * where each row corresponds to a given time interval (first_day,last_day)
 * The calibration text file can vary with nHitX of the event: in the current implementation, a text file must be specified for events with nHitX=3 and for events with nHitX>=4.
 * For a given event the associated day is determined; the FIRST (starting from first row) time interval of the calibration file, which contains the event day, gives the parameters to be inserted in the Chi2(|eta|) for the given event.
 * If the event day does not fall in any of the intervals of the calibration file, then the event is discarded. 
 *
 */

class TrkChi2DeflTimeCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * @param cutName The cut's name.
   * @param calibFile_nHitX3 Path for the calibration text file, used for events with nHitX=3.
   * @param calibFile_nHitX4 Path for the calibration text file, used for events with nHitX>=4.  
   * @param nPar Number of parameters used for the Chi2 function. Maximum allowed: 5. Default: 5.
   */
  TrkChi2DeflTimeCut(const char *cutName, const char *calibFile_nHitX3, const char *calibFile_nHitX4, int nPar=5) :
    PamCut(cutName), _calibFile_nHitX3(calibFile_nHitX3), _calibFile_nHitX4(calibFile_nHitX4), _nPar(nPar) {
 
    _chi2CutTable_nHitX3.open(_calibFile_nHitX3);
    while (! _chi2CutTable_nHitX3.eof()) {
    
      _chi2CutTable_nHitX3 >> _tstringtemp;
      _iDayFirst_nHitX3.push_back(_tstringtemp.Atoi());

      _chi2CutTable_nHitX3 >> _tstringtemp;
      _iDayLast_nHitX3.push_back(_tstringtemp.Atoi());
      
      if (_nPar>=0 && _nPar<=5) {
        for (int iPar=0; iPar<_nPar; iPar++) {
      
          _chi2CutTable_nHitX3 >> _tstringtemp;
          _p_nHitX3[iPar].push_back(_tstringtemp.Atof());
	
        }
      }
            
    }
    _chi2CutTable_nHitX3.close();

    _chi2CutTable_nHitX4.open(_calibFile_nHitX4);
    while (! _chi2CutTable_nHitX4.eof()) {
    
      _chi2CutTable_nHitX4 >> _tstringtemp;
      _iDayFirst_nHitX4.push_back(_tstringtemp.Atoi());

      _chi2CutTable_nHitX4 >> _tstringtemp;
      _iDayLast_nHitX4.push_back(_tstringtemp.Atoi());
      
      if (_nPar>=0 && _nPar<=5) {
        for (int iPar=0; iPar<_nPar; iPar++) {
      
          _chi2CutTable_nHitX4 >> _tstringtemp;
          _p_nHitX4[iPar].push_back(_tstringtemp.Atof());
	
        }
      }
            
    }
    _chi2CutTable_nHitX4.close();


  }

  /*! @brief Destructor. */
  ~TrkChi2DeflTimeCut() {

  }

  /*! @brief The tracker chi2 vs |deflection| check.
   *
   *
   * @param event The event to analyze.
   * @return #CUTOK if chi2 < Chi2(|eta|).
   * @return 0 if not
   */
  int Check(PamLevel2 *event);

private:

    virtual double _GetChi2(PamLevel2 *event);

    const char* _calibFile_nHitX3;
    const char* _calibFile_nHitX4;
    Int_t _nPar;

    TString _tstringtemp;
    
    std::vector<Int_t> _iDayFirst_nHitX3;
    std::vector<Int_t> _iDayFirst_nHitX4;
    std::vector<Int_t> _iDayFirst;
    
    std::vector<Int_t> _iDayLast_nHitX3;
    std::vector<Int_t> _iDayLast_nHitX4;
    std::vector<Int_t> _iDayLast;
    
    std::vector<Double_t> _p_nHitX3[5];
    std::vector<Double_t> _p_nHitX4[5];
    std::vector<Double_t> _p[5];
    
    ifstream _chi2CutTable_nHitX3;
    ifstream _chi2CutTable_nHitX4;
    
    Double_t _pSel[5];
    
    TTimeStamp _time;

};
#endif /* TRKCHI2DEFLTIMECUT_H_ */
