/*
 * TrkChi2DeflCut.h
 *
 *  Created on: 7-apr-2009
 *      Author: Nicola Mori
 */

/*! @file TrkChi2DeflCut.h The TrkChi2DeflCut class definition file */

#ifndef TRKCHI2DEFLCUT_H_
#define TRKCHI2DEFLCUT_H_

#include "../../PamCutBase/PamCutBase.h"

#include <TH2F.h>

/*! @brief The tracker chi2 vs deflection cut.
 *
 * Events whose track has been fitted with a chi2 greater than Chi2(eta) are discarded.
 * Chi2(eta) is a function which gives the maximum value of chi2 associated to a certain
 * deflection eta; in current implementation it is parameterized as:
 *
 *   Chi2(eta) = p0 + p1 * eta^2 + p2 * eta ^4
 *
 * where p0, p1 and p2 are parameters.
 */

class TrkChi2DeflCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * @param cutName The cut's name.
   * @param p0 The constant term in Chi2(eta).
   * @param p1 The coefficient or the quadratic term in Chi2(eta).
   * @param p2 The coefficient or the quartic term in Chi2(eta).
   */
  TrkChi2DeflCut(const char *cutName, float p0, float p1, float p2) :
    PamCut(cutName), _p0(p0), _p1(p1), _p2(p2) {

#ifdef DEBUGPAMCUT

    TString hId;
    TString hTitle;

    for (UInt_t j = 0; j < 2; j++) {
      hId.Form("h_trk_chi2_vs_defl_%i_", j);
      hId.Append(TString(cutName));
      hTitle.Form("TRK chi2 vs defl (%i)", j);
      h_trk_chi2_defl[j] = new TH2F(hId.Data(), hTitle.Data(), 50, 0, 2.5, 50, 0, 20);
    }

#endif

  }

  /*! @brief Destructor. */
  ~TrkChi2DeflCut() {

#ifdef DEBUGPAMCUT

    for (UInt_t j = 0; j < 2; j++) {
      h_trk_chi2_defl[j]->Write();
    }

#endif

  }

  /*! @brief The tracker chi2 vs deflection check.
   *
   *
   * @param event The event to analyze.
   * @return #CUTOK if chi2 < Chi2(eta).
   * @return 0 if not
   */
  int Check(PamLevel2 *event);

private:

#ifdef DEBUGPAMCUT

  TH2F* h_trk_chi2_defl[2];

#endif

  float _p0, _p1, _p2;
};
#endif /* TRKCHI2DEFLCUT_H_ */
