/*
 * LTGeoFillCut.h
 *
 *  Created on: 26-mar-2009
 *      Author: Nicola Mori
 *  Last update: 30-apr-2009, S. Ricciarini
 */

/*! @file LTGeoFillCut.h The LTGeoFillCut class definition file */

#ifndef LTGEOFILLCUT_H_
#define LTGEOFILLCUT_H_

#include "../../PamCutBase/PamCutBase.h"

/*! @brief The LT histogram filling.
 *
 * This cut fills a Live Time (LT) histogram (in seconds) binned in threshold rigidity (in GV), given by
 * the geomagnetic cutoff rigidity (the Stoermer cutoff) for that event multiplied by a threshold
 * coefficient. It discards an event only if its threshold rigidity is outside the limits of the
 * histogram.
 *
 */
class LTGeoFillCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * @param cutName The cut's name.
   * @param binning A vector containing the histogram binning in threshold rigidity.
   *                Note that it must contain both the upper and lower limits, and
   *                that the elements must be ordered (ie., lowest value in the first
   *                element and so on).
   * @param thresholdCoeff The threshold rigidity coefficient for the event selection.
   */
  LTGeoFillCut(const char *cutName, std::vector<float> binning, float thresholdCoeff) :
    PamCut(cutName), _binning(binning), _thresholdCoeff(thresholdCoeff), _histogram(binning.size() - 1, 0.),
        _histogram_tot(0.) {
  }
  /*! @brief Destructor. */
  ~LTGeoFillCut() {
  }

  /*! @brief The threshold rigidity check.
   *
   * The event is discarded if its threshold rigidity is less than the lower
   * histogram limit or higher than the upper limit.
   *
   * @param event The event to analyze.
   * @return #CUTOK if the threshold rigidity is contained in the histogram limits.
   * @return 0 otherwise.
   */
  int Check(PamLevel2 *event);

  /*! @brief The histogram filling.
   *
   * The method fills the LT histogram (in seconds) with the currently selected event.
   *
   * @param event The currently selected event.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Returns the histogram.
   *
   * This method returns a vector filled with the LT (in seconds) corresponding to each
   * threshold rigidity bin (in GV) defined in the binning argument of the constructor.
   * Element 0 is the total LT for events whose threshold rigidity lies in the
   * lowest bin and so on.
   *
   * @return The LT histogram binned in threshold rigidity.
   */
  std::vector<float> &GetHisto() {
    return _histogram;
  }

  /*! @brief Returns the histogram total.
   *
   * This method returns the LT (in seconds) summed over all threshold rigidity bins (in GV) defined in the binning argument of the constructor.
   *
   * @return The LT histogram total.
   */
  float GetHistoTot() {
    return _histogram_tot;
  }

private:

  std::vector<float> _binning;
  float _thresholdCoeff;

  std::vector<float> _histogram;
  float _histogram_tot;

};

#endif /* LTGEOFILLCUT_H_ */
