/*
 * LTQualCut.h
 *
 *  Created on: 10-mar-2009
 *      Author: Nicola Mori, S. Ricciarini
 */

/*! @file LTQualCut.h The LTQualCut class definition file */

#ifndef LTQUALCUT_H_
#define LTQUALCUT_H_

#include "../../PamCutBase/PamCutBase.h"

/*! @enum LT_Return Return values for rejected events */
enum LT_Return {
  LT_NEWRUN, ///< Discarded because it is the first event of the run.
  LT_INVALID
///< Discarded because the LT is greater than timeout.
};

/*! @brief The live-time data quality cut: rejects the first event of each run and events with LT greater than an upper threshold (or with LT smaller than 0).
 */
class LTQualCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * @param cutName The cut's name.
   * @param threshold The live time maximum accepted value in ms. It is the value above which a live time is
   *        considered invalid. It must be less than the trigger timeout on IDAQ board, which
   *        is ~ 4600 ms. Default is 4500 (use of different values must be justified).
   */
  LTQualCut(const char* cutName, float threshold=4500) :
    PamCut(cutName), _previousRun(numeric_limits<UInt_t>::max()), _LT(-1.), _timeout(threshold) {
  }
  /*! @brief Destructor. */
  ~LTQualCut() {
  }

  /*! @brief The live time quality check.
   *
   * @param event The event to analyze.
   * @return #CUTOK if LT < timeout.
   * @return #LT_INVALID if LT > timeout.
   * @return #LT_NEWRUN if the event is at the beginning of the run.
   */
  int Check(PamLevel2 *event);

  /*! @brief Returns the live time of the last examined event.
   *
   * @return The live time of the last examined event in ms (-1 -> first event of the run).
   */
  float GetLT() { return _LT; }

private:
  unsigned int _previousRun;
  float _LT;
  float _timeout;
};

#endif /* LTQUALCUT_H_ */
