/*
 * SmartCollection.h
 *
 *  Created on: 14-mag-2009
 *      Author: Nicola Mori
 */

/*! @file SmartCollection.h The SmartCollection class definition file */

#ifndef SMARTCOLLECTION_H_
#define SMARTCOLLECTION_H_

#include "../../PamCutBase/PamCutBase.h"
#include "../../CollectionActions/CollectionAction/CollectionAction.h"

/*! @brief A collection class designed to use CollectionAction objects.
 *
 * The SmartCollection class is designed to handle CollectionAction objects. These
 * defines the procedures to do when an event is selected or discarded. A SmartCollection
 * handles a vector of these objects, calling CollectionAction::OnGood() for each of them
 * when a good event is selected and CollectionAction::OnBad() when a bad one is rejected.
 * It will also call the CollectionAction::Setup() and CollectionAction::Finalize() methods
 * at the beginning and at the end of the analysis, respectively.
 *
 */
class SmartCollection: public PamCutCollection {

public:

  /*! @brief Constructor.
   *
   * @param cutName The cut's name.
   */
  SmartCollection(const char* cutName) :
    PamCutCollection(cutName), _actions(0) {
  }

  /*! @brief Destructor. */
  ~SmartCollection(){
  }

  /*! @brief Adds an action to the SmartCollection */
  virtual void AddAction(CollectionAction& action);

  /*! @brief Returns the iAction-th action.
   *
   * @param iAction The index of the desired CollectionAction, defined as the insertion order
   *        (from 0 to \#actions-1, see AddAction()).
   * @return pointer to the iAction-th action; NULL if the specified action cannot be found or if no actions are present.
   */
  CollectionAction *GetAction(unsigned int iAction);

  /*! @brief The pre-analysis task definition.
   *
   * This override of the Setup() method calls Setup() for the base class PamCutCollection, and subsequently for each
   * action contained in the SmartCollection.
   *
   * @param events The PamLevel2 pointer to the events that will be analyzed. Used only as parameter for
   *          CollectionAction::Setup().
   */
  void Setup(PamLevel2 *events);

  /*! @brief The post-analysis task definition.
   *
   * This override of the Finalize() method calls PamCutCollection::Finalize() and then the Finalize() method of
   * each action contained in the SmartCollection.
   */
  void Finalize();

  /*! @brief Post-selection tasks.
   *
   * This routine is automatically called after a good event has been selected by
   * ApplyCut(). It will simply call PamCutCollection::OnGood() and then CollectionAction::OnGood() for each
   * action in the SmartCollection.
   * @param event The event which satisfy the cut.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Post-selection tasks.
   *
   * This routine is automatically called after a bad event has been rejected by
   * ApplyCut(). It will simply call PamCutCollection::OnBad() and then CollectionAction::OnBad() for each
   * action in the SmartCollection.
   *
   * @see OnGood
   * @param event The event which don't satisfy the cut.
   * @param selectionResult The return value of the Check() routine.
   */
  void OnBad(PamLevel2 *event, int selectionResult);

private:

  std::vector<CollectionAction*> _actions;

};

#endif /* SMARTCOLLECTION_H_ */
