/*
 * BinnedEffCollection.h
 *
 *  Created on: 12/lug/2011
 *      Author: Nicola Mori
 */

/*! @file BinnedEffCollection.h The BinnedEffCollection class definition file. */

#ifndef BINNEDEFFCOLLECTION_H_
#define BINNEDEFFCOLLECTION_H_

#include "../EffCollection/EffCollection.h"

/*! @brief A binned efficiency collection.
 *
 * This abstract class provides methods for computing a binned efficiency. The binning variable is
 * not defined here but must be specified in children concrete classes by implementing the pure
 * virtual method #GetBinValue. For example, to bin the efficiency using rigidity it is sufficient
 * to implement #GetBinValue so that it returns the rigidity of current event. Likewise for beta or
 * any other possible variable (calorimeter energy, rigidity from beta and so on).
 *
 * This class inherits from #EffCollection: it has selection and detector cuts and actions, and it
 * computes and saves the integrated detector efficiency. Plus, it will compute and save the efficiency
 * for each bin given to the constructor. The output consists of text files and a ROOT file
 * with the efficiency graph.
 */
class BinnedEffCollection: public EffCollection {

public:

  /*! @brief Constructor.
   *
   * @param collectionName The collection's name. The graph in the ROOT output will have the same name, so be
   *                       careful to name the collection respecting the C++ rules for the names of the variables
   *                       (eg. don't use - in the name) otherwise the graph will be unusable.
   * @param outFileBase The output file base name. It will produce the same file as the parent class #EffCollection
   *                    with the total efficiency, plus a similar file (named outFileBase + collection's name + "-rig.txt")
   *                    with a row for each rigidity bin. At the beginning of each row there will be two additional columns
   *                    with the edges of the bin. Additionally, a ROOT file (outFileBase + collection's name +"-rig.root")
   *                    will be produced, containing the efficiency graph.
   * @param binsFile The file with the bins.
   * @param errMethod The method to use for error computation. Possible values are defined in #EffCollection_ErrMethod.
   * @param owns If true, the collection will own the cuts and the actions, ie., it will
   *             destroy them in its destructor.
   */
  BinnedEffCollection(const char *collectionName, TString outFileBase = "", TString binsFile = "", int errMethod =
      EFFERR_ROOT, bool owns = true);

  /*! @brief Destructor. */
  ~BinnedEffCollection() {

  }

  /*! @brief Applies the selection and detector cuts to the current event.
   *
   * @param event The event to analyze.
   * @return CUTOK if the event survives all the selection and detector cuts.
   */
  int ApplyCut(PamLevel2 *event);

  /*! @brief The post analysis task.
   *
   */
  void Finalize();

  /*! @brief Returns the value of the binning parameter for current event.
   *
   * This method has to be implemented in children classes, and must return the value of the binning parameter (rigidity, beta,
   * energy etc.) for current event.
   *
   * @param event The current event.
   * @return The binning value.
   */
  virtual float GetBinValue(PamLevel2 *event) = 0;

private:

  vector<float> _bins;
  vector<unsigned int> _selVector;
  vector<unsigned int> _detVector;

  unsigned int _outUp, _outDown;

};

#endif /* BINNEDEFFCOLLECTION_H_ */
