/*
 * SmartCollection.h
 *
 *  Created on: 14-mag-2009
 *      Author: Nicola Mori
 */

/*! @file SmartCollection.h The SmartCollection class definition file */

#ifndef SMARTCOLLECTION_H_
#define SMARTCOLLECTION_H_

#include "../../PamCutBase/PamCutBase.h"
#include "../../CollectionActions/CollectionAction/CollectionAction.h"

/*! @brief A collection class designed to use CollectionAction objects.
 *
 * The SmartCollection class is designed to handle CollectionAction objects. It
 * holds a vector of these objects and takes care of calling Setup and Finalize for
 * each of them at the beginning and at the end of the analysis, respectively.
 * Actions can be added to the SmartCollection by means of the AddAction() method.
 * If a bunch of cuts have been already added to the collection, the action will
 * be logically placed after the cuts. The SmartCollection will call
 * CollectionAction::OnGood() if the cuts preceding the actions are satisfied, and
 * CollectionAction::OnBad() if at least one of them fails. An action will not be
 * sensitive to cuts added to the collection after the action itself.
 * The resulting structure is composed by bunches of cuts intertwined by actions,
 * which are "executed" depending on the result of the bunch of cuts that precedes the
 * action. Note that CollectionAction::OnBad() is called only for those actions at
 * the end of the bunch where the first failed cut is: if after these actions there
 * are other bunches of cuts and actions, they will be ignored.
 * For example, in the sequence below:
 *
 * | cut1 | cut2 | action1 | action2 | cut3 | cut4 | action3 | ...
 *
 * action1 and action2 are executed (eg., OnGood is called for them) if cut1 and cut2
 * are both satisfied, then cut3 and cut4 are evaluated and if both of them are satisfied
 * then action3 is executed. If, for example, cut 1 or cut2 fail, then OnBad is called for
 * action1 and action2; however, cut3, cut4, action3 and all that follows are ignored. The
 * analysis goes on with the next event.
 */
class SmartCollection: public PamCutCollection {

public:

  /*! @brief Constructor.
   *
   * @param collectionName The collection's name.
   * @param owns If true, the collection will own the cuts and the actions, ie., it will
   *             destroy them in its destructor.
   */
  SmartCollection(const char* collectionName, bool owns = true) :
    PamCutCollection(collectionName, owns), _actions(0) {
  }

  /*! @brief Destructor. */
  ~SmartCollection();

  /*! @brief Adds an action to the SmartCollection */
  virtual void AddAction(CollectionAction *action);

  /*! @brief Returns the iAction-th action.
   *
   * @param iAction The index of the desired CollectionAction, defined as the insertion order
   *        (from 0 to \#actions-1, see AddAction()).
   * @return pointer to the iAction-th action; NULL if the specified action cannot be found or if no actions are present.
   */
  CollectionAction *GetAction(unsigned int iAction);

  /*! @brief Searches for an action by name.
   *
   * @param actionName The name of the action to search for.
   * @return pointer to the desired action; NULL if the specified action cannot be found or if no actions are present.
   */
  CollectionAction *GetAction(const char *actionName);

  /*! @brief The number of actions in the collection.
   *
   * @return The number of actions currently in the collection.
   */
  unsigned int GetActionsSize() {
    return _actions.size();
  }

  /*! @brief The pre-analysis task definition.
   *
   * This override of the Setup() method calls Setup() for the base class PamCutCollection, and subsequently for each
   * action contained in the SmartCollection.
   *
   * @param events The PamLevel2 pointer to the events that will be analyzed. Used only as parameter for
   *          CollectionAction::Setup().
   */
  void Setup(PamLevel2 *events);

  /*! @brief The post-analysis task definition.
   *
   * This override of the Finalize() method calls PamCutCollection::Finalize() and then the Finalize() method of
   * each action contained in the SmartCollection.
   */
  void Finalize();

  /*! Applies the cuts and executes the actions.
   *
   * When cuts are applied, a SmartCollection will also execute the actions at the end of the bunches of cuts.
   *
   * @param event The event to analyze.
   */
  int ApplyCut(PamLevel2 *event);

protected:

  /*! @brief The vector containing the actions */
  std::vector<CollectionAction*> _actions;

  /*! The vector of actions' positions
   *
   * An action is placed after N cuts, so that the SmartCollection, after applying N cuts, has to
   * perform the action. The numbers N for each action are stored here: the element i is N for the
   * i-th action in #_actions.
   */
  std::vector<int> _actionsPositions;

};

#endif /* SMARTCOLLECTION_H_ */
