/*
 * EffCollection.h
 *
 *  Created on: 10/ago/2009
 *      Author: Nicola Mori
 */

/*! @file EffCollection.h The EffCollection class definition file. */

#ifndef EFFCOLLECTION_H_
#define EFFCOLLECTION_H_

#include "../VerboseCollection/VerboseCollection.h"

/*! @enum EffCollection_ErrMethod to select the method of error computation */
enum EffCollection_ErrMethod {
  EFFERR_SERGIO, ///< Flag for Sergio Ricciarini's Fortran routine
  EFFERR_ROOT
///< Flag for ROOT TGraphAsymErrors::BayesDivide()
};

/*! @brief A verbose collection which computes the efficiency of a set of cuts.
 *
 * This class subdivides the cuts it contains into two classes: selection and detector
 * cuts; detector cuts are evaluated after selection cuts. The collection will compute the
 * efficiency of the whole detector cuts set evaluated using an efficiency sample selected
 * by the selection cuts. Actions will be divided in selection and detector actions as well:
 * adding a selection action, it will be placed after all the selection cuts inserted since
 * the action insertion, and before all detector cuts. Detector actions are placed after all
 * the cuts.
 * This class implements specific methods to add selection and detector cuts; cuts added
 * with the standard #AddCut method will be treated as detector cuts. The same for actions.
 * Error computation is done either using Sergio Ricciarini's Fortran routine or ROOT's
 * TGraphAsimmErrors::BayesDivide(). For both methods, the efficiency is arbitrarily set to
 * 1.1 when less than 8 events survive the selection cuts. This is needed to avoid convergence
 * problems in Sergio's routine, and the ROOT output has been consequently adapted to this
 * convention.
 *
 */
class EffCollection: public VerboseCollection {

public:

  /*! @brief Constructor.
   *
   * @param collectionName The collection's name.
   * @param outFileBase The output file base name. If "", no file output will be produced; otherwise,
   *                    a file named outFilebase + collection's name + ".txt" will be produced, containing
   *                    the number of events surviving the detector cuts (1st column), the selection cuts (2nd column),
   *                    the efficiency (3rd column), the lower (4th column) and upper (5th column) length
   *                    of the efficiency's error bar.
   * @param errMethod The method to use for error computation. Possible values are defined in #EffCollection_ErrMethod.
   * @param owns If true, the collection will own the cuts and the actions, ie., it will
   *             destroy them in its destructor.
   */
  EffCollection(const char *collectionName, TString outFileBase = "", int errMethod = EFFERR_ROOT, bool owns = true);

  /*! @brief Destructor. */
  ~EffCollection() {

  }

  /*! @brief Adds a detector cut to the collection.
   *
   * For EffCollection, cuts added wit #AddCut will be treated as detector cuts.
   *
   * @param cut The PamCut-derived object to add to the collection.
   */
  void AddCut(PamCut *cut) {
    AddDetectorCut(cut);
  }

  /*! @brief Adds a detector cut to the collection.
   *
   * Adds a detector cut. Calling #AddCut will add a detector cut as well, and not a
   * selection cut.
   * @param cut The PamCut-derived object to add to the collection.
   */
  void AddDetectorCut(PamCut *cut);

  /*! @brief Adds a selection cut to the collection.
   *
   * Adds a selection cut. Notice that calling #AddCut will add a detector cut , and not a
   * selection cut.
   * @param cut The PamCut-derived object to add to the collection.
   */
  void AddSelectionCut(PamCut *cut);

  /*! @brief Adds an action to the detector cuts queue.
   *
   * For EffCollection, actions added wit #AddAction will be inserted in the detector cuts queue.
   *
   * @param action The CollectionAction-derived object to add to the collection.
   */
  void AddAction(CollectionAction *action) {
    AddDetectorAction(action);
  }

  /*! @brief Adds an action to the detector cuts queue.
   *
   * Adds a detector action, ie., an action placed in the detector cuts queue. Calling #AddAction will
   * add a detector action as well, and not a selection action.
   *
   * @param action The CollectionAction-derived object to add to the collection.
   */
  void AddDetectorAction(CollectionAction *action);

  /*! @brief Adds an action to the selection cuts queue.
   *
   * Adds a selection action, ie., an action placed in the selection cuts queue. Notice that calling
   * #AddAction will add a detector action, and not a selection action.
   *
   * @param action The CollectionAction-derived object to add to the collection.
   */
  void AddSelectionAction(CollectionAction *action);


  /*! @brief The pre-analysis task definition.
   *
   * This override of the Setup() method sets up the selection and detector cuts and actions
   *
   * @param events The PamLevel2 pointer to the events that will be analyzed. Used only as parameter for
   *          VerboseCollection::Setup().
   */
  void Setup(PamLevel2 *events);

  /*! @brief Applies the selection and detector cuts to the current event.
    *
    * @param event The event to analyze.
    * @return CUTOK if the event survives all the selection and detector cuts.
    */

  int ApplyCut(PamLevel2 *event);

  /*TODO: redefine GetCut and the other methods to comply with the new selection/detector cuts structure. */

  /*! @brief The post analysis task.
   *
   */
  void Finalize();

protected:

  /*! This collection contains the selection cuts. */
  SmartCollection _selCollection;

  /*! This collection contains the detector cuts. */
  SmartCollection _detCollection;

  /*! The base name of the output file. */
  TString _outFileBase;

  /*! The method used for error computation. */
  int _errMethod;

  /*! The number of events surviving the detector cuts. */
  unsigned int _det;

  /*! The number of events surviving the selection cuts. */
  unsigned int _sel;

};

#endif /* EFFCOLLECTION_H_ */
