/*
 * EffRigCollection.h
 *
 *  Created on: 12/ago/2009
 *      Author: Nicola Mori
 */

/*! @file EffRigCollection.h The EffRigCollection class definition file. */

#ifndef EFFRIGCOLLECTION_H_
#define EFFRIGCOLLECTION_H_

#include "../BinnedEffCollection/BinnedEffCollection.h"

/*! @brief A rigidity binned efficiency collection
 *
 * This class inherits from #BinnedEffCollection: it only implements the #GetBinValue method
 * to return event rigidity with sign.
 *
 * NOTE: to divide the events in rigidity bins, a physical track is required from which the
 * rigidity can be calculated. It is MANDATORY to add a #TrkPhSinCut object to the selection
 * cuts, to avoid problems when computing rigidity (it will be done if the event survives the
 * selection cuts, so a #TrkPhSinCut in selection cuts will ensure a physical track being present
 * at the moment of rigidity computation).
 *
 */
class EffRigCollection: public BinnedEffCollection {

public:

  /*! @brief Constructor.
   *
   * @param collectionName The collection's name. The graph in the ROOT output will have the same name, so be
   *                       careful to name the collection respecting the C++ rules for the names of the variables
   *                       (eg. don't use - in the name) otherwise the graph will be unusable.
   * @param outFileBase The output file base name. It will produce the same file as the parent class #EffCollection
   *                    with the total efficiency, plus a similar file (named outFileBase + collection's name + "-rig.txt")
   *                    with a row for each rigidity bin. At the beginning of each row there will be two additional columns
   *                    with the edges of the bin. Additionally, a ROOT file (outFileBase + collection's name +"-rig.root")
   *                    will be produced, containing the efficiency graph.
   * @param rigBinsFile The file with the rigidity bins.
   * @param errMethod The method to use for error computation. Possible values are defined in #EffCollection_ErrMethod.
   * @param owns If true, the collection will own the cuts and the actions, ie., it will
   *             destroy them in its destructor.
   */
  EffRigCollection(const char *collectionName, TString outFileBase = "", TString rigBinsFile = "", int errMethod =
      EFFERR_SERGIO, bool owns = true);

  /*! @brief Destructor. */
  ~EffRigCollection() {

  }

  /*! @brief The value of binning parameter (rigidity).
   *
   * @param event The current event.
   * @return The rigidity (with sign) of current event.
   */
  float GetBinValue(PamLevel2 *event);
};

#endif /* EFFRIGCOLLECTION_H_ */
