/*
 * ReprocessTrackAction.h
 *
 *  Created on: 23/dic/2009
 *      Author: N. Mori
 */

#ifndef REPROCESSTRACKACTION_H_
#define REPROCESSTRACKACTION_H_

#include "../CollectionAction/CollectionAction.h"

/*! @brief Level 1 reprocessing
 *
 * This action reprocesses the level 1 to obtain a new level 2. The level 1 can be recomputed
 * from level 0 or provided externally. Remember that the standard PamLevel2::GetPamTree needs to
 * be instructed explicitly to open level 0 informations for tracker, by appending "+TRK0" to its 3rd
 * parameter. Make sure that the PamLevel2 object you will use with this action (or generally, with every
 * code that needs level 0) has been correctly initialized.
 */
class ReprocessTrackAction: public CollectionAction {
public:

  /*! @brief Constructor
   *
   * @param actionName The action's name.
   * @param newTrkL1 level1 computed outside the action. If NULL, the action will compute the level 1
   *        starting from level 0.
   * @param externalFlag This can be used to control externally if the event has to be
   *                     reprocessed or not. If NULL, all the events will be reprocessed. If it
   *                     points to a bool variable, an event will be reprocessed only if this variable
   *                     is true.
   * @param reprocL2 If false, only the level 1 will be recomputed (if newTrkL1 is NULL, otherwise nothing is done).
   */
  ReprocessTrackAction(const char *actionName, TrkLevel1 **newTrkL1 = NULL, bool *externalFlag = NULL, bool reprocL2 = true) :
    CollectionAction(actionName), _newTrkL1(newTrkL1), _externalFlag(externalFlag), _reprocL2 (reprocL2), _allEvents(0), _reprocessed(0),
        _lostTracks(0), _recoveredTracks(0) {

  }

  /*! @brief Destructor. */
  ~ReprocessTrackAction() {
    //delete _newTrkL1; // This causes a double delete (I suppose...)
  }

  /* @brief The reprocessing procedure.
     *
     * This will reprocess tracker information, depending on the flags set in constructor's parameter list.
     */
  void OnGood(PamLevel2 *event);

  /*! @brief The finalization procedure.
    *
    * Produces a screen output of the refitting statistics (reprocessed events, recovered tracks and so on).
    */
  void Finalize() {
    cout << "\n----> Reprocessing stats:\n";
    cout << "        events     : " << _allEvents << "\n";
    cout << "        reprocessed: " << _reprocessed << "\n";
    cout << "       Tracks:\n";
    cout << "        lost       : " << _lostTracks << "\n";
    cout << "        recovered  : " << _recoveredTracks << endl;

  }

private:

  TrkLevel1 **_newTrkL1;
  bool *_externalFlag;
  bool _reprocL2;
  unsigned int _allEvents, _reprocessed, _lostTracks, _recoveredTracks;

};

#endif /* REPROCESSTRACKACTION_H_ */
