/*
 * RigFillAction.h
 *
 *  Created on: 14/lug/2009
 *      Author: Nicola Mori
 */

/*! @file RigFillAction.h The RigFillAction class declaration file. */

#ifndef RIGFILLACTION_H_
#define RIGFILLACTION_H_

#include "../Histo2DAction/Histo2DAction.h"
//#include <stdint.h>

/*! @brief The rigidity vs threshold rigidity histogram filling.
 *
 * This class builds a 2D histogram binned in event rigidity modulus (Y axis)
 * and threshold rigidity (X axis). Each 2D bin will contain the number of events whose
 * event rigidity modulus and threshold rigidity (eg., Stoermer cutoff rigidity times a
 * threshold coefficient) lie in that bin. Note that the meaning of this threshold coefficient
 * is the same as in TrkRigGeoCut, so it must have the same value used for TrkRigGeoCut
 * (consider using TrkRigGeoCut::GetThresholdCoeff() to retrieve its value).
 * Events whose threshold rigidity is below the lower limit of the bins are recorded in a vector
 * of event rigidities with threshold rigidity in the "inferior threshold bin", which ranges
 * from zero to the the lowest threshold rigidity of the "normal" bins.
 * It is also possible to recover spillover events. These are defined as events with opposite sign
 * and absolute rigidity greater than a certain value (specified as a constructors' parameter).
 * They will be added to the rigidity overflow bins (YOverflow).
 * Another functionality of this action is the correction of deflection for residual misalignment when
 * selecting data with a fixed MDR cut (which reduces the effective tracker geometry and thus brings up
 * some residual misalignment).
 * For output file naming conventions, see #Finalize.
 *
 * CUT DEPENDENCIES: TrkPhSinCut for single physical track, TrkRigGeoCut for galactic event.
 *
 */
class RigFillAction: public Histo2DAction<Float_t> {

public:
  /*! @brief Constructor.
   *
   * outFileBase is the base name for output file: #Finalize will add ".txt" for the matrix histogram,
   * "-InfBins.txt" for the critical rigidity underflow and "-Spillover.txt" for the (spillover + rigidity overflow).
   * outFileBase has to contain the path (otherwise, files will be saved in the current directory).
   * The file containing the rigidity bins must be a text file. It must contain both the
   * lower and upper limits of the rigidity axis, so that if it contains N values it
   * defines a set of N-1 bins.
   *
   * @param actionName The action's name.
   * @param outFileBase The output file base name. If "" is given as name, no file will be produced.
   * @param rigBinsFile The file containing the rigidity bins
   * @param thresholdCoeff The threshold coefficient for critical rigidity.
   * @param chargeSign The sign of the rigidity of the particles that will fill the histograms. The
   *                   particles with opposite sign will be discarded (except,for spillover events, eventually).
   *                   See also aliases for positive and negative signs in CommonDefs.h.
   * @param spilloverFlag If true, spillover events (defined by spilloverLimit) will be not discarded and
   *                      added to the highest bin.
   * @param spilloverLimit The inferior limit of the spillover rigidities (absolute value).
   * @param mdrMin The fixed MDR cut (in GV) applied to data before this action, and for which the measured rigidity
   *               will be corrected. See the implementation of #OnGood for details about the correction calculation.
   *               No correction will be applied if this parameter is set to 0.
   *
   */
  RigFillAction(const char *actionName, TString outFileBase, TString rigBinsFile, float thresholdCoeff,
      float chargeSign = POSITIVE, bool spilloverFlag = false, float spilloverLimit = 0, float mdrMin = 0.);

  /*! @brief Constructor.
   *
   * outFileBase is the base name for output file: #Finalize will add .txt for ASCII output
   * and .root for ROOT output. "-report" will be also added for the report file (in which
   * the content of the zero bins will be saved). outFileBase has to contain the path (otherwise,
   * files will be saved in the current directory).
   *
   * @param actionName The action's name.
   * @param outFileBase The output file base name. If "" is given as name, no file will be produced.
   * @param bins A vector containing the bins limits.
   * @param thresholdCoeff The threshold coefficient for critical rigidity.
   * @param chargeSign The sign of the rigidity of the particles that will fill the histograms. The
   *                   particles with opposite sign will be discarded (except for spillover events, eventually).
   * @param spilloverFlag If true, spillover events (defined by spilloverLimit) will be not discarded and
   *                      added to the highest bin.
   * @param spilloverLimit The inferior limit of the spillover rigidities (absolute value).
   * @param mdrMin The fixed MDR cut (in GV) applied to data before this action, and for which the measured rigidity
   *               will be corrected. See the implementation of #OnGood for details about the correction calculation.
   *               No correction will be applied if this parameter is set to 0.
   */
  RigFillAction(const char *actionName, TString outFileBase, vector<float> &bins, float thresholdCoeff,
      float chargeSign = POSITIVE, bool spilloverFlag = false, float spilloverLimit = 0, float mdrMin = 0.);

  /*! @brief Destructor */
  ~RigFillAction() {
  }

  /*! @brief Fills histogram with the selected event.
   *
   * The current event will be added to the event rigidity bin corresponding to its event rigidity R and to the
   * threshold bin corresponding to the Stoermer cutoff rigidity (S) multiplied by the threshold coefficient.
   *
   * @param event The selected event.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Writes the output files.
   *
   * The output consists of: 1) a text file where the 2-dimensional histogram is saved. The first row of the
   * text file is the lowest rigidity bin, the second is the next bin and so on, so in the text output the positive
   * direction of the rigidity axis (Y axis) is downwards; 2) a text file with postfix "-InfBins.txt" where the
   * critical rigidity underflow array is saved (it will contain a number of entries equal to the number of bins);
   * 3) eventually, a text file with postfix "-Spillover.txt" where the (spillover + rigidity overflow) array is saved
   * (this array will contain one element more than the previous one, which is the (spillover + rigidity overflow) for the
   * critical rigidity underflow, and will be the first).
   */
  void Finalize();

private:

  float _thresholdCoeff;
  float _chargeSign;
  bool _spilloverFlag;
  float _spilloverLimit;
  float _mdrMin;
};
#endif /* RIGFILLACTION_H_ */
