/*
 * LiveTimeAction.h
 *
 *  Created on: 13/lug/2009
 *      Author: Nicola Mori
 */

/*! @file LiveTimeAction.h The LiveTimeAction class declaration file */

#ifndef LIVETIMEACTION_H_
#define LIVETIMEACTION_H_

#include "../Histo1DAction/Histo1DAction.h"

/*! @brief An action that fills a live time histogram.
 *
 * This action reads a rigidity binning from a file and fills a live time histogram (text and ROOT format).
 * The live time corresponding to each event will contribute to the bin corresponding to the cutoff rigidity
 * of the event, eg., the lowest bin whose lower limit is greater than the cutoff rigidity of the event times
 * a coefficient (threshold coefficient, see constructor). If threshold*cutoff is lower than the
 * lower limit of the rigidity axis, then the live time will be added to the "inferior bin", eg., a special bin
 * ranging from 0 to the lower limit. This special bin will be saved in a separate file.
 *
 */
class LiveTimeAction: public Histo1DAction<Double_t> {

public:
  /*! @brief Constructor.
   *
   * outFileBase is the base name for output file: #Finalize will add #GetName() + ".txt" for ASCII output
   * and #GetName() + ".root" for ROOT output. #GetName + "-report.txt" will be also added for the report file (in which
   * the content of the inferior bin will be saved). outFileBase has to contain the path (otherwise,
   * files will be saved in the current directory).
   * The file containing the rigidity bins must be a text file. It must contain both the
   * lower and upper limits of the rigidity axis, so that if it contains N values it
   * defines a set of N-1 bins.
   *
   * @param actionName The action's name.
   * @param outFileBase The output file base name. If "" is given as name,
   *                    no file output will be performed.
   * @param rigBinsFile The file containing the rigidity bins.
   * @param thresholdCoeff The threshold rigidity coefficient.
   */
  LiveTimeAction(const char *actionName, TString outFileBase, TString rigBinsFile, float thresholdCoeff);

  /*! @brief Constructor.
   *
   * outFileBase is the base name for output file: #Finalize will add .txt for ASCII output
   * and .root for ROOT output. "-report" will be also added for the report file (in which
   * the content of the inferior bin will be saved). outFileBase has to contain the path (otherwise,
   * files will be saved in the current directory).
   *
   * @param actionName The action's name.
   * @param outFileBase The output file base name. If "" is given as name,
   *                    no file output will be performed.
   * @param bins A vector containing the bins limits.
   * @param thresholdCoeff The threshold rigidity coefficient.
   */
  LiveTimeAction(const char *actionName, TString outFileBase, vector<float> &bins, float thresholdCoeff);

  /*! @brief Destructor */
  ~LiveTimeAction() {
  }

  /*! @brief Fills histogram with the selected event.
   *
   * @param event The selected event.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Writes the histogram to the output files (ASCII and ROOT).
   *
   * The output consists of a text file and of a ROOT file where the 1-dimensional rigidity
   * histogram (TH1F) is saved. Additionally, another text file will store the content of the "inferior bin"
   * and the total live time for all the "normal" bins.
   */
  void Finalize();

  /*! @brief Returns the total live time (s) spent at threshold rigidities within all "normal" bins (excluding the "inferior bin") */
  double GetTotalLTHisto() {
    return _totalNorm;
  }

  /*! @brief Returns the live time (s) spent at threshold rigidities within the "inferior bin". */
  double GetLTInfBin() {
    return GetUnderflow();
  }

  /*! @brief Returns the number of events at threshold rigidities within all "normal" bins (excluding the "inferior bin") */
  UInt_t GetNGoodHisto() {
    return _nGoodHisto;
  }

  /*! @brief Returns the number of events within the "inferior bin" */
  UInt_t GetNGoodInfBin() {
    return _nGoodInfBin;
  }

private:


  float _thresholdCoeff;

  double _totalNorm;
  UInt_t _nGoodHisto;
  UInt_t _nGoodInfBin;
  

};

#endif /* LIVETIMEACTION_H_ */
