/*
 * SaveEventsAction.h
 *
 *  Created on: 16-mag-2009
 *      Author: Nicola Mori
 */

/*! @file SaveEventsAction.h The SaveEventsAction class declaration file */

#ifndef SAVEEVENTSACTION_H_
#define SAVEEVENTSACTION_H_

#include "../CollectionAction/CollectionAction.h"

/*! @brief An action that saves the selected events
 *
 * This action saves the selected events into a ROOT file. WARNING: since this class uses the PamLevel2 clone trees, you can
 * have NO MORE THAN 1 SaveEventsAction objects in your code (otherwise they will share the same PamLevel2 clone trees, resulting
 * in an undefined behavior).
 */
class SaveEventsAction: public CollectionAction {

public:
  /*! @brief Constructor.
   *
   * outOptions is to be passed as the argument of a PamLevel2::SetWhichTrees() call,
   * so it must be encoded in a compatible format. See the PamLevel2 documentation.
   *
   * @param actionName The action's name.
   * @param outFileName The output file name, with path (absolute or relative).
   * @param outOptions The output options.
   * @param maxFileSize The maximum output file size in bytes; default is 4 GB. If the maximum size will exceed this value
   *                    an error will be generated, since the first clone tree in PamLevel2 which during its Fill will exceed
   *                    this value will try to close the current file and open another one (this is the predefined TTree behavior)
   *                    leaving the other clone trees with no file to write on.
   *                    Note: if specifying the size you get a compiler warning like "warning: this decimal constant is unsigned
   *                    only in ISO C90" then append LL to the numerical constant, to tell the compiler that it has to be interpreted
   *                    as a long long int.
   */
  SaveEventsAction(const char *actionName, TString outFileName, TString outOptions = TString(""), Long64_t maxFileSize =
      4000000000LL);

  /*! @brief Destructor */
  ~SaveEventsAction() {
  }

  /*! @brief The setup procedure.
   *
   * This routine will set the trees to be saved, by calling PamLevel2::SetWhichTrees().
   *
   * @param events The events pointer.
   */
  void Setup(PamLevel2 *events);

  /*! @brief Fills the tree with the selected event.
   *
   * @param event The selected event.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Writes the tree of saved events to the output file.  */
  void Finalize();

private:

  TFile *_outTreeFile;
  PamLevel2 *_events;
  TString _outOptions;
  TString _outFileName;
  Long64_t _maxFileSize;
};

#endif /* SAVEEVENTSACTION_H_ */
