/*
 * ChargeHistosAction.h
 *
 *  Created on: 17-mag-2009
 *      Author: Nicola Mori
 */

/*! @file ChargeHistosAction.h The ChargeHistosAction class declaration file */

#ifndef CHARGEHISTOSACTION_H_
#define CHARGEHISTOSACTION_H_

#include "TH1.h"
#include "TH2.h"

#ifndef NO_TOFNUCLEI
#include <ToFNuclei.h>
#endif

#ifndef NO_CALONUCLEI
#include <CaloNuclei.h>
#endif

#ifndef NO_TRKNUCLEI
#include <TrkNuclei.h>
#endif

#include "../CollectionAction/CollectionAction.h"

/*! @brief An action that fills some charge histograms.
 *
 * This class uses ToFNuclei, TrkNuclei and CaloNuclei to compute the charge
 * of the particle and fills various histograms. Note that if the Nuclei routines
 * are not available in the environment it is possible to exclude the corresponding
 * code by passing the appropriate flags to the compiler (see the EXCLUDEFLAGS
 * variable in the PamCut's makefile).
 * Currently, filled histograms are:
 * - charge from TrkNuclei (Siegen method);
 * - charge from CaloNuclei (first plane dE/dx Vs. beta);
 * - charge from each ToF layer (dE/dx Vs. beta);
 * - charge correlation for each pair of ToF layers.
 */
class ChargeHistosAction: public CollectionAction {

public:
  /*! @brief Constructor.
   *
   * @param actionName The action's name.
   * @param outFileName The output ROOT file name.
   * @param mode The mode of file creation (see documentation of TFile constructor
   *             in ROOT's reference guide)
   */
  ChargeHistosAction(const char *actionName, TString outFileName, TString mode = "UPDATE");

  /*! @brief Destructor. */
  ~ChargeHistosAction();

  /*! @brief The histogram filling method.
   *
   * This routine fills the charge histograms for the selected event.
   * @param event The selected event.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Saves the histograms and closes the file. */
  void Finalize();

private:

#ifndef NO_TOFNUCLEI
  ToFNuclei *_tofNuclei;

  TH1F _chargeS11;
  TH1F _chargeS12;
  TH1F _chargeS21;
  TH1F _chargeS22;
  TH1F _chargeS31;
  TH1F _chargeS32;

  TH2F _chCorrS11S12;
  TH2F _chCorrS11S21;
  TH2F _chCorrS11S22;
  TH2F _chCorrS11S31;
  TH2F _chCorrS11S32;
  TH2F _chCorrS12S21;
  TH2F _chCorrS12S22;
  TH2F _chCorrS12S31;
  TH2F _chCorrS12S32;
  TH2F _chCorrS21S22;
  TH2F _chCorrS21S31;
  TH2F _chCorrS21S32;
  TH2F _chCorrS22S31;
  TH2F _chCorrS22S32;
  TH2F _chCorrS31S32;
#endif

#ifndef NO_CALONUCLEI
  CaloNuclei *_caloNuclei;
  TH1F _chargeCalo;
#endif

#ifndef NO_TRKNUCLEI
  TrkNuclei *_trkNuclei;
  TH1F _chargeTrk;
#endif

  TString _outFileName;
  TString _mode;

};

#endif /* CHARGEHISTOSACTION_H_ */
