/*
 * CaloTrackChi2Cut.h
 *
 *  Created on: 18-mar-2009
 *      Author: mori
 */

/*! @file CaloTrackChi2Cut.h The CaloTrackChi2Cut class definition file */

#ifndef NO_CALOAXIS

#ifndef CALOTRACKCHI2CUT_H_
#define CALOTRACKCHI2CUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include <CaloAxis.h>

/*! @brief The calorimeter track chi2 quality cut.
 *
 * This cut performs a check on chi2 of the calorimeter track. The current
 * implementation uses the CaloAxis objects; to save computing time, the class
 * assumes that the track is externally computed for each event and stored in
 * CaloAxis objects; pointers to these objects are passed as arguments to the constructor.
 * The Check method will then ignore the PamLevel2 *event and assume that the
 * current content of the CaloAxis objects are relative to the current event.
 * It is an user's task to ensure that these assumptions are fulfilled every time
 * Check or ApplyCut are called.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloTrackChi2Cut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The CaloAxis arguments are pointers to objects which contain the calorimeter
   * track information for current event.
   *
   * @param cutName The cut name.
   * @param xCaloAxis Pointer to the CaloAxis object for X axis.
   * @param yCaloAxis Pointer to the CaloAxis object for Y axis.
   * @param maxChi2 The maximum allowed Chi2 (default value is optimized for single-track fit FitAxis).
   * @param minChi2 The minimum allowed Chi2 (default value is optimized for single-track fit FitAxis).
   */
  CaloTrackChi2Cut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, float maxChi2=0.03, float minChi2=0.005) :
    PamCut(cutName), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _maxChi2(maxChi2), _minChi2(minChi2) {
  }
  /*! @brief Destructor. */
  ~CaloTrackChi2Cut() {
  }

  /*! @brief The calorimeter chi2 quality check.
   *
   * @param event The event to analyze.
   * @return #CUTOK if Chi2 from CaloAxis is strictly greater than minChi2 and strictly lesser than maxChi2.
   * @return 0 otherwise
   */
  int Check(PamLevel2 *event);

private:
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  float _maxChi2, _minChi2;
};

#endif /* CALOTRACKCHI2CUT_H_ */
#endif /* NO_CALOAXIS */
