/*
 * CaloNotIntCut.h
 *
 *  Created on: 17-mar-2009
 *      Author: Nicola Mori
 */

/*! @file CaloNotIntCut.h The CaloNotIntCut class definition file */

#ifndef CALONOTINTCUT_H_
#define CALONOTINTCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include "../../CaloAxis2.h"

/*! @brief The non-interacting track cut.
 * This check discards all the events who interacts in the calorimeter. It is possible to choose the track to use for
 * the check. To improve computation speed, Check does NOT computes the track for the event; the track information
 * is supposed to lie in some external object, whose address has to be passed to the constructor. The user is then
 * demanded to fill this external object with the proper track information for the current event before calling Check
 * or ApplyCut.
 * Currently, only the check with the calorimeter track is implemented.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloNotIntCut: public PamCut {

public:
  /*! @brief Constructor for tracker track.
   * Use this constructor if you want to use the tracker track to perform the
   * non-interaction check.
   *
   * NOTE: Check() for this constructor is not implemented yet.
   *
   * @param cutName The name of the cut.
   * @param trkTrack Pointer to the tracker track object.
   * * @param qRatioMin The minimum threshold for the ratio between track charge and total charge in CALO.
   */
  CaloNotIntCut(const char *cutName, TrkTrack *trkTrack, float qRatioMin=0.8) :
    PamCut(cutName), _trkTrack(trkTrack), _xCaloAxis(NULL), _yCaloAxis(NULL), _qRatioMin(qRatioMin) {
  }

  /*! @brief Constructor for calorimeter track.
   * Use this constructor if you want to use the calorimeter track to perform the
   * non-interaction check.
   *
   * @param cutName The name of the cut.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis
   * @param qRatioMin The minimum threshold for the ratio between track charge and total charge in CALO.
   * @return
   */
  CaloNotIntCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, float qRatioMin=0.8) :
    PamCut(cutName), _trkTrack(NULL), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _qRatioMin(qRatioMin) {
  }

  //TODO: aggiungere il costruttore per il TOF.

  /*! @brief Destructor. */
  ~CaloNotIntCut() {
  }

  /*! @brief The non-interacting track check.
   *
   * @param event The event to analyze.
   * @return #CUTOK if the ratio between the charge released along the Calo track
   *                and the total released charge is greater than qRatioMin.
   * @return 0 otherwise.
   */
  int Check(PamLevel2 *event);

private:

  TrkTrack *_trkTrack;
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  float _qRatioMin;

};
#endif /* CALONOTINTCUT_H_ */
