/*
 * CaloNotIntCut.h
 *
 *  Created on: 17-mar-2009
 *      Author: Nicola Mori
 */

/*! @file CaloNotIntCut.h The CaloNotIntCut class definition file */

#ifndef CALONOTINTCUT_H_
#define CALONOTINTCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#ifndef NO_CALOAXIS
#include <CaloAxis.h>
#endif /* NO_CALOAXIS */

/*! @brief The non-interacting track cut.
 * This check discards all the events who interacts in the calorimeter. It is possible to choose the track to use for
 * the check. To improve computation speed, Check does NOT compute the track for the event; the track information
 * is supposed to lie in some external object, whose address has to be passed to the constructor. The user is then
 * demanded to fill this external object with the proper track information for the current event before calling Check
 * or ApplyCut.
 * Currently, the checks with the calorimeter L2 track and with CaloAxis track are implemented.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloNotIntCut: public PamCut {

public:
  /*! @brief Constructor for calorimeter L2 track.
   * Use this constructor if you want to use the L2 calorimeter track to perform the
   * non-interaction check.
   *
   * @param cutName The name of the cut.
   * @param pamTrack Pointer to the Pamela track object.
   * @param qRatioMin The minimum threshold for the ratio between track charge and total charge in CALO.
   */

  CaloNotIntCut(const char *cutName, PamTrack **pamTrack, float qRatioMin = 0.8) :
    PamCut(cutName), _pamTrack(pamTrack), _qRatioMin(qRatioMin) {
#ifndef NO_CALOAXIS
    _xCaloAxis = _yCaloAxis = NULL;
#endif /* NO_CALOAXIS */
  }

  /*! @brief Constructor for calorimeter track.
   * Use this constructor if you want to use the CaloAxis track to perform the
   * non-interaction check.
   *
   * @param cutName The name of the cut.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis
   * @param qRatioMin The minimum threshold for the ratio between track charge and total charge in CALO.
   * @return
   */
#ifndef NO_CALOAXIS
  CaloNotIntCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, float qRatioMin = 0.8) :
    PamCut(cutName), _pamTrack(NULL), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _qRatioMin(qRatioMin) {
  }
#endif /* NO_CALOAXIS */
  //TODO: aggiungere il costruttore per il TOF.

  /*! @brief Destructor. */
  ~CaloNotIntCut() {
  }

  /*! @brief The non-interacting track check.
   *
   * @param event The event to analyze.
   * @return #CUTOK if the ratio between the charge released along the Calo track
   *                and the total released charge is greater than qRatioMin.
   * @return 0 otherwise.
   */
  int Check(PamLevel2 *event);

private:

  PamTrack **_pamTrack;
#ifndef NO_CALOAXIS
  CaloAxis *_xCaloAxis, *_yCaloAxis;
#endif /* NO_CALOAXIS */
  float _qRatioMin;

};
#endif /* CALONOTINTCUT_H_ */
