/*
 * CaloGeomCut.cpp
 *
 *  Created on: 19-mar-2009
 *      Author: Sergio Ricciarini, Nicola Mori
 */

/*! @file CaloGeomCut.cpp The CaloCrossCut class implementation file */

#ifndef NO_CALOAXIS

#include "CaloGeomCut.h"
const Int_t CaloGeomCut::_nPoint = TrkParams::nGF + 1;
const Float_t CaloGeomCut::_zCaloTop = -26.181;

CaloGeomCut::CaloGeomCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, UInt_t iMethod,
    Float_t xTolCaloTrack, Float_t yTolCaloTrack, Float_t mass, Float_t zeta, Float_t resMax, Float_t qualCut, Float_t chi2Cut) :
    PamCut(cutName), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _iMethod(iMethod), _xTolCaloTrack(xTolCaloTrack), _yTolCaloTrack(
        yTolCaloTrack), _mass(mass),  _zeta(zeta), _resMax(resMax), _qualCut(qualCut), _chi2Cut(chi2Cut), _traj(NULL), _zIn(_nPoint) {

  if (_iMethod == 1) {
    _traj = (Trajectory*) new Trajectory(_nPoint);

    for (Int_t igf = 0; igf < TrkParams::nGF; igf++) {
      _zIn[igf] = TrkParams::zGF[igf];
    }
    _zIn[TrkParams::nGF] = _zCaloTop;

  }
}

/*! @brief Destructor. */
CaloGeomCut::~CaloGeomCut() {

  delete _traj;
}

int CaloGeomCut::Check(PamLevel2 *event) {

  Float_t xCaloTrack = 0.;
  Float_t yCaloTrack = 0.;

  Double_t tanx, tany, beta;
  Float_t alCaloTop[5]; // MUST be Float_t

  tanx = _xCaloAxis->par[1];
  tany = _yCaloAxis->par[1];

  if (_iMethod == 0) {
  }
  else if (_iMethod == 1) { // curved track in the Pamela acceptance (back-propagated starting from the straight track in the calorimeter, evaluated at the top of calorimeter, which is of course better than at higher z coordinates)

    alCaloTop[0] = _xCaloAxis->GetYfit(_zCaloTop); // Yfit   [xCaloTop]
    alCaloTop[1] = _yCaloAxis->GetYfit(_zCaloTop); // Yfit  [yCaloTop]
    alCaloTop[2] = sin(atan(sqrt(pow(tanx, 2) + pow(tany, 2)))); // [sintheta]
    alCaloTop[3] = TMath::Pi() + atan2(tany, tanx); // [phi]

    beta = event->GetToFLevel2()->CalcBeta(0, _resMax, _qualCut, _chi2Cut); // TOF stand-alone beta
    alCaloTop[4] = 1. / ((_mass/_zeta) * beta / sqrt(1. - pow(beta, 2))); // [etaP] [rho = (m/Z)*(beta/(sqrt(1-beta^2)))]

    _traj->DoTrack(alCaloTop, _zCaloTop);

  }
  else {
    return 1;
  }

  for (Int_t igf = 0; igf < TrkParams::nGF; igf++) {

    if (_iMethod == 0) { // straight track
      // NOTE: par[0] is the x (y) coordinate on plane z=0 in PAMELA reference (magnet centre)
      xCaloTrack = _xCaloAxis->par[0] + tanx * TrkParams::zGF[igf]; // cm
      yCaloTrack = _yCaloAxis->par[0] + tany * TrkParams::zGF[igf]; // cm
    }
    else if (_iMethod == 1) {
      xCaloTrack = _traj->x[igf];
      yCaloTrack = _traj->y[igf];
    }

    if (!(TrkParams::xGF_min[igf] + _xTolCaloTrack < xCaloTrack
        && xCaloTrack < TrkParams::xGF_max[igf] - _xTolCaloTrack)) {
      return 0;
    }

    if (!(TrkParams::yGF_min[igf] + _yTolCaloTrack < yCaloTrack
        && yCaloTrack < TrkParams::yGF_max[igf] - _yTolCaloTrack)) {
      return 0;
    }

  }

  return CUTOK;

}

#endif /* NO_CALOAXIS */
