/*
 * CaloGeomCut.h
 *
 *  Created on: 13-apr-2010
 *      Author: S. Ricciarini
 */

/*! @file CaloGeomYSensCut.h The CaloGeomYSensCut class definition file */

#ifndef NO_CALOAXIS

#ifndef CALOGEOMYSENSCUT_H_
#define CALOGEOMYSENSCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include <CaloAxis.h>

/*! @brief The geometric cut using the calorimeter track AND checking the TRK crossed sensor along Y.
 *
 * This cut checks if the track obtained from the calorimeter (with different methods) is inside the
 * fiducial acceptance and if the TRK crossed sensor along Y for each of the 6 TRK planes is ySens
 * (0 or 1, for increasing Y).
 * The current implementation uses the CaloAxis objects; to save computing time, the class
 * assumes that the track is externally computed for each event and stored in
 * CaloAxis objects; pointers to these objects are passed as arguments to the constructor.
 * The Check method will then ignore the PamLevel2 *event and assume that the
 * current content of the CaloAxis objects are relative to the current event.
 * It is an user's task to ensure that these assumptions are fulfilled every time
 * Check or ApplyCut are called.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloGeomYSensCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The CaloAxis arguments are pointers to objects which contain the calorimeter
   * track information for current event.
   *
   * @param cutName The cut's name.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis.
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis.
   * @param iMethod index of the method used to track: 0 means straight line;
   *                1 means curved line (taking into account magnetic field and beta
   *                from TOF stand-alone [assuming proton mass] with 3 parameters specified
   *                below) starting from the top of the calorimeter and going backward
   *                (straight line inside the calorimeter).
   * Various combinations of TOF beta quality parameters are possible: default, low-quality beta (used for beta[12]): 10.,10.,20.; medium-quality beta: 5.,15.,4; high-quality beta: 3.,20.,3.
   * @param ySens The TRK sensor identifier along Y (0 or 1 for increasing Y).
   * @param xTolCaloTrack The tolerance for X view which defines the fiducial acceptance.
   * @param yTolCaloTrack The tolerance for Y view which defines the fiducial acceptance.
   * @param mass The particle's mass (used only when iMethod = 1). Default is #H_MASS.
   * @param resMax Default: 10. (equivalent to beta[12])
   * @param qualCut Default: 10. (equivalent to beta[12])
   * @param chi2Cut Default: 20. (equivalent to beta[12])
   */
  CaloGeomYSensCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, UInt_t iMethod, UInt_t ySens,
      Float_t xTolCaloTrack = 0.7, Float_t yTolCaloTrack = 0.7, Float_t mass = H_MASS, Float_t resMax = 10.,
      Float_t qualCut = 10., Float_t chi2Cut = 20.) :
    PamCut(cutName), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _iMethod(iMethod), _ySens(ySens), _xTolCaloTrack(
        xTolCaloTrack), _yTolCaloTrack(yTolCaloTrack), _mass(mass), _resMax(resMax), _qualCut(qualCut), _chi2Cut(
        chi2Cut) {
  }
  /*! @brief Destructor. */
  ~CaloGeomYSensCut() {
  }

  /*! @brief The geometry check using the calorimeter's track.
   *
   * @param event The event to analyze.
   * @return CUTOK if CALO track is inside fiducial acceptance and crosses the same Y sensor on each TRK plane.
   */
  int Check(PamLevel2 *event);

private:
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  UInt_t _iMethod;
  UInt_t _ySens;
  Float_t _xTolCaloTrack, _yTolCaloTrack;
  Float_t _mass;
  Float_t _resMax, _qualCut, _chi2Cut;
};

#endif /* CALOGEOMYSENSCUT_H_ */
#endif /* NO_CALOAXIS */
