/*
 * CaloNucleiZCut.h
 *
 *  Created on: 30-apr-2009
 *      Author: Nicola Mori
 */

/*! @file CaloNucleiZCut.h The TofNucleiZCut class definition file */

#ifndef NO_CALONUCLEI

#ifndef CALONUCLEIZCUT_H_
#define CALONUCLEIZCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include <CaloNuclei.h>

#include <TH2F.h>

/*! @enum CaloNucleiZ_Return Return values for rejected events */
enum CaloNucleiZ_Return {
  CALONUCLEIZ_OUTOFBOUNDS, ///< Discarded because charge is out of bounds
  CALONUCLEIZ_ILLEGALZ
///< Discarded because charge is not between 1 and 8
};

/*! @enum CaloNucleiZ_method Method to use to measure Z. */
enum CaloNucleiZ_method {
  CALONUCLEIZ_SIEGEN, ///< Siegen method: stdedx vs. beta from ToF
  CALONUCLEIZ_DEDXBETA, ///< dE/dx in the first calorimeter plane vs. beta from ToF
  CALONUCLEIZ_DEDXDEFL,
///< dE/dx in the first calorimeter plane vs. deflection from Tracker
};

/*! @brief The CaloNuclei charge cut.
 *
 * This cut uses the CaloNuclei library to cut events depending on particle's charge value.
 *
 */

class CaloNucleiZCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The charge to look for is passed to the CaloNucleiZCut object as Z. For each charge, there is an associated distribution of
   * events, with mean Zmean and standard deviation sigmaZ. These values will be used to check the events.
   *
   * @param cutName The cut's name.
   * @param Z The charge value.
   * @param lowerLimit The lower bound (in units of sigma_Z) of the desired charge range, defined as Zmean - lowerBound*sigmaZ.
   * @param upperLimit The upper bound (in units of sigma_Z) of the desired charge range, defined as Zmean + upperBound*sigmaZ.
   * @param method The method to use to measure the particle's charge (see #CaloNucleiZ_method).
   * @return
   */
  CaloNucleiZCut(const char *cutName, unsigned int Z, float lowerLimit, float upperLimit, unsigned int method =
      CALONUCLEIZ_SIEGEN) :
    PamCut(cutName), _Z(Z), _lowerLimit(lowerLimit), _upperLimit(upperLimit), _caloNuclei(NULL), _method(method) {

  }
  /*! @brief Destructor. */
  ~CaloNucleiZCut() {
    delete _caloNuclei;

  }

  /*! @brief The CaloNucleiZ check.
   *
   * The routine selects events whose value of Zmean obtained from the CaloNuclei Class is in the range [Zmean - sigma*lowerBound, Zmean + sigma*upperBound].
   * Zmean and sigma are defined respectively as the mean and the standard deviation of the charge distributions obtained using CaloNuclei.
   * In current implementation, means are taken to be equal to the charge value (Zmean = Z) and sigmas are:
   *
   *   Z |  sigma
   *   -----------
   *   1 |  0.11
   *   2 |  0.14
   *   3 |  0.21
   *   4 |  0.28
   *   5 |  0.28
   *   6 |  0.31
   *   7 |  0.32
   *   8 |  0.32
   *
   * independent of the method used.
   *
   * @param event The event to analyze.
   *
   */
  int Check(PamLevel2 *event);

private:

  unsigned int _Z;
  static const float _sigmaZ[];

  float _lowerLimit;
  float _upperLimit;
  CaloNuclei *_caloNuclei;

  unsigned int _method;
};
#endif /* CALONUCLEIZCUT_H_ */

#endif /* NO_CALONUCLEI */
