/*
 * CaloGeomCut.h
 *
 *  Created on: 19-mar-2009
 *      Author: Sergio Ricciarini, Nicola Mori
 */

/*! @file CaloGeomCut.h The CaloGeomCut class definition file */

#ifndef CALOGEOMCUT_H_
#define CALOGEOMCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include "../../CaloAxis2.h"

/*! @brief The geometric cut using the calorimeter's track.
 *
 * This cut checks if the track obtained from the calorimeter is inside the fiducial acceptance.
 * The current implementation uses the CaloAxis objects; to save computing time, the class
 * assumes that the track is externally computed for each event and stored in
 * CaloAxis objects; pointers to these objects are passed as arguments to the constructor.
 * The Check method will then ignore the PamLevel2 *event and assume that the
 * current content of the CaloAxis objects are relative to the current event.
 * It is an user's task to ensure that these assumptions are fulfilled every time
 * Check or ApplyCut are called.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloGeomCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The CaloAxis arguments are pointers to objects which contain the calorimeter
   * track information for current event.
   *
   * @param cutName The cut's name.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis.
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis.
   * @param xTolCaloTrack The tolerance for X view which defines the fiducial acceptance.
   * @param yTolCaloTrack The tolerance for Y view which defines the fiducial acceptance.
   */
  CaloGeomCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, float xTolCaloTrack = 0.7, float yTolCaloTrack = 0.7) :
    PamCut(cutName), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _xTolCaloTrack(xTolCaloTrack), _yTolCaloTrack(
        yTolCaloTrack) {
  }
  /*! @brief Destructor. */
  ~CaloGeomCut() {
  }

  /*! @brief The geometry check using the calorimeter's track.
   *
   * @param event The event to analyze.
   * @return #CUTOK if charge released in plane 22 is greater than 0 (from CaloAxis) for both X and Y
   * @return 0 otherwise
   */
  int Check(PamLevel2 *event);

private:
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  float _xTolCaloTrack, _yTolCaloTrack;
};

#endif /* CALOGEOMCUT_H_ */
