/*
 * CaloIonCut.h
 *
 *  Created on: 19-mar-2009
 *      Author: Sergio Ricciarini, Nicola Mori
 */

/*! @file TrkIonCut.h The TrkIonCut class definition file */

#ifndef CALOIONCUT_H_
#define CALOIONCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include "../../CaloAxis2.h"

/*! @brief The calorimeter ionization cut.
 *
 * Checks the energy release on the first calorimeter plane. The current
 * implementation uses the CaloAxis objects; to save computing time, the class
 * assumes that the track is externally computed for each event and stored in
 * CaloAxis objects; pointers to these objects are passed as arguments to the constructor.
 * The Check method will then ignore the PamLevel2 *event and assume that the
 * current content of the CaloAxis objects are relative to the current event.
 * It is an user's task to ensure that these assumptions are fulfilled every time
 * Check or ApplyCut are called.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloIonCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The CaloAxis arguments are objects which contain the calorimeter
   * track information for current event.
   *
   * @param cutName The cut's name.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis.
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis.
   * @param maxRelease The maximum mean ionization (in MIP) in the tracker planes above which an event is discarded. Default: 3.
   * @param minRelease The minimum mean ionization (in MIP) in the tracker planes below which an event is discarded. Default: 0.
   */
  CaloIonCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, float minRelease = 0., float maxRelease = 3.) :
    PamCut(cutName), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _minRelease(minRelease), _maxRelease(maxRelease) {
  }

  /*! @brief Destructor. */
  ~CaloIonCut() {
  }
  /*! @brief The calorimeter ionization check.
   *
   * @param event The event to analyze.
   * @return #CUTOK if charge released in plane 0 (from CaloAxis) is greater than minRelease and
   *                lesser than maxRelease, for both X and Y
   * @return 0 otherwise
   */
  int Check(PamLevel2 *event);

private:
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  float _minRelease, _maxRelease;
};

#endif /* CALOIONCUT_H_ */
