/*
 * CaloAngleCut.h
 *
 *  Created on: 11-jan-2010
 *      Author: S. Ricciarini
 */

/*! @file CaloAngleCut.h The CaloAngleCut class definition file */

#ifndef NO_CALOAXIS

#ifndef CALOANGLECUT_H_
#define CALOANGLECUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include <CaloAxis.h>

/*! @brief Cut on vertical angle, using the calorimeter track.
 *
 * This cut checks if the track obtained from the calorimeter is inside the vertical angle range.
 * The current implementation uses the CaloAxis objects; to save computing time, the class
 * assumes that the track is externally computed for each event and stored in
 * CaloAxis objects; pointers to these objects are passed as arguments to the constructor.
 * The Check method will then ignore the PamLevel2 *event and assume that the
 * current content of the CaloAxis objects are relative to the current event.
 * It is an user's task to ensure that these assumptions are fulfilled every time
 * Check or ApplyCut are called.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloAngleCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The CaloAxis arguments are pointers to objects which contain the calorimeter
   * track information for current event.
   *
   * @param cutName The cut's name.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis.
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis.
   * @param minXAngle minimum vertical angle defining the range
   * @param maxXAngle maximum vertical angle defining the range
   */
  CaloAngleCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, Double_t minXAngle, Double_t maxXAngle) :
    PamCut(cutName), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _minXAngle(minXAngle), _maxXAngle(maxXAngle) {
  }
  /*! @brief Destructor. */
  ~CaloAngleCut() {
  }

  /*! @brief The check of vertical angle using the calorimeter's track.
   *
   * @param event The event to analyze.
   * @return #CUTOK if charge released in plane 22 is greater than 0 (from CaloAxis) for both X and Y
   * @return 0 otherwise
   */
  int Check(PamLevel2 *event);

private:
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  Double_t _minXAngle, _maxXAngle;
};

#endif /* CALOANGLECUT_H_ */
#endif /* NO_CALOAXIS */
