/*
 * RigFillCut.h
 *
 *  Created on: 27-mar-2009
 *      Author: Nicola Mori, S. Ricciarini
 *  Last update: 30-apr-2009
 */

/*! @file RigFillCut.h The RigFillCut class definition file */

#ifndef RIGFILLCUT_H_
#define RIGFILLCUT_H_

#include "../../PamCutBase/PamCutBase.h"

/*! @enum RIGFILL_return Return codes for RigFillCut discarded events */
enum RIGFILL_return {
  RIGOUT, ///< Event is discarded because its rigidity is outside the rigidity binning limits.
  THRESHRIGOUT ///< Event is discarded because its threshold rigidity is outside the threshold rigidity binning limits.
};

/*! @brief The rigidity vs threshold rigidity histogram filling.
 *
 * This class checks the events and builds a 2D histogram binned in rigidity modulus
 * and threshold rigidity. Each 2D bin will contain the number of events whose
 * rigidity modulus and threshold rigidity (eg., Stoermer cutoff rigidity times a threshold coefficient)
 * lie in that bin. Note that the meaning of this threshold coefficient is the same as in
 * TrkRigGeoCut, so it must have the same value used for TrkRigGeoCut (consider
 * using TrkRigGeoCut::GetThresholdCoeff() to retrieve its value).
 * Events are discarded only if their rigidity modulus or cutoff rigidity lies
 * outside the histogram bounds.
 *
 * CUT DEPENDECIES: TrkPhSinCut for single physical track, TrkRigGeoCut for galactic event.
 *
 */

class RigFillCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The binning vectors must contain both the upper and lower limits, and
   * the elements must be ordered (ie., lowest value in the first element and so on)
   * and positive, so that charge sign will be irrelevant.
   *
   * @param cutName The cut's name.
   * @param binning A vector containing the histogram binning in rigidity modulus (X axis) and cutoff rigidity (Y axis).
   * @param
   *
   */
  RigFillCut(const char *cutName, std::vector<float> binning, float thresholdCoeff) :
    PamCut(cutName), _binning(binning), _histogram(binning.size() - 1, binning.size() - 1, 0),
    _thresholdCoeff(thresholdCoeff) {
  }

  /*! @brief Destructor. */
  ~RigFillCut() {
  }

  /*! @brief The rigidity and cutoff rigidity check.
   *
   * The event is discarded if its rigidity modulus or threshold rigidity are outside the limits of the
   * binnings. Please note that only absolute rigidity is considered.
   *
   * @param event The event to analyze.
   * @return #CUTOK if both rigidity modulus and threshold rigidity is contained in the histogram limits.
   * @return 0 otherwise.
   */
  int Check(PamLevel2 *event);

  /*! @brief The histogram filling.
   *
   * The method fills the histogram with the currently selected event.
   *
   * @param event The currently selected event.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Returns the histogram.
   *
   * This method returns a SimpleMatrix. Its [i][j] element contain the number of analyzed events
   * whose rigidity modulus falls in the i-th rigidity bin and whose threshold rigidity falls in the j-th rigidity bin.
   *
   * @return The rigidity modulus - threshold rigidity 2D histogram
   */
  SimpleMatrix<UInt_t> &GetHisto() {
    return _histogram;
  }

private:

  std::vector<float> _binning;
  SimpleMatrix<UInt_t> _histogram;
  float _thresholdCoeff;
};
#endif /* RIGFILLCUT_H_ */
