/*
 * CaloGeomCut.h
 *
 *  Created on: 19-mar-2009
 *      Author: Sergio Ricciarini
 */

/*! @file CaloGeomCut.h The CaloGeomCut class definition file */

#ifndef NO_CALOAXIS

#ifndef CALOGEOMCUT_H_
#define CALOGEOMCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include <CaloAxis.h>

/*! @brief The geometric cut using the calorimeter track.
 *
 * This cut checks if the track obtained from the calorimeter (with different methods) is inside the fiducial acceptance.
 * The current implementation uses the CaloAxis objects; to save computing time, the class
 * assumes that the track is externally computed for each event and stored in
 * CaloAxis objects; pointers to these objects are passed as arguments to the constructor.
 * The Check method will then ignore the PamLevel2 *event and assume that the
 * current content of the CaloAxis objects are relative to the current event.
 * It is an user's task to ensure that these assumptions are fulfilled every time
 * Check or ApplyCut are called.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloGeomCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * The CaloAxis arguments are pointers to objects which contain the calorimeter
   * track information for current event.
   *
   * @param cutName The cut's name.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis.
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis.
   * @param iMethod index of the method used to track: 0 means straight line;
   *                1 means curved line (taking into account magnetic field and beta
   *                from TOF stand-alone with 3 parameters specified
   *                below) starting from the top of the calorimeter and going backward
   *                (straight line inside the calorimeter).
   *                Various combinations of TOF beta quality parameters are possible:
   *                default, low-quality beta (used for beta[12]): 10.,10.,20.;
   *                medium-quality beta: 5.,15.,4; high-quality beta: 3.,20.,3.
   * @param xTolCaloTrack The tolerance for X view which defines the fiducial acceptance.
   * @param yTolCaloTrack The tolerance for Y view which defines the fiducial acceptance.
   * @param mass The particle's mass (used only when iMethod = 1). Default is #H_MASS.
   * @param resMax Default: 10. (equivalent to beta[12])
   * @param zeta The particle Z, with sign (used only when iMethod = 1). Default is +1.
   * @param qualCut Default: 10. (equivalent to beta[12])
   * @param chi2Cut Default: 20. (equivalent to beta[12])
   */
  CaloGeomCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, UInt_t iMethod,
      Float_t xTolCaloTrack = 0.7, Float_t yTolCaloTrack = 0.7, Float_t mass = H_MASS, Float_t zeta = +1.,
      Float_t resMax = 10., Float_t qualCut = 10., Float_t chi2Cut = 20.);

  /*! @brief Destructor. */
  ~CaloGeomCut();

  /*! @brief The geometry check using the calorimeter's track.
   *
   * @param event The event to analyze.
   * @return #CUTOK if charge released in plane 22 is greater than 0 (from CaloAxis) for both X and Y
   * @return 0 otherwise
   */
  int Check(PamLevel2 *event);

private:
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  UInt_t _iMethod;
  Float_t _xTolCaloTrack, _yTolCaloTrack;
  Float_t _mass, _zeta;
  Float_t _resMax, _qualCut, _chi2Cut;

  Trajectory* _traj;
  static const Int_t _nPoint;
  vector<Float_t> _zIn; // MUST be Float_t
  static const Float_t _zCaloTop; // cm MUST be Float_t
};

#endif /* CALOGEOMCUT_H_ */
#endif /* NO_CALOAXIS */
