/*
 * CaloNHitCut.h
 *
 *  Created on: 18-mar-2009
 *      Author: Nicola Mori, S. Ricciarini
 */

/*! @file CaloNHitCut.h The CaloNHitCut class definition file */

#ifndef NO_CALOAXIS

#ifndef CALONHITCUT_H_
#define CALONHITCUT_H_

#include "../../PamCutBase/PamCutBase.h"
#include <CaloAxis.h>

/*! @brief The number of hits cut for the calorimeter.
 * An event survives this selection if the number of hit planes in the calorimeter
 * is greater than some value. To save computing time, the class
 * assumes that the track is externally computed for each event and stored in
 * CaloAxis objects; pointers to these objects are passed as arguments to the constructor.
 * The Check method will then ignore the PamLevel2 *event and assume that the
 * current content of the CaloAxis objects are relative to the current event.
 * It is an user's task to ensure that these assumptions are fulfilled every time
 * Check or ApplyCut are called.
 *
 * CUT DEPENDENCIES: CaloTrackCut for the existence of the track (it can also provide the CaloAxis objects).
 */

class CaloNHitCut: public PamCut {

public:

  /*! @brief Constructor.
   *
   * The CaloAxis arguments are pointers to objects which contain the calorimeter
   * track information for current event.
   *
   * @param cutName The cut name.
   * @param xCaloAxis The pointer to the CaloAxis object for X axis.
   * @param yCaloAxis The pointer to the CaloAxis object for Y axis.
   * @param nMinHit The minimum number of hits.
   */
  CaloNHitCut(const char *cutName, CaloAxis *xCaloAxis, CaloAxis *yCaloAxis, int nMinHit=10) :
    PamCut(cutName), _xCaloAxis(xCaloAxis), _yCaloAxis(yCaloAxis), _nMinHit(nMinHit) {
  }
  /*! @brief Destructor. */
  ~CaloNHitCut() {
  }

  /*! @brief The number of hit planes check.
   *
   * The check is done by looking if there is a charge release near the track, using
   * the CaloAxis class. If a charge greater than 0 is found by invoking CaloAxis::GetQ for at
   * least nMinHit planes (both for X and Y) the condition is fulfilled.
   *
   * @param event The event to analyze.
   * @return #CUTOK if the track hits at least nMinHit planes both in X and Y.
   * @return 0 otherwise.
   * @see CaloNHitCut::CaloNHitCut
   */
  int Check(PamLevel2 *event);

private:
  CaloAxis *_xCaloAxis, *_yCaloAxis;
  int _nMinHit;

};
#endif /* CALONHITCUT_H_ */
#endif /* NO_CALOAXIS */
