/*
 * CollectionAction.h
 *
 *  Created on: 13-mag-2009
 *      Author: Nicola Mori
 */

/*! @file CollectionAction.h The CollectionAction class definition file */

#ifndef COLLECTIONACTION_H_
#define COLLECTIONACTION_H_

#include <PamLevel2.h>
#include "../../CommonDefs.h"

/*! @brief An abstract class that defines the interface of a generic collection action.
 *
 * The CollectionAction abstract class defines an interface for a generic action object
 * which can be used to implement common actions that can be done by collections, like
 * filling histograms or saving selected events. The structure of this class mimicks that
 * of a standard collection: it has a Setup() and a Finalize() methods that are meant
 * to be called at the beginning and at the end of the selection procedure, respectively. In
 * Setup() one can prepare the analysis actions like instantiating a histogram object (this
 * can be done also in the constructor), while in Finalize() the analysis can be closed by,
 * eg., writing histograms on a file and closing that file.
 * The two methods OnGood() and OnBad() have exactly the same purpose they have in a
 * Collection: to define action to perform each time an event is selected as good or bad.
 * The class SmartCollection is specially designed to handle CollectionAction objects: see
 * the documentation of SmartCollection for info on how these objects are handled.
 */
class CollectionAction {
public:

  /*! @brief Constructor.
   *
   * @param actionName The action's name.
   */
  CollectionAction(const char *actionName) :
    _actionName(actionName) {
  }
  /*! @brief Destructor. */
  virtual ~CollectionAction() {

  }

  /*! @brief The initialization routine.
   *
   * In this routine the initialization procedure is to be defined. SmartCollection objects will call
   * Setup() for each CollectionAction object they hold during SmartCollection::Setup().
   * This method is intended for some possible post-constructor initialization, like those who
   * need to know the PamLevel2 object (eg., clone trees selection). This method is likely to be
   * unnecessary most of the times, so it is implemented as a void function and it's not pure virtual.
   *
   * @param events The PamLevel2 events which will be analyzed by the collection hosting the action.
   */
  virtual void Setup(PamLevel2 *events) {
  }

  /*! @brief Routine for selected events.
   *
   * The concrete implementations define what to do when an event is selected. This is conceptually the
   * same as PamCut::OnGood(). Call to this method is automatically done in SmartCollection::OnGood().
   *
   * @param event The selected event.
   */
  virtual void OnGood(PamLevel2 *event) = 0;

  /*! @brief Routine for discarded events.
   *
   * The concrete implementations define what to do when an event is discarded. This is conceptually the
   * same as PamCut::OnBad(). Call to this method is automatically done in SmartCollection::OnBad(). This
   * method is not pure virtual, since many times there's nothing to do when a cut fails; this way, it's not
   * necessary to implement it in derived classes. Current implementation is void.
   *
   * @param event The selected event.
   * @param selectionResult The code describing the reason of cut's failure (collections will automatically pass
   *        the index of the failed cut as selectionResult).
   */
  virtual void OnBad(PamLevel2 *event, int selectionResult) {
  }

  /*! @brief The finalization routine.
   *
   * This method is intended to be called at the end of the analysis. Its implementations contain the finalizing
   * procedures, like writing histograms to file or printing reports. It is automatically called by SmartCollection::Finalize().
   */
  virtual void Finalize() = 0;

  /*! @brief Returns the action name. */
  const char *GetName(){
    return _actionName;
  }

private:
  const char* _actionName;
};

#endif /* COLLECTIONACTION_H_ */
