/*
 * CoordHistoAction.cpp
 *
 *  Created on: 11-giu-2009
 *      Author: mori
 */

/*! @file CoordHistoAction.cpp The CoordHistoAction class implementation file. */

#include "CoordHistoAction.h"

CoordHistoAction::CoordHistoAction(const char *actionName, TString outFileName, TString mode) :
  CollectionAction(actionName), _outFileName(outFileName), _histoLatLong(), _histoRateS11(), _histoRateS12(),
      _histoAlt(), _histoEta(), _histoTheta(), _histoPhi(), _histoPitch(), _mode(mode), _rateRate() {

  _histoLatLong.SetName(TString(GetName()) + "_LatLong");
  _histoLatLong.SetTitle("Latitude Vs Longitude");
  _histoLatLong.GetXaxis()->SetTitle("Longitude");
  _histoLatLong.GetYaxis()->SetTitle("Latitude");
  _histoLatLong.SetBins(720, -185., 185., 360, -95., 95.);

  _histoRateS11.SetName(TString(GetName()) + "_RateS11LatLong");
  _histoRateS11.SetTitle("S11 event rate");
  _histoRateS11.GetXaxis()->SetTitle("Longitude");
  _histoRateS11.GetYaxis()->SetTitle("Latitude");
  _histoRateS11.SetBins(720, -185., 185., 360, -95., 95.);

  _histoRateS12.SetName(TString(GetName()) + "_RateS12LatLong");
  _histoRateS12.SetTitle("S12 event rate");
  _histoRateS12.GetXaxis()->SetTitle("Longitude");
  _histoRateS12.GetYaxis()->SetTitle("Latitude");
  _histoRateS12.SetBins(720, -185., 185., 360, -95., 95.);

  _histoAlt.SetName(TString(GetName()) + "_Alt");
  _histoAlt.SetTitle("Altitude");
  _histoAlt.GetXaxis()->SetTitle("Altitude (m)");
  _histoAlt.GetYaxis()->SetTitle("Events");
  _histoAlt.SetBins(100, 300., 650.);

  _histoEta.SetName(TString(GetName()) + "_Eta");
  _histoEta.SetTitle("Eta");
  _histoEta.GetXaxis()->SetTitle("Eta");
  _histoEta.GetYaxis()->SetTitle("Events");
  _histoEta.SetBins(100, -30., 30.);

  _histoTheta.SetName(TString(GetName()) + "_Theta");
  _histoTheta.SetTitle("Theta");
  _histoTheta.GetXaxis()->SetTitle("Theta");
  _histoTheta.GetYaxis()->SetTitle("Events");
  _histoTheta.SetBins(100, -1., 1.);

  _histoPhi.SetName(TString(GetName()) + "_Phi");
  _histoPhi.SetTitle("Phi");
  _histoPhi.GetXaxis()->SetTitle("Phi");
  _histoPhi.GetYaxis()->SetTitle("Events");
  _histoPhi.SetBins(100, -4., 4.);

  _histoPitch.SetName(TString(GetName()) + "_Pitch");
  _histoPitch.SetTitle("Pitch");
  _histoPitch.GetXaxis()->SetTitle("Pitch");
  _histoPitch.GetYaxis()->SetTitle("Events");
  _histoPitch.SetBins(100, 0., 180.);

  _rateRate.SetName(TString(GetName()) + "_rateRate");
  _rateRate.SetTitle("Rate on S11 / Rate on S12");
  _rateRate.GetXaxis()->SetTitle("Rate of rates");
  _rateRate.GetYaxis()->SetTitle("Events");
  _rateRate.SetBins(100, 0., 10.);

}

void CoordHistoAction::OnGood(PamLevel2 *event) {

  _histoLatLong.Fill(event->GetOrbitalInfo()->lon, event->GetOrbitalInfo()->lat);
  int totRateS11 = 0;
  for (int i = 0; i < 16; i++)
    totRateS11 += event->GetTrigLevel2()->pmtcount1[i];
  _histoRateS11.Fill(event->GetOrbitalInfo()->lon, event->GetOrbitalInfo()->lat, (float) totRateS11 / 16.);
  int totRateS12 = 0;
  for (int i = 16; i < 24; i++) // First 8 PMTs of S12
    totRateS12 += event->GetTrigLevel2()->pmtcount1[i];
  for (int i = 0; i < 4; i++) // Last 4 PMTs of S12
    totRateS12 += event->GetTrigLevel2()->pmtcount2[i];
  _histoRateS12.Fill(event->GetOrbitalInfo()->lon, event->GetOrbitalInfo()->lat, (float) totRateS12 / 12.);
  _rateRate.Fill(((float) totRateS11 / 16.) / (totRateS12 / 12.));

  _histoAlt.Fill(event->GetOrbitalInfo()->alt);
  if (event->GetOrbitalInfo()->etha != -1000.)
    _histoEta.Fill(event->GetOrbitalInfo()->etha);
  if (event->GetOrbitalInfo()->theta != -1000.)
    _histoTheta.Fill(event->GetOrbitalInfo()->theta);
  if (event->GetOrbitalInfo()->phi != -1000.)
    _histoPhi.Fill(event->GetOrbitalInfo()->phi);

  int seqno = event->GetTrack(0)->GetTrkTrack()->GetSeqNo();
  int i = 0;
  while (seqno != event->GetOrbitalInfo()->GetOrbitalInfoTrkVar(i)->trkseqno)
    i++;
  _histoPitch.Fill(event->GetOrbitalInfo()->GetOrbitalInfoTrkVar(i)->pitch);

}

void CoordHistoAction::Finalize() {

  TFile outFile(_outFileName, _mode);
  outFile.cd();
  _histoLatLong.Write();
  _histoRateS11.Write();
  _histoRateS12.Write();
  _histoAlt.Write();
  _histoEta.Write();
  _histoTheta.Write();
  _histoPhi.Write();
  _histoPitch.Write();
  _rateRate.Write();
  outFile.Close();
}
