/*
 * FluxHistoAction.h
 *
 *  Created on: 20-mag-2009
 *      Author: Nicola Mori
 */

/*! @file FluxHistoAction.h The FluxHistoAction class declaration file */

#ifndef FLUXHISTOACTION_H_
#define FLUXHISTOACTION_H_

#include "../CollectionAction/CollectionAction.h"
#include <TH1I.h>
#include <stdint.h>

/*! @brief An action that fills a flux histogram.
 *
 * This action reads a rigidity binning from a file and fills a flux histogram (text and ROOT format).
 * Currently, it fills an event-count histogram: in future implementations, it could also
 * read live time, efficiency and geometrical factor from external files and fill a
 * true flux histogram.
 */
class FluxHistoAction: public CollectionAction {

public:
  /*! @brief Constructor.
   *
   * outFileBase is the base name for output file: #Finalize will add .txt for ASCII output
   * and .root for ROOT output. outFileBase has to contain the path (otherwise, files will be
   * saved in the current directory).
   * The file containing the rigidity bins must be a text file. It must contain both the
   * lower and upper limits of the rigidity axis, so that if it contains N values it
   * defines a set of N-1 bins.
   *
   * @param actionName The action's name.
   * @param outFileBase The output file base name.
   * @param rigBinsFile The file containing the rigidity bins.
   * @param mode The mode of ROOT file creation (see documentation of TFile constructor
   *             in ROOT's reference guide). In current implementation the text file output
   *             does not support append.
   */
  FluxHistoAction(const char *actionName, TString outFileBase, TString rigBinsFile, TString mode = "UPDATE");

  /*! @brief Destructor */
  ~FluxHistoAction() {
  }

  /*! @brief Fills histogram with the selected event.
   *
   * @param event The selected event.
   */
  void OnGood(PamLevel2 *event);

  /*! @brief Writes the histogram to the output files (ASCII and ROOT).
   *
   * The output consists of a text file and of a ROOT file where the 1-dimensional rigidity
   * histogram (TH1F) is saved.
   */
  void Finalize();

private:

  TString _outFileBase;
  vector<float> _bins;
  TH1I _rootHisto;
  vector<uint64_t> _textHisto;
  TString _mode;

#ifdef DEBUGPAMCUT
  int _outUp, _outDown;
#endif

};

#endif /* FLUXHISTOACTION_H_ */
