/*
 * TrkChi2DeflTimeCut.h
 *
 *  Created on: 6-oct-2009
 *      Author: S. Ricciarini
 */

/*! @file TrkChi2DeflTimeCut.h The TrkChi2DeflTimeCut class definition file */

#ifndef TRKCHI2DEFLTIMECUT_H_
#define TRKCHI2DEFLTIMECUT_H_

#include "../../PamCutBase/PamCutBase.h"

#include <TH2F.h>
#include <TTimeStamp.h>

/*! @brief The tracker chi2 vs |deflection| cut.
 *
 * Events whose track has been fitted with a chi2 greater than Chi2(|eta|) are discarded.
 * Chi2(eta) is a function which gives the maximum value of chi2 associated to a certain
 * |deflection| (|eta|); in current implementation it is parameterized as:
 *
 *   Chi2(|eta|) = p0 + ( p1 * |eta|^p2 * (1 + (p3 * |eta|)^2) )
 *   with eta expressed in GV^-1
 *
 * where p0, p1 and p2 are parameters. These are read from a calibration text file, whose format must be:
 * ...
 * first_day	last_day	p0	p1	p2	p3
 * with first_day and last_day expressed in the format YYMMDD
 * ...
 * where each row corresponds to a given time interval (first_day,last_day)
 *
 * For a given event the associated day is determined; the FIRST (starting from first row) time interval of the calibration file, which contains the event day, gives the parameters to be inserted in the Chi2(|eta|) for the given event.
 * If the event day does not fall in any of the intervals of the calibration file, then the event is discarded. 
 *
 */

class TrkChi2DeflTimeCut: public PamCut {

public:
  /*! @brief Constructor.
   *
   * @param cutName The cut's name.
   * @param calibFile Path for the calibration text file.
   */
  TrkChi2DeflTimeCut(const char *cutName, const char *calibFile) :
    PamCut(cutName), _calibFile(calibFile) {
 
    _chi2CutTable.open(_calibFile);
    while (! _chi2CutTable.eof()) {
    
      _chi2CutTable >> _tstringtemp;
      _iDayFirst.push_back(_tstringtemp.Atoi());

      _chi2CutTable >> _tstringtemp;
      _iDayLast.push_back(_tstringtemp.Atoi());
      
      _chi2CutTable >> _tstringtemp;
      _p0.push_back(_tstringtemp.Atof());
      
      _chi2CutTable >> _tstringtemp;
      _p1.push_back(_tstringtemp.Atof());

      _chi2CutTable >> _tstringtemp;
      _p2.push_back(_tstringtemp.Atof());      

      _chi2CutTable >> _tstringtemp;
      _p3.push_back(_tstringtemp.Atof());      
      
    }
    _chi2CutTable.close();

  }

  /*! @brief Destructor. */
  ~TrkChi2DeflTimeCut() {

  }

  /*! @brief The tracker chi2 vs |deflection| check.
   *
   *
   * @param event The event to analyze.
   * @return #CUTOK if chi2 < Chi2(|eta|).
   * @return 0 if not
   */
  int Check(PamLevel2 *event);

private:

    virtual double _GetChi2(PamLevel2 *event);

    const char* _calibFile;

    TString _tstringtemp;
    
    std::vector<Int_t> _iDayFirst;
    std::vector<Int_t> _iDayLast;
    std::vector<Double_t> _p0;
    std::vector<Double_t> _p1;
    std::vector<Double_t> _p2;
    std::vector<Double_t> _p3;

    ifstream _chi2CutTable;
    
    Double_t _p0sel, _p1sel, _p2sel, _p3sel;
    
    TTimeStamp _time;

};
#endif /* TRKCHI2DEFLTIMECUT_H_ */
