/**
 * \file TrigLevel2.h
 * \author Gianfranca DeRosa / Wolfgang Menn
 */


#ifndef TrigLevel2_h
#define TrigLevel2_h
//
#include <TObject.h>
#include <TClonesArray.h>

#include <TrigStruct.h>

/**
 * \brief  Class which contains the Trigger data
 *
 */

class TrigLevel2 : public TObject {
 private:

 public:

  UInt_t         evcount; ///<event number starts from 0 at each new run
  UInt_t         pmtpl[3]; ///<Rate counters top / middle /bottom
  UInt_t         trigrate[6]; ///<Rate counters of trigger configurations TOF 1/2/.../6 (time base 4 seconds)
  UInt_t         dltime[2];  ///<livetime  or deadtime live=0.16*dltime[0]; dead=0.01*dltime[1]
  UInt_t         s4calcount[2]; ///<Rate counters of s4 and calorimeter (time base 4 seconds)
  UInt_t         pmtcount1[24]; ///<Rate counters of the first 24 pmt's (S111a,...,S118a,S111b,...,S118b etc. look at the QL sequence)
  UInt_t         pmtcount2[24]; ///<Rate counters of the last 24 pmt's
  UInt_t         patternbusy[3]; ///<Too complex to describe, basically there are 3 registers which store the status of the busy lines of all the detector in the 3 phases of acq. (after the trigger, when IDAQ set the busy and when iDAQ remove it)

/**
 * \brief patterntrig[6]  What is used in the trigger. One has to do some bit shifting to check the values. Details see below.
 *
 * You can see which channel was used in the trigger. One has to do some bit shifting to check the values....\n
 * Format of patterntrig(6) variable [referring to bit in TB_READ_PATTERN_TRIGGER raw variable] \n
 * patterntrig(1) --> Calo [56 to 53] 4 bits \n
 * patterntrig(2) --> S4 [52 to 50] 3 bits: 7 means S4 trigger, 6 means no S4 trigger \n
 * patterntrig(3) --> S3 [49 to 38] 12 bits: (S32b,S32a,S31b,S31a) \n
 * patterntrig(4) --> S2 [37 to 30] 8 bits: (S22b,S22a,S21b,S21a) \n
 * patterntrig(5) --> S12 [29 to 18] 12 bits: (S12b,S12a) \n
 * patterntrig(6) --> S11 [17 to 2] 16 bits: (S11b,S11a)  \n
 */
  UInt_t         patterntrig[6];

/**
 * \brief trigconf  Gives the trigger configuration by bit shifting. For example: "trigconf & (1<<0)" => TOF1. More Details see below
 *
 * Gives the trigger configuration by bit shifting, for example: \n
 * if ( pam_event->GetTrigLevel2()->trigconf & (1<<0) ) printf(" TOF1\n")  \n
 * and then analogue: \n
 * trigconf & (1<<1) ) => TOF2 \n
 * trigconf & (1<<2) ) => TOF3 \n
 * trigconf & (1<<3) ) => TOF4 \n
 * trigconf & (1<<4) ) => TOF5 \n
 * trigconf & (1<<5) ) => TOF6 \n
 * trigconf & (1<<6) ) => TOF7 \n
 * trigconf & (1<<7) ) => S4 \n
 * trigconf & (1<<8) ) => CALO \n
 * trigconf & (1<<9) ) => CALIB_ON \n
 * \n
 * The "TRIGGER COMMAND INDEX" \n
 * TB_SET_TRIGGER_CONF_01 => TOF_1 \n
 * TB_SET_TRIGGER_CONF_02 => TOF_2 \n
 * TB_SET_TRIGGER_CONF_03 => TOF_3 \n
 * TB_SET_TRIGGER_CONF_04 => TOF_4 \n
 * TB_SET_TRIGGER_CONF_05 => TOF_5 \n
 * TB_SET_TRIGGER_CONF_06 => TOF_6 \n
 * TB_SET_TRIGGER_CONF_07 => TOF_7 \n
 * TB_SET_TRIGGER_CONF_08 => S4 \n
 * TB_SET_TRIGGER_CONF_09 => CALO \n
 * TB_SET_TRIGGER_CONF_10 => CALO + S4 \n
 * TB_SET_TRIGGER_CONF_11 => TOF_1 + S4 \n
 * TB_SET_TRIGGER_CONF_12 => TOF_2 + S4 \n
 * TB_SET_TRIGGER_CONF_13 => TOF_3 + S4 \n
 * TB_SET_TRIGGER_CONF_14 => TOF_4 + S4 \n
 * TB_SET_TRIGGER_CONF_15 => TOF_5 + S4 \n
 * TB_SET_TRIGGER_CONF_16 => TOF_6 + S4 \n
 * TB_SET_TRIGGER_CONF_17 => TOF_7 + S4 \n
 * TB_SET_TRIGGER_CONF_18 => TOF_1 + CALO \n
 * TB_SET_TRIGGER_CONF_19 => TOF_2 + CALO \n
 * TB_SET_TRIGGER_CONF_20 => TOF_3 + CALO \n
 * TB_SET_TRIGGER_CONF_21 => TOF_4 + CALO \n
 * TB_SET_TRIGGER_CONF_22 => TOF_5 + CALO \n
 * TB_SET_TRIGGER_CONF_23 => TOF_6 + CALO \n
 * TB_SET_TRIGGER_CONF_24 => TOF_7 + CALO \n
 * TB_SET_TRIGGER_CONF_25 => TOF_1 + CALO + S4 \n
 * TB_SET_TRIGGER_CONF_26 => TOF_2 + CALO + S4 \n
 * TB_SET_TRIGGER_CONF_27 => TOF_3 + CALO + S4 \n
 * TB_SET_TRIGGER_CONF_28 => TOF_4 + CALO + S4 \n
 * TB_SET_TRIGGER_CONF_29 => TOF_5 + CALO + S4 \n
 * TB_SET_TRIGGER_CONF_30 => TOF_6 + CALO + S4 \n
 * TB_SET_TRIGGER_CONF_31 => TOF_7 + CALO + S4 \n
 * \n
 * TOF_1 = (S11 + S12) * (S21 + S22) * (S31 + S32) \n
 * TOF_2 = (S11 * S12) * (S21 * S22) * (S31 * S32) \n
 * TOF_3 = (S21 + S22) * (S31 + S32) \n
 * TOF_4 = (S21 * S22) * (S31 * S32) \n
 * TOF_5 = S12 * (S21 * S22) \n
 * TOF_6 = (S11 + S12) * (S31 + S32) \n
 * TOF_7 = (S11 * S12) * (S31 * S32) \n
 * S4    = S4_1 \n
 * CALO  = CALO_1 + CALO_2 + CALO_3 + CALO_4 \n
 */
  UInt_t         trigconf;
  Int_t          unpackError; ///<is a flag set in case of CRC errors during the unpacking of data
  //
  // here you can define methods to make life simplier during the analysis
  //

  //
  // constructor
  //
  TrigLevel2();
  //
  // 16 June 06: Included GetTrigLevel2
  TrigLevel2*   GetTrigLevel2(){return this;};
  void SetFromLevel2Struct(cTrigLevel2 *l2);
  void GetLevel2Struct(cTrigLevel2 *l2) const;
  void printpatterntrig();
  bool checkPMTpatterntrig();
  bool bit(int decimal, char pos);


  void Clear(Option_t *t=""); //emiliano

  ClassDef(TrigLevel2,3);
};

#endif

