/**
 * \file TrkLevel1.h
 * \author Elena Vannuccini
 */
#ifndef trklevel1_h
#define trklevel1_h

#include <TObject.h>
#include <TClonesArray.h>
#include <TRef.h>
#include <TROOT.h>
#include <TSystem.h>

//#include <TrkStruct.h>
#include <TrkParams.h>

#define INC_CUT 4.
/**
 * \brief Class to describe tracker clusters. 
 *
 * A cluster is defined as a group of adjacent strips whose signals exceed noise by INC_CUT sigma, and at least one strip has signal/noise > SEED_CUT. Tipical values are SEED_CUT = 7 and INC_CUT = 4.
 * 
 */
// ==================================================================
class TrkCluster : public TObject {

private:


public:
	
    int view; ///< view
    int maxs; ///< strip number (1-3072) of cluster center
    int indmax; 
    
    Int_t         CLlength; ///< number of stored strip info (signal+sigma+adc+bad)
    Float_t      *clsignal; //[CLlength] 
    Float_t      *clsigma;  //[CLlength]
    Int_t        *cladc;    //[CLlength]
    Bool_t       *clbad;    //[CLlength]
    
    TrkCluster();
    ~TrkCluster(){Clear();};
    TrkCluster(const TrkCluster&);

    void Dump();
    void Clear();
    void Clear(Option_t *option){Clear();};
    void Delete(){Clear();};
	
    Float_t GetSignal(Int_t, Float_t, Bool_t);    ///< cluster signal
    Float_t GetSignal(Float_t cut)                { return GetSignal( 0, cut, false); };
    Float_t GetSignal(Int_t nstrip, Bool_t force) { return GetSignal( nstrip, -1000., force ); };
    Float_t GetSignal(Int_t nstrip)               { return GetSignal( nstrip, -1000., false ); };
    Float_t GetSignal()                           { return GetSignal( 0,(Float_t)INC_CUT, false); };    
    
    Float_t GetSignalToNoise(Int_t,Float_t);  ///< cluster signal/noise
    Float_t GetSignalToNoise(Float_t cut)  { return GetSignalToNoise( 0, cut); };
    Float_t GetSignalToNoise(Int_t nstrip) { return GetSignalToNoise( nstrip,-1000. ); };
    Float_t GetSignalToNoise()             { return GetSignalToNoise( 0, (Float_t)INC_CUT); };


    Int_t   GetMultiplicity(Float_t);         ///< cluster multiplicity	
    Int_t   GetMultiplicity()  { return GetMultiplicity(INC_CUT);  };
    
    void GetLevel1Struct(cTrkLevel1* l1);
    void GetLevel1Struct(){GetLevel1Struct(&level1event_);};
    
    Float_t GetCOG(Int_t);
    Float_t GetCOG(Float_t angle);
    Float_t GetCOG(){ return GetCOG(0); };
    Float_t GetETA(Int_t,float,bool);
//    Float_t GetETA(float angle){ return GetETA(0,angle); };
    Float_t GetPositionPU(float);
    Int_t   GetPFAstrips(float angle);

   
    Bool_t IsBad(Int_t);                ///< bad-cluster flag
    Bool_t IsSaturated(Int_t);          ///< saturated-cluster flag

    Int_t GetPlane() { return (Int_t)((view+1)/2);};         ///< plane number (1-6)
    Int_t GetLadder(){ return 1+(Int_t)((maxs-1)/1024);  };  ///< ladder number (1-3)
    Bool_t IsY(){ return (Bool_t)(view%2); };
    Bool_t IsX(){ return !((Bool_t)(view%2)); };
	
    TrkCluster* GetTrkCluster(){ return this; };

    friend class TrkLevel1;

    ClassDef(TrkCluster,2);

};

/**
 * \brief Class to describe tracker LEVEL1 data.
 *
 * A Level1 tracker event is defined as a collection of clusters ( TrkCluster objects ).
 * The result of common-noise computation for each viking is also stored ( cnev[24][12] ).
 * A general flag summarize the event status (missing sections, crc failures, decoding errors ans so on...).
 */
class TrkLevel1 : public TObject {

 private:
    
 public:

    Int_t         good[12];       ///< event status
    Float_t       cn[24][12];     ///< CN
//    Float_t       cnrms[24][12];  ///< CN RMS
    Int_t         cnn[24][12];    ///< number of strips for CN computation

	
    TClonesArray *Cluster;        ///< clusters
    	
    TrkLevel1();
    ~TrkLevel1(){Delete();};
    
    int nclstr() {return Cluster->GetEntries();}    ///< number of stored clusters
    
    void Dump();
//    void SetFromLevel1Struct(cTrkLevel1 *, Bool_t);
//	void GetLevel1Struct(cTrkLevel1 *) const;
    void GetLevel1Struct(cTrkLevel1 *l1);
    void SetFromLevel1Struct(cTrkLevel1 *l1, Bool_t full);    
    void GetLevel1Struct(){GetLevel1Struct(&level1event_);};
    void SetFromLevel1Struct(Bool_t full){SetFromLevel1Struct(&level1event_,full);};
    void SetFromLevel1Struct(){SetFromLevel1Struct(true);};

    void Clear();
    void Delete();
    void Set();
    
    TrkCluster*   GetCluster(int);
    
    TrkLevel1*    GetTrkLevel1(){return this;}
    TClonesArray* GetClusters(){return Cluster;};   ///< returns pointer to the cluster array
    
/*     int LoadPfaParam(TString); */
/*     int LoadFieldParam(TString); */
/*     int LoadChargeParam(TString); */
/*     int LoadAlignmentParam(TString); */
/*     int LoadMipParam(TString); */
/*     int LoadVKMaskParam(TString); */
/*     int LoadPfaParam(){return LoadPfaParam("");}; */
/*     int LoadFieldParam(){return LoadFieldParam("");}; */
/*     int LoadChargeParam(){return LoadChargeParam("");}; */
/*     int LoadAlignmentParam(){return LoadAlignmentParam("");}; */
/*     int LoadMipParam(){return LoadMipParam("");}; */
/*     int LoadVKMaskParam(){return LoadVKMaskParam("");}; */
/*     int LoadParams(); */

//    int ProcessEvent(int);
//    int ProcessEvent(){return ProcessEvent();};
    int ProcessEvent();

// spostati in TrkParams
/*     int GetPfaNbinsAngle(); */
/*     int GetPfaNbinsETA(); */
/*     int GetPfaNbinsCharge(){return 0;}; */
/*     float* GetPfaCoord(TString pfa, int nview, int nladder, int nang); */
/*     float* GetPfaAbs(TString pfa, int nang); */

    void   StatusDump(int view);
    Bool_t StatusCheck(int view, int flagmask);

    
    void SetPFA(int pfaid){ sw_.pfaid = pfaid; };
    int  GetPFA(){ return sw_.pfaid; };
    

    ClassDef(TrkLevel1,2);

};


#endif
